/*
   Copyright 2020 Docker Compose CLI authors

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/

package e2e

import (
	"strings"
	"testing"

	"gotest.tools/v3/icmd"
)

func TestSecretFromEnv(t *testing.T) {
	c := NewParallelCLI(t)
	defer c.cleanupWithDown(t, "env-secret")

	t.Run("compose run", func(t *testing.T) {
		res := icmd.RunCmd(c.NewDockerComposeCmd(t, "-f", "./fixtures/env-secret/compose.yaml", "run", "foo"),
			func(cmd *icmd.Cmd) {
				cmd.Env = append(cmd.Env, "SECRET=BAR")
			})
		res.Assert(t, icmd.Expected{Out: "BAR"})
	})
	t.Run("secret uid", func(t *testing.T) {
		res := icmd.RunCmd(c.NewDockerComposeCmd(t, "-f", "./fixtures/env-secret/compose.yaml", "run", "foo", "ls", "-al", "/var/run/secrets/bar"),
			func(cmd *icmd.Cmd) {
				cmd.Env = append(cmd.Env, "SECRET=BAR")
			})
		res.Assert(t, icmd.Expected{Out: "-r--r-----    1 1005     1005"})
	})
	t.Run("secret uid from user", func(t *testing.T) {
		res := c.RunDockerCmd(t, "version", "--format", "{{ .Server.Version }}")
		if strings.HasPrefix(res.Stdout(), "27.") {
			t.Skip("USER uid:gid is not supported")
		}
		res = icmd.RunCmd(c.NewDockerComposeCmd(t, "-f", "./fixtures/env-secret/compose.yaml", "run", "bar", "ls", "-al", "/var/run/secrets/bar"),
			func(cmd *icmd.Cmd) {
				cmd.Env = append(cmd.Env, "SECRET=BAR")
			})
		res.Assert(t, icmd.Expected{Out: "-r--r--r--    1 1005     root"})
	})
	t.Run("secret uid:gid from user", func(t *testing.T) {
		res := c.RunDockerCmd(t, "version", "--format", "{{ .Server.Version }}")
		if strings.HasPrefix(res.Stdout(), "27.") {
			t.Skip("USER uid:gid is not supported")
		}
		res = icmd.RunCmd(c.NewDockerComposeCmd(t, "-f", "./fixtures/env-secret/compose.yaml", "run", "zot", "ls", "-al", "/var/run/secrets/bar"),
			func(cmd *icmd.Cmd) {
				cmd.Env = append(cmd.Env, "SECRET=BAR")
			})
		res.Assert(t, icmd.Expected{Out: "-r--r--r--    1 1005     1005"})
	})
}

func TestSecretFromInclude(t *testing.T) {
	c := NewParallelCLI(t)
	defer c.cleanupWithDown(t, "env-secret-include")

	t.Run("compose run", func(t *testing.T) {
		res := c.RunDockerComposeCmd(t, "-f", "./fixtures/env-secret/compose.yaml", "run", "included")
		res.Assert(t, icmd.Expected{Out: "this-is-secret"})
	})
}
