﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/codecommit/CodeCommitRequest.h>
#include <aws/codecommit/CodeCommit_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CodeCommit {
namespace Model {

/**
 */
class MergePullRequestByFastForwardRequest : public CodeCommitRequest {
 public:
  AWS_CODECOMMIT_API MergePullRequestByFastForwardRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "MergePullRequestByFastForward"; }

  AWS_CODECOMMIT_API Aws::String SerializePayload() const override;

  AWS_CODECOMMIT_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The system-generated ID of the pull request. To get this ID, use
   * <a>ListPullRequests</a>.</p>
   */
  inline const Aws::String& GetPullRequestId() const { return m_pullRequestId; }
  inline bool PullRequestIdHasBeenSet() const { return m_pullRequestIdHasBeenSet; }
  template <typename PullRequestIdT = Aws::String>
  void SetPullRequestId(PullRequestIdT&& value) {
    m_pullRequestIdHasBeenSet = true;
    m_pullRequestId = std::forward<PullRequestIdT>(value);
  }
  template <typename PullRequestIdT = Aws::String>
  MergePullRequestByFastForwardRequest& WithPullRequestId(PullRequestIdT&& value) {
    SetPullRequestId(std::forward<PullRequestIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the repository where the pull request was created.</p>
   */
  inline const Aws::String& GetRepositoryName() const { return m_repositoryName; }
  inline bool RepositoryNameHasBeenSet() const { return m_repositoryNameHasBeenSet; }
  template <typename RepositoryNameT = Aws::String>
  void SetRepositoryName(RepositoryNameT&& value) {
    m_repositoryNameHasBeenSet = true;
    m_repositoryName = std::forward<RepositoryNameT>(value);
  }
  template <typename RepositoryNameT = Aws::String>
  MergePullRequestByFastForwardRequest& WithRepositoryName(RepositoryNameT&& value) {
    SetRepositoryName(std::forward<RepositoryNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The full commit ID of the original or updated commit in the pull request
   * source branch. Pass this value if you want an exception thrown if the current
   * commit ID of the tip of the source branch does not match this commit ID.</p>
   */
  inline const Aws::String& GetSourceCommitId() const { return m_sourceCommitId; }
  inline bool SourceCommitIdHasBeenSet() const { return m_sourceCommitIdHasBeenSet; }
  template <typename SourceCommitIdT = Aws::String>
  void SetSourceCommitId(SourceCommitIdT&& value) {
    m_sourceCommitIdHasBeenSet = true;
    m_sourceCommitId = std::forward<SourceCommitIdT>(value);
  }
  template <typename SourceCommitIdT = Aws::String>
  MergePullRequestByFastForwardRequest& WithSourceCommitId(SourceCommitIdT&& value) {
    SetSourceCommitId(std::forward<SourceCommitIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_pullRequestId;
  bool m_pullRequestIdHasBeenSet = false;

  Aws::String m_repositoryName;
  bool m_repositoryNameHasBeenSet = false;

  Aws::String m_sourceCommitId;
  bool m_sourceCommitIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace CodeCommit
}  // namespace Aws
