﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/s3-crt/S3Crt_EXPORTS.h>
#include <aws/s3-crt/model/Protocol.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace S3Crt {
namespace Model {

/**
 * <p>Specifies the redirect behavior of all requests to a website endpoint of an
 * Amazon S3 bucket.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/s3-2006-03-01/RedirectAllRequestsTo">AWS
 * API Reference</a></p>
 */
class RedirectAllRequestsTo {
 public:
  AWS_S3CRT_API RedirectAllRequestsTo() = default;
  AWS_S3CRT_API RedirectAllRequestsTo(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_S3CRT_API RedirectAllRequestsTo& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_S3CRT_API void AddToNode(Aws::Utils::Xml::XmlNode& parentNode) const;

  ///@{
  /**
   * <p>Name of the host where requests are redirected.</p>
   */
  inline const Aws::String& GetHostName() const { return m_hostName; }
  inline bool HostNameHasBeenSet() const { return m_hostNameHasBeenSet; }
  template <typename HostNameT = Aws::String>
  void SetHostName(HostNameT&& value) {
    m_hostNameHasBeenSet = true;
    m_hostName = std::forward<HostNameT>(value);
  }
  template <typename HostNameT = Aws::String>
  RedirectAllRequestsTo& WithHostName(HostNameT&& value) {
    SetHostName(std::forward<HostNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Protocol to use when redirecting requests. The default is the protocol that
   * is used in the original request.</p>
   */
  inline Protocol GetProtocol() const { return m_protocol; }
  inline bool ProtocolHasBeenSet() const { return m_protocolHasBeenSet; }
  inline void SetProtocol(Protocol value) {
    m_protocolHasBeenSet = true;
    m_protocol = value;
  }
  inline RedirectAllRequestsTo& WithProtocol(Protocol value) {
    SetProtocol(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_hostName;

  Protocol m_protocol{Protocol::NOT_SET};
  bool m_hostNameHasBeenSet = false;
  bool m_protocolHasBeenSet = false;
};

}  // namespace Model
}  // namespace S3Crt
}  // namespace Aws
