'use strict';

var _lodash = require('lodash.find');

var _lodash2 = _interopRequireDefault(_lodash);

var _importType = require('../core/importType');

var _importType2 = _interopRequireDefault(_importType);

var _staticRequire = require('../core/staticRequire');

var _staticRequire2 = _interopRequireDefault(_staticRequire);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

var defaultGroups = ['builtin', 'external', 'parent', 'sibling', 'index'];

// REPORTING

function reverse(array) {
  return array.map(function (v) {
    return {
      name: v.name,
      rank: -v.rank,
      node: v.node
    };
  }).reverse();
}

function findOutOfOrder(imported) {
  if (imported.length === 0) {
    return [];
  }
  var maxSeenRankNode = imported[0];
  return imported.filter(function (importedModule) {
    var res = importedModule.rank < maxSeenRankNode.rank;
    if (maxSeenRankNode.rank < importedModule.rank) {
      maxSeenRankNode = importedModule;
    }
    return res;
  });
}

function reportOutOfOrder(context, imported, outOfOrder, order) {
  outOfOrder.forEach(function (imp) {
    var found = (0, _lodash2.default)(imported, function hasHigherRank(importedItem) {
      return importedItem.rank > imp.rank;
    });
    context.report(imp.node, '`' + imp.name + '` import should occur ' + order + ' import of `' + found.name + '`');
  });
}

function makeOutOfOrderReport(context, imported) {
  var outOfOrder = findOutOfOrder(imported);
  if (!outOfOrder.length) {
    return;
  }
  // There are things to report. Try to minimize the number of reported errors.
  var reversedImported = reverse(imported);
  var reversedOrder = findOutOfOrder(reversedImported);
  if (reversedOrder.length < outOfOrder.length) {
    reportOutOfOrder(context, reversedImported, reversedOrder, 'after');
    return;
  }
  reportOutOfOrder(context, imported, outOfOrder, 'before');
}

// DETECTING

function computeRank(context, ranks, name, type) {
  return ranks[(0, _importType2.default)(name, context)] + (type === 'import' ? 0 : 100);
}

function registerNode(context, node, name, type, ranks, imported) {
  var rank = computeRank(context, ranks, name, type);
  if (rank !== -1) {
    imported.push({ name: name, rank: rank, node: node });
  }
}

function isInVariableDeclarator(node) {
  return node && (node.type === 'VariableDeclarator' || isInVariableDeclarator(node.parent));
}

var types = ['builtin', 'external', 'internal', 'parent', 'sibling', 'index'];

// Creates an object with type-rank pairs.
// Example: { index: 0, sibling: 1, parent: 1, external: 1, builtin: 2, internal: 2 }
// Will throw an error if it contains a type that does not exist, or has a duplicate
function convertGroupsToRanks(groups) {
  var rankObject = groups.reduce(function (res, group, index) {
    if (typeof group === 'string') {
      group = [group];
    }
    group.forEach(function (groupItem) {
      if (types.indexOf(groupItem) === -1) {
        throw new Error('Incorrect configuration of the rule: Unknown type `' + JSON.stringify(groupItem) + '`');
      }
      if (res[groupItem] !== undefined) {
        throw new Error('Incorrect configuration of the rule: `' + groupItem + '` is duplicated');
      }
      res[groupItem] = index;
    });
    return res;
  }, {});

  var omittedTypes = types.filter(function (type) {
    return rankObject[type] === undefined;
  });

  return omittedTypes.reduce(function (res, type) {
    res[type] = groups.length;
    return res;
  }, rankObject);
}

function makeNewlinesBetweenReport(context, imported, newlinesBetweenImports) {
  var getNumberOfEmptyLinesBetween = function getNumberOfEmptyLinesBetween(currentImport, previousImport) {
    var linesBetweenImports = context.getSourceCode().lines.slice(previousImport.node.loc.end.line, currentImport.node.loc.start.line - 1);

    return linesBetweenImports.filter(function (line) {
      return !line.trim().length;
    }).length;
  };
  var previousImport = imported[0];

  imported.slice(1).forEach(function (currentImport) {
    if (newlinesBetweenImports === 'always') {
      if (currentImport.rank !== previousImport.rank && getNumberOfEmptyLinesBetween(currentImport, previousImport) === 0) {
        context.report(previousImport.node, 'There should be at least one empty line between import groups');
      } else if (currentImport.rank === previousImport.rank && getNumberOfEmptyLinesBetween(currentImport, previousImport) > 0) {
        context.report(previousImport.node, 'There should be no empty line within import group');
      }
    } else {
      if (getNumberOfEmptyLinesBetween(currentImport, previousImport) > 0) {
        context.report(previousImport.node, 'There should be no empty line between import groups');
      }
    }

    previousImport = currentImport;
  });
}

module.exports = function importOrderRule(context) {
  var options = context.options[0] || {};
  var ranks = void 0;

  try {
    ranks = convertGroupsToRanks(options.groups || defaultGroups);
  } catch (error) {
    // Malformed configuration
    return {
      Program: function Program(node) {
        context.report(node, error.message);
      }
    };
  }
  var imported = [];
  var level = 0;

  function incrementLevel() {
    level++;
  }
  function decrementLevel() {
    level--;
  }

  return {
    ImportDeclaration: function handleImports(node) {
      if (node.specifiers.length) {
        // Ignoring unassigned imports
        var name = node.source.value;
        registerNode(context, node, name, 'import', ranks, imported);
      }
    },
    CallExpression: function handleRequires(node) {
      if (level !== 0 || !(0, _staticRequire2.default)(node) || !isInVariableDeclarator(node.parent)) {
        return;
      }
      var name = node.arguments[0].value;
      registerNode(context, node, name, 'require', ranks, imported);
    },
    'Program:exit': function reportAndReset() {
      makeOutOfOrderReport(context, imported);

      if ('newlines-between' in options) {
        makeNewlinesBetweenReport(context, imported, options['newlines-between']);
      }

      imported = [];
    },
    FunctionDeclaration: incrementLevel,
    FunctionExpression: incrementLevel,
    ArrowFunctionExpression: incrementLevel,
    BlockStatement: incrementLevel,
    ObjectExpression: incrementLevel,
    'FunctionDeclaration:exit': decrementLevel,
    'FunctionExpression:exit': decrementLevel,
    'ArrowFunctionExpression:exit': decrementLevel,
    'BlockStatement:exit': decrementLevel,
    'ObjectExpression:exit': decrementLevel
  };
};

module.exports.schema = [{
  type: 'object',
  properties: {
    groups: {
      type: 'array'
    },
    'newlines-between': {
      enum: ['always', 'never']
    }
  },
  additionalProperties: false
}];
//# sourceMappingURL=data:application/json;base64,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