#!/usr/bin/env python

from __future__ import print_function

__description__ = 'Process PCAP files to calculate IP data statistics'
__author__ = 'Didier Stevens'
__version__ = '0.0.1'
__date__ = '2020/05/30'

"""

Source code put in the public domain by Didier Stevens, no Copyright
https://DidierStevens.com
Use at your own risk

History:
  2020/05/26: start based on simple_tcp_stats.py
  2020/05/27: continue
  2020/05/30: updated man

Todo:
"""

import optparse
import glob
import collections
import time
import sys
import textwrap
import os
import gzip
import fnmatch
import dpkt
import socket
import struct
import hashlib
import string
import math
if sys.version_info[0] < 3:
    import cPickle
else:
    import pickle as cPickle
import atexit
if sys.platform == 'win32' and sys.version_info[0] < 3:
    import win_inet_pton
from contextlib import contextmanager

CSV_SEPARATOR = ';'
ENTRY_HEAD = 'head'
ENTRY_PREVALENCE = 'prevalence'

def PrintManual():
    manual = '''
Manual:

This tool processes PCAP files to calculate statistics of IP data.

This Python script was developed with Python 2.7 and tested with Python 2.7 and 3.7.

All data is grouped by connection ID: the combination of source IP, source port, destination IP and destination port for TCP and UDP, and the combination of source IP and source port for other protocols.
The are 2 connection IDs per connection: one per flow direction.

This tool does not reassemble TCP streams: it just takes TCP data in the order it is found inside the PCAP file(s).

The output produced by this tool is a CSV file with header.
The first field is the protocol (TCP, UDP, ICMP, ...).
The second field is the connection ID.
The third field is the head: the first 4 byte of the data of the first packet.
The fourth field is the size: the total amount of data bytes.
The fifth field is the entropy: the total entropy of all data bytes.

Example:

Protocol;ConnectionID;head;Size;Entropy
TCP;96.126.103.196:80-192.168.10.10:50236;'HTTP';493;6.73520107812
TCP;192.168.10.10:50236-96.126.103.196:80;'GET ';364;5.42858024035
TCP;192.168.10.10:50235-96.126.103.196:80;'GET ';426;5.46464090792
UDP;192.168.10.10:56372-239.255.255.250:1900;'M-SE';173;5.35104059717
TCP;96.126.103.196:80-192.168.10.10:50235;'HTTP';3308;6.06151478505

This tool can help identifying data connections that are not encrypted/encoded/compressed, as these connections will have a lower entropy.

Output is to stdout by default, and can be redirected with option -o.

Errors occuring when opening a file are reported (and logged if logging is turned on), and the program moves on to the next file.
Errors occuring when reading & processing a file are reported (and logged if logging is turned on), and the program stops unless option ignoreprocessingerrors is used.

The lines are written to standard output, except when option -o is used. When option -o is used, the lines are written to the filename specified by option -o.
Filenames used with option -o starting with # have special meaning.
#c#example.txt will write output both to the console (stdout) and file example.txt.
#g# will write output to a file with a filename generated by the tool like this: toolname-date-time.txt.
#g#KEYWORD will write output to a file with a filename generated by the tool like this: toolname-KEYWORD-date-time.txt.
Use #p#filename to display execution progress.
To process several files while creating seperate output files for each input file, use -o #s#%f%.result *.
This will create output files with the name of the inputfile and extension .result.
There are several variables available when creating separate output files:
 %f% is the full filename (with directory if present)
 %b% is the base name: the filename without directory
 %d% is the directory
 %r% is the root: the filename without extension
 %ru% is the root made unique by appending a counter (if necessary)
 %e% is the extension
Most options can be combined, like #ps# for example.
#l# is used for literal filenames: if the output filename has to start with # (#example.txt for example), use filename #l##example.txt for example.

'''
    for line in manual.split('\n'):
        print(textwrap.fill(line))

DEFAULT_SEPARATOR = ','
QUOTE = '"'

def PrintError(*args, **kwargs):
    print(*args, file=sys.stderr, **kwargs)

#Convert 2 Integer If Python 2
def C2IIP2(data):
    if sys.version_info[0] > 2:
        return data
    else:
        return ord(data)

#Convert 2 Bytes If Python 3
def C2BIP3(string):
    if sys.version_info[0] > 2:
        return bytes([ord(x) for x in string])
    else:
        return string

#Convert 2 String If Python 3
def C2SIP3(data):
    if sys.version_info[0] > 2:
        return data.decode('utf8')
    else:
        return data

def File2Strings(filename):
    try:
        f = open(filename, 'r')
    except:
        return None
    try:
        return map(lambda line:line.rstrip('\n'), f.readlines())
    except:
        return None
    finally:
        f.close()

def ProcessAt(argument):
    if argument.startswith('@'):
        strings = File2Strings(argument[1:])
        if strings == None:
            raise Exception('Error reading %s' % argument)
        else:
            return strings
    else:
        return [argument]

# CIC: Call If Callable
def CIC(expression):
    if callable(expression):
        return expression()
    else:
        return expression

# IFF: IF Function
def IFF(expression, valueTrue, valueFalse):
    if expression:
        return CIC(valueTrue)
    else:
        return CIC(valueFalse)

def Serialize(object, filename=None):
    try:
        fPickle = open(filename, 'wb')
    except:
        return False
    try:
        cPickle.dump(object, fPickle, cPickle.HIGHEST_PROTOCOL)
    except:
        return False
    finally:
        fPickle.close()
    return True

def DeSerialize(filename=None):
    if os.path.isfile(filename):
        try:
            fPickle = open(filename, 'rb')
        except:
            return None
        try:
            object = cPickle.load(fPickle)
        except:
            return None
        finally:
            fPickle.close()
        return object
    else:
        return None

class cVariables():
    def __init__(self, variablesstring='', separator=DEFAULT_SEPARATOR):
        self.dVariables = {}
        if variablesstring == '':
            return
        for variable in variablesstring.split(separator):
            name, value = VariableNameValue(variable)
            self.dVariables[name] = value

    def SetVariable(self, name, value):
        self.dVariables[name] = value

    def Instantiate(self, astring):
        for key, value in self.dVariables.items():
            astring = astring.replace('%' + key + '%', value)
        return astring

class cOutput():
    def __init__(self, filenameOption=None):
        self.starttime = time.time()
        self.filenameOption = filenameOption
        self.separateFiles = False
        self.progress = False
        self.console = False
        self.fOut = None
        self.rootFilenames = {}
        if self.filenameOption:
            if self.ParseHash(self.filenameOption):
                if not self.separateFiles and self.filename != '':
                    self.fOut = open(self.filename, 'w')
            elif self.filenameOption != '':
                self.fOut = open(self.filenameOption, 'w')

    def ParseHash(self, option):
        if option.startswith('#'):
            position = self.filenameOption.find('#', 1)
            if position > 1:
                switches = self.filenameOption[1:position]
                self.filename = self.filenameOption[position + 1:]
                for switch in switches:
                    if switch == 's':
                        self.separateFiles = True
                    elif switch == 'p':
                        self.progress = True
                    elif switch == 'c':
                        self.console = True
                    elif switch == 'l':
                        pass
                    elif switch == 'g':
                        if self.filename != '':
                            extra = self.filename + '-'
                        else:
                            extra = ''
                        self.filename = '%s-%s%s.txt' % (os.path.splitext(os.path.basename(sys.argv[0]))[0], extra, self.FormatTime())
                    else:
                        return False
                return True
        return False

    @staticmethod
    def FormatTime(epoch=None):
        if epoch == None:
            epoch = time.time()
        return '%04d%02d%02d-%02d%02d%02d' % time.localtime(epoch)[0:6]

    def RootUnique(self, root):
        if not root in self.rootFilenames:
            self.rootFilenames[root] = None
            return root
        iter = 1
        while True:
            newroot = '%s_%04d' % (root, iter)
            if not newroot in self.rootFilenames:
                self.rootFilenames[newroot] = None
                return newroot
            iter += 1

    def Line(self, line, eol='\n'):
        if self.fOut == None or self.console:
            try:
                print(line, end=eol)
            except UnicodeEncodeError:
                encoding = sys.stdout.encoding
                print(line.encode(encoding, errors='backslashreplace').decode(encoding), end=eol)
#            sys.stdout.flush()
        if self.fOut != None:
            self.fOut.write(line + '\n')
            self.fOut.flush()

    def LineTimestamped(self, line):
        self.Line('%s: %s' % (self.FormatTime(), line))

    def Filename(self, filename, index, total):
        self.separateFilename = filename
        if self.progress:
            if index == 0:
                eta = ''
            else:
                seconds = int(float((time.time() - self.starttime) / float(index)) * float(total - index))
                eta = 'estimation %d seconds left, finished %s ' % (seconds, self.FormatTime(time.time() + seconds))
            PrintError('%d/%d %s%s' % (index + 1, total, eta, self.separateFilename))
        if self.separateFiles and self.filename != '':
            oFilenameVariables = cVariables()
            oFilenameVariables.SetVariable('f', self.separateFilename)
            basename = os.path.basename(self.separateFilename)
            oFilenameVariables.SetVariable('b', basename)
            oFilenameVariables.SetVariable('d', os.path.dirname(self.separateFilename))
            root, extension = os.path.splitext(basename)
            oFilenameVariables.SetVariable('r', root)
            oFilenameVariables.SetVariable('ru', self.RootUnique(root))
            oFilenameVariables.SetVariable('e', extension)

            self.Close()
            self.fOut = open(oFilenameVariables.Instantiate(self.filename), 'w')

    def Close(self):
        if self.fOut != None:
            self.fOut.close()
            self.fOut = None

class cExpandFilenameArguments():
    def __init__(self, filenames, literalfilenames=False, recursedir=False, checkfilenames=False, expressionprefix=None):
        self.containsUnixShellStyleWildcards = False
        self.warning = False
        self.message = ''
        self.filenameexpressions = []
        self.expressionprefix = expressionprefix
        self.literalfilenames = literalfilenames

        expression = ''
        if len(filenames) == 0:
            self.filenameexpressions = [['', '']]
        elif literalfilenames:
            self.filenameexpressions = [[filename, ''] for filename in filenames]
        elif recursedir:
            for dirwildcard in filenames:
                if expressionprefix != None and dirwildcard.startswith(expressionprefix):
                    expression = dirwildcard[len(expressionprefix):]
                else:
                    if dirwildcard.startswith('@'):
                        for filename in ProcessAt(dirwildcard):
                            self.filenameexpressions.append([filename, expression])
                    elif os.path.isfile(dirwildcard):
                        self.filenameexpressions.append([dirwildcard, expression])
                    else:
                        if os.path.isdir(dirwildcard):
                            dirname = dirwildcard
                            basename = '*'
                        else:
                            dirname, basename = os.path.split(dirwildcard)
                            if dirname == '':
                                dirname = '.'
                        for path, dirs, files in os.walk(dirname):
                            for filename in fnmatch.filter(files, basename):
                                self.filenameexpressions.append([os.path.join(path, filename), expression])
        else:
            for filename in list(collections.OrderedDict.fromkeys(sum(map(self.Glob, sum(map(ProcessAt, filenames), [])), []))):
                if expressionprefix != None and filename.startswith(expressionprefix):
                    expression = filename[len(expressionprefix):]
                else:
                    self.filenameexpressions.append([filename, expression])
            self.warning = self.containsUnixShellStyleWildcards and len(self.filenameexpressions) == 0
            if self.warning:
                self.message = "Your filename argument(s) contain Unix shell-style wildcards, but no files were matched.\nCheck your wildcard patterns or use option literalfilenames if you don't want wildcard pattern matching."
                return
        if self.filenameexpressions == [] and expression != '':
            self.filenameexpressions = [['', expression]]
        if checkfilenames:
            self.CheckIfFilesAreValid()

    def Glob(self, filename):
        if not ('?' in filename or '*' in filename or ('[' in filename and ']' in filename)):
            return [filename]
        self.containsUnixShellStyleWildcards = True
        return glob.glob(filename)

    def CheckIfFilesAreValid(self):
        valid = []
        doesnotexist = []
        isnotafile = []
        for filename, expression in self.filenameexpressions:
            hashfile = False
            try:
                hashfile = FilenameCheckHash(filename, self.literalfilenames)[0] == FCH_DATA
            except:
                pass
            if filename == '' or hashfile:
                valid.append([filename, expression])
            elif not os.path.exists(filename):
                doesnotexist.append(filename)
            elif not os.path.isfile(filename):
                isnotafile.append(filename)
            else:
                valid.append([filename, expression])
        self.filenameexpressions = valid
        if len(doesnotexist) > 0:
            self.warning = True
            self.message += 'The following files do not exist and will be skipped: ' + ' '.join(doesnotexist) + '\n'
        if len(isnotafile) > 0:
            self.warning = True
            self.message += 'The following files are not regular files and will be skipped: ' + ' '.join(isnotafile) + '\n'

    def Filenames(self):
        if self.expressionprefix == None:
            return [filename for filename, expression in self.filenameexpressions]
        else:
            return self.filenameexpressions

def ToString(value):
    if isinstance(value, str):
        return value
    else:
        return str(value)

def Quote(value, separator, quote):
    value = ToString(value)
    if separator in value or value == '':
        return quote + value + quote
    else:
        return value

def MakeCSVLine(row, separator, quote):
    return separator.join([Quote(value, separator, quote) for value in row])

class cLogfile():
    def __init__(self, keyword, comment):
        self.starttime = time.time()
        self.errors = 0
        if keyword == '':
            self.oOutput = None
        else:
            self.oOutput = cOutput('%s-%s-%s.log' % (os.path.splitext(os.path.basename(sys.argv[0]))[0], keyword, self.FormatTime()))
        self.Line('Start')
        self.Line('UTC', '%04d%02d%02d-%02d%02d%02d' % time.gmtime(time.time())[0:6])
        self.Line('Comment', comment)
        self.Line('Args', repr(sys.argv))
        self.Line('Version', __version__)
        self.Line('Python', repr(sys.version_info))
        self.Line('Platform', sys.platform)
        self.Line('CWD', repr(os.getcwd()))

    @staticmethod
    def FormatTime(epoch=None):
        if epoch == None:
            epoch = time.time()
        return '%04d%02d%02d-%02d%02d%02d' % time.localtime(epoch)[0:6]

    def Line(self, *line):
        if self.oOutput != None:
            self.oOutput.Line(MakeCSVLine((self.FormatTime(), ) + line, DEFAULT_SEPARATOR, QUOTE))

    def LineError(self, *line):
        self.Line('Error', *line)
        self.errors += 1

    def Close(self):
        if self.oOutput != None:
            self.Line('Finish', '%d error(s)' % self.errors, '%d second(s)' % (time.time() - self.starttime))
            self.oOutput.Close()

def AnalyzeFileError(filename):
    PrintError('Error opening file %s' % filename)
    PrintError(sys.exc_info()[1])
    try:
        if not os.path.exists(filename):
            PrintError('The file does not exist')
        elif os.path.isdir(filename):
            PrintError('The file is a directory')
        elif not os.path.isfile(filename):
            PrintError('The file is not a regular file')
    except:
        pass

@contextmanager
def PcapFile(filename, oLogfile):
    if filename == '':
        fIn = sys.stdin
    elif os.path.splitext(filename)[1].lower() == '.gz':
        try:
            fIn = gzip.GzipFile(filename, 'rb')
        except:
            AnalyzeFileError(filename)
            oLogfile.LineError('Opening file %s %s' % (filename, repr(sys.exc_info()[1])))
            fIn = None
    else:
        try:
            fIn = open(filename, 'rb')
        except:
            AnalyzeFileError(filename)
            oLogfile.LineError('Opening file %s %s' % (filename, repr(sys.exc_info()[1])))
            fIn = None

    if fIn != None:
        oLogfile.Line('Success', 'Opening file %s' % filename)

    yield fIn

    if fIn != None:
        if sys.exc_info()[1] != None:
            oLogfile.LineError('Reading file %s %s' % (filename, repr(sys.exc_info()[1])))
        if fIn != sys.stdin:
            fIn.close()

def IP2String(address):
    try:
        return socket.inet_ntop(socket.AF_INET, address)
    except (socket.error, ValueError):
        return socket.inet_ntop(socket.AF_INET6, address)

def CalculateByteStatistics(dPrevalence):
    sumValues = sum(dPrevalence.values())
    countNullByte = dPrevalence[0]
    countControlBytes = 0
    countWhitespaceBytes = 0
    for iter in range(1, 0x21):
        if chr(iter) in string.whitespace:
            countWhitespaceBytes += dPrevalence[iter]
        else:
            countControlBytes += dPrevalence[iter]
    countControlBytes += dPrevalence[0x7F]
    countPrintableBytes = 0
    for iter in range(0x21, 0x7F):
        countPrintableBytes += dPrevalence[iter]
    countHighBytes = 0
    for iter in range(0x80, 0x100):
        countHighBytes += dPrevalence[iter]
    entropy = 0.0
    for iter in range(0x100):
        if dPrevalence[iter] > 0:
            prevalence = float(dPrevalence[iter]) / float(sumValues)
            entropy += - prevalence * math.log(prevalence, 2)
    return sumValues, entropy, countNullByte, countControlBytes, countWhitespaceBytes, countPrintableBytes, countHighBytes

def ProcessPcapFile(filename, dConnections, oOutput, oLogfile, options):
    # https://en.wikipedia.org/wiki/List_of_IP_Protocol_numbers
    dIPProtocolNumbers = {
        0x00: 'HOPOPT',
        0x01: 'ICMP',
        0x02: 'IGMP',
        0x03: 'GGP',
        0x04: 'IP-in-IP',
        0x05: 'ST',
        0x06: 'TCP',
        0x07: 'CBT',
        0x08: 'EGP',
        0x09: 'IGP',
        0x0A: 'BBN-RCC-MON',
        0x0B: 'NVP-II',
        0x0C: 'PUP',
        0x0D: 'ARGUS',
        0x0E: 'EMCON',
        0x0F: 'XNET',
        0x10: 'CHAOS',
        0x11: 'UDP',
        0x12: 'MUX',
        0x13: 'DCN-MEAS',
        0x14: 'HMP',
        0x15: 'PRM',
        0x16: 'XNS-IDP',
        0x17: 'TRUNK-1',
        0x18: 'TRUNK-2',
        0x19: 'LEAF-1',
        0x1A: 'LEAF-2',
        0x1B: 'RDP',
        0x1C: 'IRTP',
        0x1D: 'ISO-TP4',
        0x1E: 'NETBLT',
        0x1F: 'MFE-NSP',
        0x20: 'MERIT-INP',
        0x21: 'DCCP',
        0x22: '3PC',
        0x23: 'IDPR',
        0x24: 'XTP',
        0x25: 'DDP',
        0x26: 'IDPR-CMTP',
        0x27: 'TP++',
        0x28: 'IL',
        0x29: 'IPv6',
        0x2A: 'SDRP',
        0x2B: 'IPv6-Route',
        0x2C: 'IPv6-Frag',
        0x2D: 'IDRP',
        0x2E: 'RSVP',
        0x2F: 'GREs',
        0x30: 'DSR',
        0x31: 'BNA',
        0x32: 'ESP',
        0x33: 'AH',
        0x34: 'I-NLSP',
        0x35: 'SwIPe',
        0x36: 'NARP',
        0x37: 'MOBILE',
        0x38: 'TLSP',
        0x39: 'SKIP',
        0x3A: 'IPv6-ICMP',
        0x3B: 'IPv6-NoNxt',
        0x3C: 'IPv6-Opts',
        #0x3D: '',
        0x3E: 'CFTP',
        0x3F: '',
        0x40: 'SAT-EXPAK',
        0x41: 'KRYPTOLAN',
        0x42: 'RVD',
        0x43: 'IPPC',
        #0x44: '',
        0x45: 'SAT-MON',
        0x46: 'VISA',
        0x47: 'IPCU',
        0x48: 'CPNX',
        0x49: 'CPHB',
        0x4A: 'WSN',
        0x4B: 'PVP',
        0x4C: 'BR-SAT-MON',
        0x4D: 'SUN-ND',
        0x4E: 'WB-MON',
        0x4F: 'WB-EXPAK',
        0x50: 'ISO-IP',
        0x51: 'VMTP',
        0x52: 'SECURE-VMTP',
        0x53: 'VINES',
        0x54: 'TTP',
        0x54: 'IPTM',
        0x55: 'NSFNET-IGP',
        0x56: 'DGP',
        0x57: 'TCF',
        0x58: 'EIGRP',
        0x59: 'OSPF',
        0x5A: 'Sprite-RPC',
        0x5B: 'LARP',
        0x5C: 'MTP',
        0x5D: 'AX.25',
        0x5E: 'OS',
        0x5F: 'MICP',
        0x60: 'SCC-SP',
        0x61: 'ETHERIP',
        0x62: 'ENCAP',
        #0x63: '',
        0x64: 'GMTP',
        0x65: 'IFMP',
        0x66: 'PNNI',
        0x67: 'PIM',
        0x68: 'ARIS',
        0x69: 'SCPS',
        0x6A: 'QNX',
        0x6B: 'A/N',
        0x6C: 'IPComp',
        0x6D: 'SNP',
        0x6E: 'Compaq-Peer',
        0x6F: 'IPX-in-IP',
        0x70: 'VRRP',
        0x71: 'PGM',
        #0x72: '',
        0x73: 'L2TP',
        0x74: 'DDX',
        0x75: 'IATP',
        0x76: 'STP',
        0x77: 'SRP',
        0x78: 'UTI',
        0x79: 'SMP',
        0x7A: 'SM',
        0x7B: 'PTP',
        0x7C: 'IS-IS over IPv4',
        0x7D: 'FIRE',
        0x7E: 'CRTP',
        0x7F: 'CRUDP',
        0x80: 'SSCOPMCE',
        0x81: 'IPLT',
        0x82: 'SPS',
        0x83: 'PIPE',
        0x84: 'SCTP',
        0x85: 'FC',
        0x86: 'RSVP-E2E-IGNORE',
        0x87: 'Mobility Header',
        0x88: 'UDPLite',
        0x89: 'MPLS-in-IP',
        0x8A: 'manet',
        0x8B: 'HIP',
        0x8C: 'Shim6',
        0x8D: 'WESP',
        0x8E: 'ROHC',
        0x8F: 'Ethernet',
    }
    with PcapFile(filename, oLogfile) as fIn:
        try:
            for timestamp, buffer in dpkt.pcap.Reader(fIn):
                # ----- Put your line processing code here -----
                try:
                    frame = dpkt.ethernet.Ethernet(buffer)
                except KeyboardInterrupt:
                    raise
                except:
                    continue

                if not isinstance(frame.data, dpkt.ip.IP) and not isinstance(frame.data, dpkt.ip6.IP6):
                    continue
                ipPacket = frame.data
                protocolPacket = ipPacket.data
                if isinstance(frame.data.data, dpkt.tcp.TCP):
                    protocol = 'TCP'
                elif isinstance(frame.data.data, dpkt.udp.UDP):
                    protocol = 'UDP'
                else:
                    protocol = dIPProtocolNumbers.get(ipPacket.p, str(ipPacket.p))

                if protocol in ['TCP', 'UDP']:
                    connectionid = (protocol, '%s:%d-%s:%d' % (IP2String(ipPacket.src), protocolPacket.sport, IP2String(ipPacket.dst), protocolPacket.dport))
                else:
                    connectionid = (protocol, '%s-%s' % (IP2String(ipPacket.src), IP2String(ipPacket.dst)))

                if not connectionid in dConnections:
                    dConnections[connectionid] = {ENTRY_HEAD: b'', ENTRY_PREVALENCE: {iter: 0 for iter in range(0x100)}}

                if sys.version_info[0] < 3:
                    data = protocolPacket.data
                    while not isinstance(data, str): # necessary for ICMP, ...
                        data = data.data
                    for char in data:
                        dConnections[connectionid][ENTRY_PREVALENCE][ord(char)] += 1
                        if len(dConnections[connectionid][ENTRY_HEAD]) < 4:
                            dConnections[connectionid][ENTRY_HEAD] += char
                else:
                    data = protocolPacket.data
                    while not isinstance(data, bytes): # necessary for ICMP, ...
                        data = data.data
                    for byte in data:
                        dConnections[connectionid][ENTRY_PREVALENCE][byte] += 1
                        if len(dConnections[connectionid][ENTRY_HEAD]) < 4:
                            dConnections[connectionid][ENTRY_HEAD] += bytes([byte])
                # ----------------------------------------------
        except:
            oLogfile.LineError('Processing file %s %s' % (filename, repr(sys.exc_info()[1])))
            if sys.exc_info()[0] == KeyboardInterrupt:
                raise
            if not options.ignoreprocessingerrors:
                raise
            if sys.version_info[0] < 3:
                sys.exc_clear()

def InstantiateCOutput(options):
    filenameOption = None
    if options.output != '':
        filenameOption = options.output
    return cOutput(filenameOption)

def ProcessPcapFiles(filenames, oLogfile, options):
    dConnections = {}

    if options.processedfilesdb != None:
        data = DeSerialize(options.processedfilesdb)
        if data == None:
            dProcessedFiles = {}
        else:
            dProcessedFiles = data[0]
        atexit.register(Serialize, [dProcessedFiles], options.processedfilesdb)
    else:
        dProcessedFiles = {}

    oOutput = InstantiateCOutput(options)

    for index, filename in enumerate(filenames):
        if not filename in dProcessedFiles:
            oOutput.Filename(filename, index, len(filenames))
            ProcessPcapFile(filename, dConnections, oOutput, oLogfile, options)
            dProcessedFiles[filename] = time.time()

    oOutput.Line(MakeCSVLine(['Protocol', 'ConnectionID', ENTRY_HEAD, 'Size', 'Entropy'], CSV_SEPARATOR, QUOTE))
    for protocol_connectionid, dValues in dConnections.items():
        protocol, connectionid = protocol_connectionid
        sumValues, entropy, countNullByte, countControlBytes, countWhitespaceBytes, countPrintableBytes, countHighBytes = CalculateByteStatistics(dValues[ENTRY_PREVALENCE])
        oOutput.Line(MakeCSVLine([protocol, connectionid, repr(dValues[ENTRY_HEAD]), sumValues, entropy], CSV_SEPARATOR, QUOTE))
    
    oOutput.Close()

def Main():
    moredesc = '''

Arguments:
@file: process each file listed in the text file specified
wildcards are supported

Source code put in the public domain by Didier Stevens, no Copyright
Use at your own risk
https://DidierStevens.com'''

    oParser = optparse.OptionParser(usage='usage: %prog [options] [[@]file ...]\n' + __description__ + moredesc, version='%prog ' + __version__)
    oParser.add_option('-m', '--man', action='store_true', default=False, help='Print manual')
    oParser.add_option('-o', '--output', type=str, default='', help='Output to file (# supported)')
    oParser.add_option('--literalfilenames', action='store_true', default=False, help='Do not interpret filenames')
    oParser.add_option('--recursedir', action='store_true', default=False, help='Recurse directories (wildcards and here files (@...) allowed)')
    oParser.add_option('--checkfilenames', action='store_true', default=False, help='Perform check if files exist prior to file processing')
    oParser.add_option('-p', '--processedfilesdb', default=None, help='File database (pickle) of processed files')
    oParser.add_option('--logfile', type=str, default='', help='Create logfile with given keyword')
    oParser.add_option('--logcomment', type=str, default='', help='A string with comments to be included in the log file')
    oParser.add_option('--ignoreprocessingerrors', action='store_true', default=False, help='Ignore errors during file processing')
    (options, args) = oParser.parse_args()

    if options.man:
        oParser.print_help()
        PrintManual()
        return

    oLogfile = cLogfile(options.logfile, options.logcomment)

    oExpandFilenameArguments = cExpandFilenameArguments(args, options.literalfilenames, options.recursedir, options.checkfilenames)
    oLogfile.Line('FilesCount', str(len(oExpandFilenameArguments.Filenames())))
    oLogfile.Line('Files', repr(oExpandFilenameArguments.Filenames()))
    if oExpandFilenameArguments.warning:
        PrintError('\nWarning:')
        PrintError(oExpandFilenameArguments.message)
        oLogfile.Line('Warning', repr(oExpandFilenameArguments.message))

    ProcessPcapFiles(oExpandFilenameArguments.Filenames(), oLogfile, options)

    if oLogfile.errors > 0:
        PrintError('Number of errors: %d' % oLogfile.errors)
    oLogfile.Close()

if __name__ == '__main__':
    Main()
