/*
 * Decompiled with CFR 0.152.
 */
package com.maltego.cloud.crypto;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.maltego.cloud.CloudAuth;
import com.maltego.cloud.crypto.MasterKeyStore;
import com.maltego.cloud.crypto.util.CryptoUtilities;
import com.maltego.cloud.data.KeyDerivation;
import com.maltego.cloud.data.KeyPair;
import com.maltego.cloud.rest.GraphStoreRestClient;
import com.paterva.maltego.util.MaltegoUrl;
import com.paterva.maltego.util.NormalException;
import java.awt.Cursor;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.IOException;
import java.net.URL;
import java.security.GeneralSecurityException;
import java.security.KeyFactory;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.spec.X509EncodedKeySpec;
import java.util.Base64;
import java.util.UUID;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.prefs.Preferences;
import javax.crypto.SecretKey;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JPasswordField;
import javax.swing.SwingUtilities;
import org.openide.DialogDisplayer;
import org.openide.NotifyDescriptor;
import org.openide.awt.HtmlBrowser;
import org.openide.util.Lookup;
import org.openide.util.NbPreferences;
import org.openide.util.UserCancelException;

public interface CloudGraphKeystore {
    public static CloudGraphKeystore getDefault() {
        return (CloudGraphKeystore)Lookup.getDefault().lookup(CloudGraphKeystore.class);
    }

    public boolean init();

    public void reset();

    public UUID getKeyPairID() throws GeneralSecurityException, IOException;

    public PublicKey getPublicKey() throws GeneralSecurityException, IOException;

    public PrivateKey getPrivateKey() throws GeneralSecurityException, IOException;

    public static class DefaultCloudGraphKeystore
    implements CloudGraphKeystore {
        static final String PREF_KEYPAIR = "CloudGraph.Keypair";
        static final String PREF_MASTER_KEY_LAST_UPDATE = "CloudGraph.MasterKeyLastUpdate";
        static final long MASTER_KEY_UPDATE_INTERVAL = TimeUnit.DAYS.toMillis(14L);
        private static final Logger LOG = Logger.getLogger(DefaultCloudGraphKeystore.class.getName());
        private final Preferences prefs = NbPreferences.forModule(DefaultCloudGraphKeystore.class);
        private final MasterKeyStore masterKeyStore = MasterKeyStore.getDefault();
        private UUID keyPairId;
        private PrivateKey privateKey;
        private PublicKey publicKey;

        @Override
        public boolean init() {
            try {
                String keypairJson = this.prefs.get(PREF_KEYPAIR, null);
                if (keypairJson == null) {
                    LOG.info("Key pair is missing, initialization failed");
                    this.reset();
                    return false;
                }
                long lastUpdate = this.prefs.getLong(PREF_MASTER_KEY_LAST_UPDATE, 0L);
                long timeElapsedSinceLastUpdate = System.currentTimeMillis() - lastUpdate;
                long timeRemaining = MASTER_KEY_UPDATE_INTERVAL - timeElapsedSinceLastUpdate;
                SecretKey masterKey = this.masterKeyStore.readMasterKeyFromStore();
                if (masterKey == null) {
                    LOG.info("Master Key not found");
                    this.scheduleMasterKeyLifetimeCheck(0L);
                } else if (timeRemaining <= 0L) {
                    LOG.info("Master Password lifetime expired, needs to be entered by user again");
                    this.scheduleMasterKeyLifetimeCheck(0L);
                } else {
                    ObjectMapper mapper = new ObjectMapper();
                    KeyPair keyPair = (KeyPair)mapper.readValue(keypairJson, KeyPair.class);
                    this.refreshKeypairObjects(keyPair);
                    LOG.log(Level.INFO, "Scheduling lifetime check for Master Password in {0} seconds", timeRemaining / 1000L);
                    this.scheduleMasterKeyLifetimeCheck(timeRemaining);
                }
                LOG.info("Keystore initialized");
                return true;
            }
            catch (UserCancelException ex) {
                LOG.info("Keystore initialization aborted");
                NotifyDescriptor.Message message = new NotifyDescriptor.Message((Object)"Master Password will be asked on next startup or cloud graph operation", 1);
                message.setTitle("Master Password");
                DialogDisplayer.getDefault().notify((NotifyDescriptor)message);
                return false;
            }
            catch (IOException | GeneralSecurityException ex) {
                NotifyDescriptor.Message message = new NotifyDescriptor.Message((Object)ex.getMessage(), 0);
                message.setTitle("Master Password");
                DialogDisplayer.getDefault().notify((NotifyDescriptor)message);
                LOG.log(Level.WARNING, "Keystore initialization failed", ex);
                return false;
            }
        }

        @Override
        public void reset() {
            LOG.log(Level.INFO, "Reset Cloud Graph Keystore");
            this.keyPairId = null;
            this.publicKey = null;
            this.privateKey = null;
            this.prefs.remove(PREF_KEYPAIR);
            this.prefs.remove(PREF_MASTER_KEY_LAST_UPDATE);
            this.masterKeyStore.reset();
        }

        @Override
        public UUID getKeyPairID() throws GeneralSecurityException, IOException {
            if (this.keyPairId == null) {
                this.fetchFromCloud();
            }
            return this.keyPairId;
        }

        @Override
        public PublicKey getPublicKey() throws GeneralSecurityException, IOException {
            if (this.publicKey == null) {
                this.fetchFromCloud();
            }
            return this.publicKey;
        }

        @Override
        public PrivateKey getPrivateKey() throws GeneralSecurityException, IOException {
            if (this.privateKey == null) {
                this.fetchFromCloud();
            }
            return this.privateKey;
        }

        synchronized void fetchFromCloud() throws GeneralSecurityException, IOException {
            if (!CloudAuth.getDefault().isCloudAuthEnabled()) {
                LOG.info("Cloud auth is disabled, skipping fetching keypair");
                return;
            }
            if (!CloudAuth.getDefault().isAuthorized()) {
                LOG.info("Not logged in to cloud, skipping fetching keypair");
                return;
            }
            LOG.info("Fetching keypair from cloud");
            KeyPair keyPair = GraphStoreRestClient.fetchKeyPair();
            if (keyPair == null) {
                LOG.info("No keypair found, master password should be initialized on the portal");
                CreateMasterPasswordPrompt.show();
                return;
            }
            this.refreshKeypairObjects(keyPair);
            LOG.info("Storing keypair");
            ObjectMapper mapper = new ObjectMapper();
            this.prefs.put(PREF_KEYPAIR, mapper.writeValueAsString((Object)keyPair));
        }

        void refreshKeypairObjects(KeyPair keyPair) throws GeneralSecurityException, IOException {
            PrivateKey privKey;
            SecretKey masterKey = this.masterKeyStore.readMasterKeyFromStore();
            boolean userEnteredMasterPassword = false;
            if (masterKey == null) {
                char[] masterPassword = this.askMasterPassword();
                KeyDerivation passwordKeyDerivationConfig = keyPair.getPasswordKeyDerivationConfig();
                masterKey = CryptoUtilities.createMasterKey(masterPassword, Base64.getDecoder().decode(passwordKeyDerivationConfig.getSalt()), passwordKeyDerivationConfig.getIterations(), passwordKeyDerivationConfig.getLength());
                userEnteredMasterPassword = true;
            }
            byte[] publicKeyBytes = Base64.getDecoder().decode(keyPair.getPublicKey());
            byte[] privateEncrypted = Base64.getDecoder().decode(keyPair.getEncryptedPrivateKey());
            byte[] ivBytes = Base64.getDecoder().decode(keyPair.getPrivateKeyEncryptionConfig().getIv());
            LOG.info("Decrypting keypair");
            KeyFactory keyFactory = KeyFactory.getInstance("RSA");
            PublicKey pubKey = keyFactory.generatePublic(new X509EncodedKeySpec(publicKeyBytes));
            try {
                privKey = CryptoUtilities.decryptPrivateKey(privateEncrypted, masterKey, ivBytes);
            }
            catch (GeneralSecurityException e) {
                LOG.log(Level.WARNING, "Keypair decryption failed. Master key is invalid or wrong.", e);
                throw new GeneralSecurityException("Master password is invalid or wrong.", e);
            }
            this.keyPairId = keyPair.getId();
            this.publicKey = pubKey;
            this.privateKey = privKey;
            if (userEnteredMasterPassword) {
                this.masterKeyStore.storeMasterKey(masterKey);
                this.prefs.putLong(PREF_MASTER_KEY_LAST_UPDATE, System.currentTimeMillis());
                LOG.log(Level.INFO, "Scheduling lifetime check for Master Password in {0} seconds", MASTER_KEY_UPDATE_INTERVAL / 1000L);
                this.scheduleMasterKeyLifetimeCheck(MASTER_KEY_UPDATE_INTERVAL);
            }
            LOG.info("Keypair refreshed");
        }

        char[] askMasterPassword() throws UserCancelException {
            LOG.info("Asking user to enter the master password");
            PasswordPrompt passwordPrompt = new PasswordPrompt();
            if (NotifyDescriptor.OK_OPTION.equals(DialogDisplayer.getDefault().notify((NotifyDescriptor)passwordPrompt))) {
                return passwordPrompt.getPassword();
            }
            throw new UserCancelException("Entering Master Password is aborted");
        }

        void scheduleMasterKeyLifetimeCheck(long timeRemaining) {
            ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor();
            executor.schedule(() -> {
                try {
                    this.reset();
                    this.fetchFromCloud();
                    executor.shutdownNow();
                }
                catch (UserCancelException e) {
                    LOG.info("Entering Master Password is aborted");
                    NotifyDescriptor.Message message = new NotifyDescriptor.Message((Object)"Master Password will be asked on next startup or cloud graph operation", 1);
                    message.setTitle("Master Password");
                    DialogDisplayer.getDefault().notify((NotifyDescriptor)message);
                }
                catch (GeneralSecurityException ex) {
                    NotifyDescriptor.Message message = new NotifyDescriptor.Message((Object)ex.getMessage(), 0);
                    message.setTitle("Master Password");
                    DialogDisplayer.getDefault().notify((NotifyDescriptor)message);
                    LOG.log(Level.WARNING, "Keystore initialization failed", ex);
                }
                catch (Exception e) {
                    NormalException.logStackTrace((Throwable)e);
                }
            }, timeRemaining, TimeUnit.MILLISECONDS);
        }

        private static class CreateMasterPasswordPrompt
        extends NotifyDescriptor {
            private CreateMasterPasswordPrompt() {
                super(null, "Master Password", -1, 1, new Object[0], null);
                final JPanel panel = new JPanel();
                JLabel messageLabel = new JLabel("Your Master Password has not been set. Please set it here:");
                JLabel linkLabel = new JLabel("<html><a href=''>Click here to set Master Password</a></html>");
                linkLabel.setCursor(Cursor.getPredefinedCursor(12));
                panel.add(messageLabel);
                panel.add(linkLabel);
                linkLabel.addMouseListener(new MouseAdapter(){

                    @Override
                    public void mouseClicked(MouseEvent e) {
                        try {
                            URL url = new URL(MaltegoUrl.getCurrent().getGotoMaltegoIDConfigureEncryption());
                            HtmlBrowser.URLDisplayer.getDefault().showURL(url);
                            SwingUtilities.getWindowAncestor(panel).dispose();
                        }
                        catch (IOException ex) {
                            NormalException.logStackTrace((Throwable)ex);
                        }
                    }
                });
                super.setMessage((Object)panel);
            }

            public static void show() {
                DialogDisplayer.getDefault().notify((NotifyDescriptor)new CreateMasterPasswordPrompt());
            }
        }

        private class PasswordPrompt
        extends NotifyDescriptor {
            private JPasswordField passwordField;

            private PasswordPrompt() {
                super(null, "Master Password", 2, 1, null, NotifyDescriptor.OK_OPTION);
                JPanel panel = new JPanel();
                JLabel label = new JLabel("Please enter the Master Password ");
                this.passwordField = new JPasswordField(20);
                panel.add(label);
                panel.add(this.passwordField);
                super.setMessage((Object)panel);
            }

            public char[] getPassword() {
                return this.passwordField.getPassword();
            }
        }
    }
}

