from unittest.mock import MagicMock, patch

from prowler.providers.aws.services.kafka.kafka_service import (
    Cluster,
    EncryptionInTransit,
)
from tests.providers.aws.utils import AWS_REGION_US_EAST_1


class Test_kafka_cluster_mutual_tls_authentication_enabled:
    def test_kafka_no_clusters(self):
        kafka_client = MagicMock
        kafka_client.clusters = {}

        with (
            patch(
                "prowler.providers.aws.services.kafka.kafka_service.Kafka",
                new=kafka_client,
            ),
            patch(
                "prowler.providers.aws.services.kafka.kafka_client.kafka_client",
                new=kafka_client,
            ),
        ):
            from prowler.providers.aws.services.kafka.kafka_cluster_mutual_tls_authentication_enabled.kafka_cluster_mutual_tls_authentication_enabled import (
                kafka_cluster_mutual_tls_authentication_enabled,
            )

            check = kafka_cluster_mutual_tls_authentication_enabled()
            result = check.execute()

            assert len(result) == 0

    def test_kafka_cluster_not_using_mutual_tls_authentication(self):
        kafka_client = MagicMock
        kafka_client.clusters = {
            "arn:aws:kafka:us-east-1:123456789012:cluster/demo-cluster-1/6357e0b2-0e6a-4b86-a0b4-70df934c2e31-5": Cluster(
                id="6357e0b2-0e6a-4b86-a0b4-70df934c2e31-5",
                name="demo-cluster-1",
                arn="arn:aws:kafka:us-east-1:123456789012:cluster/demo-cluster-1/6357e0b2-0e6a-4b86-a0b4-70df934c2e31-5",
                region=AWS_REGION_US_EAST_1,
                tags=[],
                state="ACTIVE",
                kafka_version="2.2.1",
                data_volume_kms_key_id=f"arn:aws:kms:{AWS_REGION_US_EAST_1}:123456789012:key/a7ca56d5-0768-4b64-a670-339a9fbef81c",
                encryption_in_transit=EncryptionInTransit(
                    client_broker="TLS_PLAINTEXT",
                    in_cluster=True,
                ),
                tls_authentication=False,
                public_access=True,
                unauthentication_access=False,
                enhanced_monitoring="DEFAULT",
            )
        }

        with (
            patch(
                "prowler.providers.aws.services.kafka.kafka_service.Kafka",
                new=kafka_client,
            ),
            patch(
                "prowler.providers.aws.services.kafka.kafka_client.kafka_client",
                new=kafka_client,
            ),
        ):
            from prowler.providers.aws.services.kafka.kafka_cluster_mutual_tls_authentication_enabled.kafka_cluster_mutual_tls_authentication_enabled import (
                kafka_cluster_mutual_tls_authentication_enabled,
            )

            check = kafka_cluster_mutual_tls_authentication_enabled()
            result = check.execute()

            assert len(result) == 1
            assert result[0].status == "FAIL"
            assert (
                result[0].status_extended
                == "Kafka cluster 'demo-cluster-1' does not have mutual TLS authentication enabled."
            )
            assert result[0].resource_id == "6357e0b2-0e6a-4b86-a0b4-70df934c2e31-5"
            assert (
                result[0].resource_arn
                == "arn:aws:kafka:us-east-1:123456789012:cluster/demo-cluster-1/6357e0b2-0e6a-4b86-a0b4-70df934c2e31-5"
            )
            assert result[0].region == AWS_REGION_US_EAST_1
            assert result[0].resource_tags == []

    def test_kafka_cluster_using_mutual_tls_authentication(self):
        kafka_client = MagicMock
        kafka_client.clusters = {
            "arn:aws:kafka:us-east-1:123456789012:cluster/demo-cluster-1/6357e0b2-0e6a-4b86-a0b4-70df934c2e31-5": Cluster(
                id="6357e0b2-0e6a-4b86-a0b4-70df934c2e31-5",
                name="demo-cluster-1",
                arn="arn:aws:kafka:us-east-1:123456789012:cluster/demo-cluster-1/6357e0b2-0e6a-4b86-a0b4-70df934c2e31-5",
                region=AWS_REGION_US_EAST_1,
                tags=[],
                state="ACTIVE",
                kafka_version="2.2.1",
                data_volume_kms_key_id=f"arn:aws:kms:{AWS_REGION_US_EAST_1}:123456789012:key/a7ca56d5-0768-4b64-a670-339a9fbef81c",
                encryption_in_transit=EncryptionInTransit(
                    client_broker="TLS",
                    in_cluster=True,
                ),
                tls_authentication=True,
                public_access=True,
                unauthentication_access=False,
                enhanced_monitoring="DEFAULT",
            )
        }

        with (
            patch(
                "prowler.providers.aws.services.kafka.kafka_service.Kafka",
                new=kafka_client,
            ),
            patch(
                "prowler.providers.aws.services.kafka.kafka_client.kafka_client",
                new=kafka_client,
            ),
        ):
            from prowler.providers.aws.services.kafka.kafka_cluster_mutual_tls_authentication_enabled.kafka_cluster_mutual_tls_authentication_enabled import (
                kafka_cluster_mutual_tls_authentication_enabled,
            )

            check = kafka_cluster_mutual_tls_authentication_enabled()
            result = check.execute()

            assert len(result) == 1
            assert result[0].status == "PASS"
            assert (
                result[0].status_extended
                == "Kafka cluster 'demo-cluster-1' has mutual TLS authentication enabled."
            )
            assert result[0].resource_id == "6357e0b2-0e6a-4b86-a0b4-70df934c2e31-5"
            assert (
                result[0].resource_arn
                == "arn:aws:kafka:us-east-1:123456789012:cluster/demo-cluster-1/6357e0b2-0e6a-4b86-a0b4-70df934c2e31-5"
            )
            assert result[0].region == AWS_REGION_US_EAST_1
            assert result[0].resource_tags == []

    def test_kafka_cluster_serverless_mutual_tls_authentication(self):
        kafka_client = MagicMock
        kafka_client.clusters = {
            "arn:aws:kafka:us-east-1:123456789012:cluster/serverless-cluster-1/6357e0b2-0e6a-4b86-a0b4-70df934c2e31-6": Cluster(
                id="6357e0b2-0e6a-4b86-a0b4-70df934c2e31-6",
                name="serverless-cluster-1",
                arn="arn:aws:kafka:us-east-1:123456789012:cluster/serverless-cluster-1/6357e0b2-0e6a-4b86-a0b4-70df934c2e31-6",
                region=AWS_REGION_US_EAST_1,
                tags=[],
                state="ACTIVE",
                kafka_version="SERVERLESS",
                data_volume_kms_key_id="AWS_MANAGED",
                encryption_in_transit=EncryptionInTransit(
                    client_broker="TLS",
                    in_cluster=True,
                ),
                tls_authentication=True,
                public_access=False,
                unauthentication_access=False,
                enhanced_monitoring="DEFAULT",
            )
        }

        with (
            patch(
                "prowler.providers.aws.services.kafka.kafka_service.Kafka",
                new=kafka_client,
            ),
            patch(
                "prowler.providers.aws.services.kafka.kafka_client.kafka_client",
                new=kafka_client,
            ),
        ):
            from prowler.providers.aws.services.kafka.kafka_cluster_mutual_tls_authentication_enabled.kafka_cluster_mutual_tls_authentication_enabled import (
                kafka_cluster_mutual_tls_authentication_enabled,
            )

            check = kafka_cluster_mutual_tls_authentication_enabled()
            result = check.execute()

            assert len(result) == 1
            assert result[0].status == "PASS"
            assert (
                result[0].status_extended
                == "Kafka cluster 'serverless-cluster-1' is serverless and always has TLS authentication enabled by default."
            )
            assert result[0].resource_id == "6357e0b2-0e6a-4b86-a0b4-70df934c2e31-6"
            assert (
                result[0].resource_arn
                == "arn:aws:kafka:us-east-1:123456789012:cluster/serverless-cluster-1/6357e0b2-0e6a-4b86-a0b4-70df934c2e31-6"
            )
            assert result[0].region == AWS_REGION_US_EAST_1
            assert result[0].resource_tags == []
