# frozen_string_literal: true

module Wpxf
  # Provides functionality for authenticating modules with a WordPress target.
  module ModuleAuthentication
    # Initialize a new instance of {ModuleAuthentication}.
    def initialize
      super
      return unless requires_authentication

      register_options([
        StringOption.new(
          name: 'username',
          desc: 'The WordPress username to authenticate with',
          required: true
        ),
        StringOption.new(
          name: 'password',
          desc: 'The WordPress password to authenticate with',
          required: true
        )
      ])
    end

    # @return [Boolean] true if the module requires the user to authenticate.
    def requires_authentication
      false
    end

    # Authenticate with WordPress and return the cookie.
    # @param username [String] the username to authenticate with.
    # @param password [String] the password to authenticate with.
    # @return [CookieJar, Boolean] the cookie in a CookieJar if successful,
    #   otherwise, returns false.
    def authenticate_with_wordpress(username, password)
      emit_info "Authenticating with WordPress using #{username}:#{password}..."
      cookie = wordpress_login(username, password)
      if cookie.nil?
        emit_error 'Failed to authenticate with WordPress'
        return false
      else
        store_credentials username, password
        emit_success 'Authenticated with WordPress', true
        return cookie
      end
    end
  end
end
