# frozen_string_literal: true

require_relative '../../../../../spec_helper'
require 'wpxf/modules'

describe Wpxf::Auxiliary::DirectDownloadForWoocommerceFileDownload do
  let(:subject) { described_class.new }
  let(:http_res) { Wpxf::Net::HttpResponse.new(nil) }

  before :each, 'setup subject' do
    allow(subject).to receive(:execute_get_request).and_return(http_res)
  end

  it 'should return a {Wpxf::Module}' do
    expect(subject).to be_a Wpxf::Module
  end

  context 'if the direct download page returns an error message' do
    it 'should presume the target is vulnerable' do
      http_res.body = 'This product is not available for direct free download'
      expect(subject.check).to eql :vulnerable
    end
  end

  context 'if the direct download page does not appear to exist' do
    it 'should report an unknown state when checking the target' do
      expect(subject.check).to eql :unknown
    end
  end

  it 'should register a product_id option' do
    expect(subject.get_option('product_id')).to be_a Wpxf::IntegerOption
  end

  it 'should not require authentication' do
    expect(subject.requires_authentication).to be false
  end

  it 'should configure a default remote file path' do
    expected = 'wp-config.php'
    expect(subject.default_remote_file_path).to eql expected
  end

  it 'should configure the working directory' do
    expected = 'the WordPress installation directory'
    expect(subject.working_directory).to eql expected
  end

  it 'should configure the downloader url' do
    subject.set_option_value('product_id', 5)
    subject.set_option_value('remote_file', 'test.php')
    expect(subject.downloader_url).to match(%r{/direct-download/NXx0ZXN0LnBocA==})
  end

  it 'should GET the download request' do
    expect(subject.download_request_method).to eql :get
  end

  context 'if the result contains an error message from the plugin' do
    it 'should fail the content validation' do
      res = subject.validate_content('This product is not available for direct free download')
      expect(res).to be false
    end
  end

  context 'if the result does not contain an error message from the plugin' do
    it 'should pass the content validation' do
      res = subject.validate_content('success!')
      expect(res).to be true
    end
  end
end
