"use strict";
/**
 * Copyright (c) "Neo4j"
 * Neo4j Sweden AB [http://neo4j.com]
 *
 * This file is part of Neo4j.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
var neo4j_driver_core_1 = require("neo4j-driver-core");
var _a = neo4j_driver_core_1.internal.constants, ACCESS_MODE_READ = _a.ACCESS_MODE_READ, FETCH_ALL = _a.FETCH_ALL, assertString = neo4j_driver_core_1.internal.util.assertString;
/* eslint-disable no-unused-vars */
// Signature bytes for each request message type
var INIT = 0x01; // 0000 0001 // INIT <user_agent> <authentication_token>
var ACK_FAILURE = 0x0e; // 0000 1110 // ACK_FAILURE - unused
var RESET = 0x0f; // 0000 1111 // RESET
var RUN = 0x10; // 0001 0000 // RUN <query> <parameters>
var DISCARD_ALL = 0x2f; // 0010 1111 // DISCARD_ALL - unused
var PULL_ALL = 0x3f; // 0011 1111 // PULL_ALL
var HELLO = 0x01; // 0000 0001 // HELLO <metadata>
var GOODBYE = 0x02; // 0000 0010 // GOODBYE
var BEGIN = 0x11; // 0001 0001 // BEGIN <metadata>
var COMMIT = 0x12; // 0001 0010 // COMMIT
var ROLLBACK = 0x13; // 0001 0011 // ROLLBACK
var ROUTE = 0x66; // 0110 0110 // ROUTE
var DISCARD = 0x2f; // 0010 1111 // DISCARD
var PULL = 0x3f; // 0011 1111 // PULL
var READ_MODE = 'r';
/* eslint-enable no-unused-vars */
var NO_STATEMENT_ID = -1;
var RequestMessage = /** @class */ (function () {
    function RequestMessage(signature, fields, toString) {
        this.signature = signature;
        this.fields = fields;
        this.toString = toString;
    }
    /**
     * Create a new INIT message.
     * @param {string} clientName the client name.
     * @param {Object} authToken the authentication token.
     * @return {RequestMessage} new INIT message.
     */
    RequestMessage.init = function (clientName, authToken) {
        return new RequestMessage(INIT, [clientName, authToken], function () { return "INIT " + clientName + " {...}"; });
    };
    /**
     * Create a new RUN message.
     * @param {string} query the cypher query.
     * @param {Object} parameters the query parameters.
     * @return {RequestMessage} new RUN message.
     */
    RequestMessage.run = function (query, parameters) {
        return new RequestMessage(RUN, [query, parameters], function () { return "RUN " + query + " " + neo4j_driver_core_1.json.stringify(parameters); });
    };
    /**
     * Get a PULL_ALL message.
     * @return {RequestMessage} the PULL_ALL message.
     */
    RequestMessage.pullAll = function () {
        return PULL_ALL_MESSAGE;
    };
    /**
     * Get a RESET message.
     * @return {RequestMessage} the RESET message.
     */
    RequestMessage.reset = function () {
        return RESET_MESSAGE;
    };
    /**
     * Create a new HELLO message.
     * @param {string} userAgent the user agent.
     * @param {Object} authToken the authentication token.
     * @param {Object} optional server side routing, set to routing context to turn on server side routing (> 4.1)
     * @return {RequestMessage} new HELLO message.
     */
    RequestMessage.hello = function (userAgent, authToken, routing) {
        if (routing === void 0) { routing = null; }
        var metadata = Object.assign({ user_agent: userAgent }, authToken);
        if (routing) {
            metadata.routing = routing;
        }
        return new RequestMessage(HELLO, [metadata], function () { return "HELLO {user_agent: '" + userAgent + "', ...}"; });
    };
    /**
     * Create a new BEGIN message.
     * @param {Bookmark} bookmark the bookmark.
     * @param {TxConfig} txConfig the configuration.
     * @param {string} database the database name.
     * @param {string} mode the access mode.
     * @param {string} impersonatedUser the impersonated user.
     * @return {RequestMessage} new BEGIN message.
     */
    RequestMessage.begin = function (_a) {
        var _b = _a === void 0 ? {} : _a, bookmark = _b.bookmark, txConfig = _b.txConfig, database = _b.database, mode = _b.mode, impersonatedUser = _b.impersonatedUser;
        var metadata = buildTxMetadata(bookmark, txConfig, database, mode, impersonatedUser);
        return new RequestMessage(BEGIN, [metadata], function () { return "BEGIN " + neo4j_driver_core_1.json.stringify(metadata); });
    };
    /**
     * Get a COMMIT message.
     * @return {RequestMessage} the COMMIT message.
     */
    RequestMessage.commit = function () {
        return COMMIT_MESSAGE;
    };
    /**
     * Get a ROLLBACK message.
     * @return {RequestMessage} the ROLLBACK message.
     */
    RequestMessage.rollback = function () {
        return ROLLBACK_MESSAGE;
    };
    /**
     * Create a new RUN message with additional metadata.
     * @param {string} query the cypher query.
     * @param {Object} parameters the query parameters.
     * @param {Bookmark} bookmark the bookmark.
     * @param {TxConfig} txConfig the configuration.
     * @param {string} database the database name.
     * @param {string} mode the access mode.
     * @param {string} impersonatedUser the impersonated user.
     * @return {RequestMessage} new RUN message with additional metadata.
     */
    RequestMessage.runWithMetadata = function (query, parameters, _a) {
        var _b = _a === void 0 ? {} : _a, bookmark = _b.bookmark, txConfig = _b.txConfig, database = _b.database, mode = _b.mode, impersonatedUser = _b.impersonatedUser;
        var metadata = buildTxMetadata(bookmark, txConfig, database, mode, impersonatedUser);
        return new RequestMessage(RUN, [query, parameters, metadata], function () {
            return "RUN " + query + " " + neo4j_driver_core_1.json.stringify(parameters) + " " + neo4j_driver_core_1.json.stringify(metadata);
        });
    };
    /**
     * Get a GOODBYE message.
     * @return {RequestMessage} the GOODBYE message.
     */
    RequestMessage.goodbye = function () {
        return GOODBYE_MESSAGE;
    };
    /**
     * Generates a new PULL message with additional metadata.
     * @param {Integer|number} stmtId
     * @param {Integer|number} n
     * @return {RequestMessage} the PULL message.
     */
    RequestMessage.pull = function (_a) {
        var _b = _a === void 0 ? {} : _a, _c = _b.stmtId, stmtId = _c === void 0 ? NO_STATEMENT_ID : _c, _d = _b.n, n = _d === void 0 ? FETCH_ALL : _d;
        var metadata = buildStreamMetadata(stmtId === null || stmtId === undefined ? NO_STATEMENT_ID : stmtId, n || FETCH_ALL);
        return new RequestMessage(PULL, [metadata], function () { return "PULL " + neo4j_driver_core_1.json.stringify(metadata); });
    };
    /**
     * Generates a new DISCARD message with additional metadata.
     * @param {Integer|number} stmtId
     * @param {Integer|number} n
     * @return {RequestMessage} the PULL message.
     */
    RequestMessage.discard = function (_a) {
        var _b = _a === void 0 ? {} : _a, _c = _b.stmtId, stmtId = _c === void 0 ? NO_STATEMENT_ID : _c, _d = _b.n, n = _d === void 0 ? FETCH_ALL : _d;
        var metadata = buildStreamMetadata(stmtId === null || stmtId === undefined ? NO_STATEMENT_ID : stmtId, n || FETCH_ALL);
        return new RequestMessage(DISCARD, [metadata], function () { return "DISCARD " + neo4j_driver_core_1.json.stringify(metadata); });
    };
    /**
     * Generate the ROUTE message, this message is used to fetch the routing table from the server
     *
     * @param {object} routingContext The routing context used to define the routing table. Multi-datacenter deployments is one of its use cases
     * @param {string[]} bookmarks The list of the bookmark should be used
     * @param {string} databaseName The name of the database to get the routing table for.
     * @return {RequestMessage} the ROUTE message.
     */
    RequestMessage.route = function (routingContext, bookmarks, databaseName) {
        if (routingContext === void 0) { routingContext = {}; }
        if (bookmarks === void 0) { bookmarks = []; }
        if (databaseName === void 0) { databaseName = null; }
        return new RequestMessage(ROUTE, [routingContext, bookmarks, databaseName], function () {
            return "ROUTE " + neo4j_driver_core_1.json.stringify(routingContext) + " " + neo4j_driver_core_1.json.stringify(bookmarks) + " " + databaseName;
        });
    };
    /**
     * Generate the ROUTE message, this message is used to fetch the routing table from the server
     *
     * @param {object} routingContext The routing context used to define the routing table. Multi-datacenter deployments is one of its use cases
     * @param {string[]} bookmarks The list of the bookmark should be used
     * @param {object} databaseContext The context inforamtion of the database to get the routing table for.
     * @param {string} databaseContext.databaseName The name of the database to get the routing table.
     * @param {string} databaseContext.impersonatedUser The name of the user to impersonation when getting the routing table.
     * @return {RequestMessage} the ROUTE message.
     */
    RequestMessage.routeV4x4 = function (routingContext, bookmarks, databaseContext) {
        if (routingContext === void 0) { routingContext = {}; }
        if (bookmarks === void 0) { bookmarks = []; }
        if (databaseContext === void 0) { databaseContext = {}; }
        var dbContext = {};
        if (databaseContext.databaseName) {
            dbContext.db = databaseContext.databaseName;
        }
        if (databaseContext.impersonatedUser) {
            dbContext.imp_user = databaseContext.impersonatedUser;
        }
        return new RequestMessage(ROUTE, [routingContext, bookmarks, dbContext], function () {
            return "ROUTE " + neo4j_driver_core_1.json.stringify(routingContext) + " " + neo4j_driver_core_1.json.stringify(bookmarks) + " " + neo4j_driver_core_1.json.stringify(dbContext);
        });
    };
    return RequestMessage;
}());
exports.default = RequestMessage;
/**
 * Create an object that represent transaction metadata.
 * @param {Bookmark} bookmark the bookmark.
 * @param {TxConfig} txConfig the configuration.
 * @param {string} database the database name.
 * @param {string} mode the access mode.
 * @param {string} impersonatedUser the impersonated user mode.
 * @return {Object} a metadata object.
 */
function buildTxMetadata(bookmark, txConfig, database, mode, impersonatedUser) {
    var metadata = {};
    if (!bookmark.isEmpty()) {
        metadata.bookmarks = bookmark.values();
    }
    if (txConfig.timeout) {
        metadata.tx_timeout = txConfig.timeout;
    }
    if (txConfig.metadata) {
        metadata.tx_metadata = txConfig.metadata;
    }
    if (database) {
        metadata.db = assertString(database, 'database');
    }
    if (impersonatedUser) {
        metadata.imp_user = assertString(impersonatedUser, 'impersonatedUser');
    }
    if (mode === ACCESS_MODE_READ) {
        metadata.mode = READ_MODE;
    }
    return metadata;
}
/**
 * Create an object that represents streaming metadata.
 * @param {Integer|number} stmtId The query id to stream its results.
 * @param {Integer|number} n The number of records to stream.
 * @returns {Object} a metadata object.
 */
function buildStreamMetadata(stmtId, n) {
    var metadata = { n: (0, neo4j_driver_core_1.int)(n) };
    if (stmtId !== NO_STATEMENT_ID) {
        metadata.qid = (0, neo4j_driver_core_1.int)(stmtId);
    }
    return metadata;
}
// constants for messages that never change
var PULL_ALL_MESSAGE = new RequestMessage(PULL_ALL, [], function () { return 'PULL_ALL'; });
var RESET_MESSAGE = new RequestMessage(RESET, [], function () { return 'RESET'; });
var COMMIT_MESSAGE = new RequestMessage(COMMIT, [], function () { return 'COMMIT'; });
var ROLLBACK_MESSAGE = new RequestMessage(ROLLBACK, [], function () { return 'ROLLBACK'; });
var GOODBYE_MESSAGE = new RequestMessage(GOODBYE, [], function () { return 'GOODBYE'; });
