import type { CommonClientOptions, OperationOptions } from "@azure/core-client";
import type { TableGetAccessPolicyHeaders, TableInsertEntityHeaders } from "./generated/models/index.js";
/**
 * Represents the Create or Delete Entity operation to be included in a Transaction request
 */
export type CreateDeleteEntityAction = ["create" | "delete", TableEntity];
/**
 * Represents the Update or Upsert Entity operation to be included in a Transaction request
 */
export type UpdateEntityAction = ["update" | "upsert", TableEntity] | ["update" | "upsert", TableEntity, "Merge" | "Replace"] | ["update" | "upsert", TableEntity, "Merge" | "Replace", UpdateTableEntityOptions | undefined];
/**
 * Represents the union of all the available transactional actions
 */
export type TransactionAction = CreateDeleteEntityAction | UpdateEntityAction;
/**
 * Client options used to configure Tables Api requests
 */
export type TableServiceClientOptions = CommonClientOptions & {
    endpoint?: string;
    version?: string;
};
/**
 * Contains response data for the createEntity operation.
 */
export type CreateTableEntityResponse = TableInsertEntityHeaders;
/**
 * Contains response data for the listEntities operation.
 */
export type GetTableEntityResponse<T extends object> = TableEntityResult<T>;
/**
 * Optional parameters for DeleteTableEntity operation
 */
export type DeleteTableEntityOptions = OperationOptions & {
    /**
     * UTC date/time value generated by the service that indicates the time at which the response was initiated
     */
    etag?: string;
};
/** The properties for the table item. */
export interface TableItem {
    /** The name of the table. */
    name?: string;
}
/**
 * OData Query options to limit the set of tables returned.
 */
export interface TableQueryOptions {
    /**
     * OData filter expression.
     */
    filter?: string;
}
/**
 * OData Query options to limit the set of entities returned.
 */
export interface TableEntityQueryOptions {
    /**
     * OData filter expression.
     */
    filter?: string;
    /**
     * A select expression limits the properties on each entity to just those requested.
     */
    select?: string[];
}
/**
 * List tables optional parameters.
 */
export type ListTableItemsOptions = OperationOptions & {
    /**
     * Query options group
     */
    queryOptions?: TableQueryOptions;
};
/**
 * Output type for query operations
 */
export type TableEntityResult<T> = T & {
    /**
     * etag property. Always returned by the service
     */
    etag: string;
    /**
     * Partition key property. Omitted if a select filter is set and this property is not requested
     */
    partitionKey?: string;
    /**
     * Row key property. Omitted if a select filter is set and this property is not requested
     */
    rowKey?: string;
    /**
     * Timestamp property. This property is assinged by the service on entity creation
     * Omitted if a select filter is set and this property is not requested
     */
    timestamp?: string;
};
/**
 * Output page type for query operations
 */
export type TableEntityResultPage<T> = Array<TableEntityResult<T>> & {
    /**
     * Continuation token to get the next page
     */
    continuationToken?: string;
};
/**
 * List entities optional parameters.
 */
export type ListTableEntitiesOptions = OperationOptions & {
    /**
     * Query options group
     */
    queryOptions?: TableEntityQueryOptions;
    /**
     * If true, automatic type conversion will be disabled and entity properties will
     * be represented by full metadata types. For example, an Int32 value will be \{value: "123", type: "Int32"\} instead of 123.
     * This option applies for all the properties
     */
    disableTypeConversion?: boolean;
};
/**
 * GetEntity optional parameters.
 */
export type GetTableEntityOptions = OperationOptions & {
    /**
     * Parameter group
     */
    queryOptions?: TableEntityQueryOptions;
    /**
     * If true, automatic type conversion will be disabled and entity properties will
     * be represented by full metadata types. For example, an Int32 value will be \{value: "123", type: "Int32"\} instead of 123.
     * This option applies for all the properties
     */
    disableTypeConversion?: boolean;
};
/**
 * Update entity optional parameters.
 */
export type UpdateTableEntityOptions = OperationOptions & {
    /**
     * Match condition for an entity to be updated. If specified and a matching entity is not found, an error will be raised. To force an unconditional update, set to the wildcard character (*). If not specified, an insert will be performed when no existing entity is found to update and a replace will be performed if an existing entity is found.
     */
    etag?: string;
};
/**
 * A set of key-value pairs representing the table entity.
 */
export type TableEntity<T extends object = Record<string, unknown>> = T & {
    /**
     * The PartitionKey property of the entity.
     */
    partitionKey: string;
    /**
     * The RowKey property of the entity.
     */
    rowKey: string;
};
/**
 * Supported EDM Types by Azure Tables.
 */
export type EdmTypes = "Binary" | "Boolean" | "DateTime" | "Double" | "Guid" | "Int32" | "Int64" | "String";
/**
 * Entity Data Model representation for an entity property.
 */
export interface Edm<T extends EdmTypes> {
    /**
     * The value of the entity property
     */
    value: T extends "Binary" ? string : T extends "Boolean" ? boolean : T extends "Double" ? number : T extends "Int32" ? number : string;
    /**
     * The type of the entity property
     */
    type: T;
}
/**
 * The different modes for Update and Upsert methods
 * - Merge: Updates an entity by updating the entity's properties without replacing the existing entity.
 * - Replace: Updates an existing entity by replacing the entire entity.
 */
export type UpdateMode = "Merge" | "Replace";
/**
 * Represents the response of a Transaction operation
 */
export interface TableTransactionResponse {
    /**
     * Collection of sub responses
     */
    subResponses: TableTransactionEntityResponse[];
    /**
     * Main Transaction request status code
     */
    status: number;
    /**
     * Gets a specific response given a row key
     */
    getResponseForEntity: (rowKey: string) => TableTransactionEntityResponse | undefined;
}
/** The properties for the table query response. */
export interface TableQueryResponse {
    /** List of tables. */
    value?: TableItem[];
}
/**
 * Output page type for table query operations
 */
export interface TableItemResultPage extends Array<TableItem> {
    /**
     * Continuation token to get the next TableItem page
     */
    continuationToken?: string;
}
/**
 * Represents a sub-response of a Transaction operation
 */
export interface TableTransactionEntityResponse {
    /**
     * Entity's etag
     */
    etag?: string;
    /**
     * Entity's rowKey
     */
    rowKey?: string;
    /**
     * Sub-response status
     */
    status: number;
}
/** A signed identifier. */
export interface SignedIdentifier {
    /** A unique id. */
    id: string;
    /** The access policy. */
    accessPolicy?: AccessPolicy;
}
/** An Access policy. */
export interface AccessPolicy {
    /** The start datetime from which the policy is active. */
    start?: Date;
    /** The datetime that the policy expires. */
    expiry?: Date;
    /** The permissions for the acl policy. */
    permission?: string;
}
/** Contains response data for the getAccessPolicy operation. */
export type GetAccessPolicyResponse = TableGetAccessPolicyHeaders & SignedIdentifier[];
//# sourceMappingURL=models.d.ts.map