"use strict";
// Copyright (c) Microsoft Corporation.
// Licensed under the MIT License.
Object.defineProperty(exports, "__esModule", { value: true });
exports.accountSasPermissionsFromString = accountSasPermissionsFromString;
exports.accountSasPermissionsToString = accountSasPermissionsToString;
/**
 * Parse initializes the AccountSASPermissions fields from a string.
 *
 * @param permissions -
 */
function accountSasPermissionsFromString(permissions) {
    const accountSasPermissions = {};
    for (const c of permissions) {
        switch (c) {
            case "r":
                accountSasPermissions.query = true;
                break;
            case "w":
                accountSasPermissions.write = true;
                break;
            case "d":
                accountSasPermissions.delete = true;
                break;
            case "l":
                accountSasPermissions.list = true;
                break;
            case "a":
                accountSasPermissions.add = true;
                break;
            case "u":
                accountSasPermissions.update = true;
                break;
            default:
                throw new RangeError(`Invalid permission character: ${c}`);
        }
    }
    return accountSasPermissions;
}
/**
 * Produces the SAS permissions string for an Azure Storage account.
 * Call this method to set AccountSASSignatureValues Permissions field.
 *
 * Using this method will guarantee the resource types are in
 * an order accepted by the service.
 *
 * @see https://learn.microsoft.com/rest/api/storageservices/constructing-an-account-sas
 *
 */
function accountSasPermissionsToString(permissions) {
    // The order of the characters should be as specified here to ensure correctness:
    // https://learn.microsoft.com/rest/api/storageservices/constructing-an-account-sas
    // Use a string array instead of string concatenating += operator for performance
    const permissionString = [];
    if (permissions.query) {
        permissionString.push("r");
    }
    if (permissions.write) {
        permissionString.push("w");
    }
    if (permissions.delete) {
        permissionString.push("d");
    }
    if (permissions.list) {
        permissionString.push("l");
    }
    if (permissions.add) {
        permissionString.push("a");
    }
    if (permissions.update) {
        permissionString.push("u");
    }
    return permissionString.join("");
}
//# sourceMappingURL=accountSasPermissions.js.map