// Copyright (c) Microsoft Corporation.
// Licensed under the MIT License.
import { ipRangeToString } from "./sasIPRange.js";
import { SasQueryParameters } from "./sasQueryParameters.js";
import { tableSasPermissionsToString } from "./tableSasPermisions.js";
import { SERVICE_VERSION } from "../utils/constants.js";
import { computeHMACSHA256 } from "../utils/computeHMACSHA256.js";
import { truncatedISO8061Date } from "../utils/truncateISO8061Date.js";
/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * Creates an instance of SASQueryParameters.
 *
 * **Note**: When identifier is not provided, permissions has a default value of "read" and expiresOn of one hour from the time the token is generated.
 */
export function generateTableSasQueryParameters(tableName, credential, tableSasSignatureValues) {
    const version = tableSasSignatureValues.version ?? SERVICE_VERSION;
    if (credential === undefined) {
        throw TypeError("Invalid NamedKeyCredential");
    }
    if (!tableName) {
        throw new Error("Must provide a 'tableName'");
    }
    const signedPermissions = tableSasPermissionsToString(tableSasSignatureValues.permissions);
    const signedStart = tableSasSignatureValues.startsOn
        ? truncatedISO8061Date(tableSasSignatureValues.startsOn, false /** withMilliseconds */)
        : "";
    const signedExpiry = tableSasSignatureValues.expiresOn
        ? truncatedISO8061Date(tableSasSignatureValues.expiresOn, false /** withMilliseconds */)
        : "";
    const canonicalizedResource = getCanonicalName(credential.name, tableName);
    const signedIdentifier = tableSasSignatureValues.identifier ?? "";
    const signedIP = ipRangeToString(tableSasSignatureValues.ipRange);
    const signedProtocol = tableSasSignatureValues.protocol || "";
    const startingPartitionKey = tableSasSignatureValues.startPartitionKey ?? "";
    const startingRowKey = tableSasSignatureValues.startRowKey ?? "";
    const endingPartitionKey = tableSasSignatureValues.endPartitionKey ?? "";
    const endingRowKey = tableSasSignatureValues.endRowKey ?? "";
    const stringToSign = [
        signedPermissions,
        signedStart,
        signedExpiry,
        canonicalizedResource,
        signedIdentifier,
        signedIP,
        signedProtocol,
        version,
        startingPartitionKey,
        startingRowKey,
        endingPartitionKey,
        endingRowKey,
    ].join("\n");
    const signature = computeHMACSHA256(stringToSign, credential.key);
    return new SasQueryParameters(version, signature, {
        permissions: signedPermissions,
        protocol: tableSasSignatureValues.protocol,
        startsOn: tableSasSignatureValues.startsOn,
        expiresOn: tableSasSignatureValues.expiresOn,
        ipRange: tableSasSignatureValues.ipRange,
        identifier: tableSasSignatureValues.identifier,
        tableName,
        startPartitionKey: tableSasSignatureValues.startPartitionKey,
        startRowKey: tableSasSignatureValues.startRowKey,
        endPartitionKey: tableSasSignatureValues.endPartitionKey,
        endRowKey: tableSasSignatureValues.endRowKey,
    });
}
function getCanonicalName(accountName, tableName) {
    // Sample CanonicalName for URL = https://myaccount.table.core.windows.net/Employees(PartitionKey='Jeff',RowKey='Price'):
    //   canonicalizedResource = "/table/myaccount/employees"
    return `/table/${accountName}/${tableName.toLowerCase()}`;
}
//# sourceMappingURL=tableSasSignatureValues.js.map