import type { AbortSignalLike } from "@azure/abort-controller";
import type { FileServiceProperties, ShareCreateResponse, ShareDeleteResponse, ServiceGetPropertiesResponse, ServiceSetPropertiesResponse, ServiceListSharesSegmentHeaders, ListSharesResponseModel, SharePropertiesInternal } from "./generatedModels.js";
import type { Pipeline } from "./Pipeline.js";
import type { CommonOptions } from "./StorageClient.js";
import { StorageClient } from "./StorageClient.js";
import type { ShareCreateOptions, ShareDeleteMethodOptions } from "./Clients.js";
import { ShareClient } from "./Clients.js";
import type { WithResponse } from "./utils/utils.common.js";
import { Credential } from "@azure/storage-common";
import type { PagedAsyncIterableIterator } from "@azure/core-paging";
import type { ShareClientConfig, ShareClientOptions, ShareProtocols } from "./models.js";
import { AccountSASPermissions } from "./AccountSASPermissions.js";
import type { SASProtocol } from "./SASQueryParameters.js";
import type { SasIPRange } from "./SasIPRange.js";
import type { TokenCredential } from "@azure/core-auth";
/**
 * Options to configure the {@link ShareServiceClient.listShares} operation.
 */
export interface ServiceListSharesOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * Filters the results to return only entries whose
     * name begins with the specified prefix.
     */
    prefix?: string;
    /**
     * Specifies that share snapshots should be included in the enumeration. Share Snapshots are listed from oldest to newest in the response.
     */
    includeMetadata?: boolean;
    /**
     * Specifies that share snapshot should be returned in the response.
     */
    includeSnapshots?: boolean;
    /**
     * Specifies that share soft deleted should be returned in the response.
     */
    includeDeleted?: boolean;
}
/**
 * Options to configure the {@link ShareServiceClient.getProperties} operation.
 */
export interface ServiceGetPropertiesOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
}
/**
 * Options to configure the {@link ShareServiceClient.setProperties} operation.
 */
export interface ServiceSetPropertiesOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
}
/**
 * Options to configure the {@link ShareServiceClient.undelete} operation.
 */
export interface ServiceUndeleteShareOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
}
/**
 * Properties of a share.
 */
export type ShareProperties = SharePropertiesInternal & {
    /**
     * The protocols that have been enabled on the share.
     */
    protocols?: ShareProtocols;
};
/**
 * A listed Azure Storage share item.
 */
export interface ShareItem {
    name: string;
    snapshot?: string;
    deleted?: boolean;
    version?: string;
    properties: ShareProperties;
    metadata?: {
        [propertyName: string]: string;
    };
}
/**
 * An enumeration of shares.
 */
export interface ListSharesResponse {
    serviceEndpoint: string;
    prefix?: string;
    marker?: string;
    maxResults?: number;
    shareItems?: ShareItem[];
    continuationToken: string;
}
/**
 * Contains response data for the {@link ShareServiceClient.listShares} operation.
 */
export type ServiceListSharesSegmentResponse = WithResponse<ListSharesResponse & ServiceListSharesSegmentHeaders, ServiceListSharesSegmentHeaders, ListSharesResponseModel>;
/**
 * Options to configure {@link ShareServiceClient.generateAccountSasUrl} operation.
 */
export interface ServiceGenerateAccountSasUrlOptions {
    /**
     * The version of the service this SAS will target. If not specified, it will default to the version targeted by the
     * library.
     */
    version?: string;
    /**
     * Optional. SAS protocols allowed.
     */
    protocol?: SASProtocol;
    /**
     * Optional. When the SAS will take effect.
     */
    startsOn?: Date;
    /**
     * Optional. IP range allowed.
     */
    ipRange?: SasIPRange;
}
/**
 * A ShareServiceClient represents a URL to the Azure Storage File service allowing you
 * to manipulate file shares.
 */
export declare class ShareServiceClient extends StorageClient {
    /**
     * serviceContext provided by protocol layer.
     */
    private serviceContext;
    private shareClientConfig?;
    /**
     *
     * Creates an instance of ShareServiceClient from connection string.
     *
     * @param connectionString - Account connection string or a SAS connection string of an Azure storage account.
     *                                  [ Note - Account connection string can only be used in NODE.JS runtime. ]
     *                                  Account connection string example -
     *                                  `DefaultEndpointsProtocol=https;AccountName=myaccount;AccountKey=accountKey;EndpointSuffix=core.windows.net`
     *                                  SAS connection string example -
     *                                  `BlobEndpoint=https://myaccount.blob.core.windows.net/;QueueEndpoint=https://myaccount.queue.core.windows.net/;FileEndpoint=https://myaccount.file.core.windows.net/;TableEndpoint=https://myaccount.table.core.windows.net/;SharedAccessSignature=sasString`
     * @param options - Options to configure the HTTP pipeline.
     * @returns A new ShareServiceClient from the given connection string.
     */
    static fromConnectionString(connectionString: string, options?: ShareClientOptions): ShareServiceClient;
    /**
     * Creates an instance of ShareServiceClient.
     *
     * @param url - A URL string pointing to Azure Storage file service, such as
     *                     "https://myaccount.file.core.windows.net". You can Append a SAS
     *                     if using AnonymousCredential, such as "https://myaccount.file.core.windows.net?sasString".
     * @param credential - Such as AnonymousCredential, StorageSharedKeyCredential, or TokenCredential,
     *                                  If not specified, AnonymousCredential is used.
     * @param options - Optional. Options to configure the HTTP pipeline.
     */
    constructor(url: string, credential?: Credential | TokenCredential, options?: ShareClientOptions);
    /**
     * Creates an instance of ShareServiceClient.
     *
     * @param url - A URL string pointing to Azure Storage file service, such as
     *                     "https://myaccount.file.core.windows.net". You can Append a SAS
     *                     if using AnonymousCredential, such as "https://myaccount.file.core.windows.net?sasString".
     * @param pipeline - Call newPipeline() to create a default
     *                            pipeline, or provide a customized pipeline.
     * @param options - Optional. Options to configure the HTTP pipeline.
     */
    constructor(url: string, pipeline: Pipeline, options?: ShareClientConfig);
    /**
     * Creates a ShareClient object.
     *
     * @param shareName - Name of a share.
     * @returns The ShareClient object for the given share name.
     *
     * Example usage:
     *
     * ```ts snippet:ReadmeSampleCreateShareClient
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * const shareName = "<share name>";
     * const shareClient = serviceClient.getShareClient(shareName);
     * await shareClient.create();
     * ```
     */
    getShareClient(shareName: string): ShareClient;
    /**
     * Creates a Share.
     *
     * @param shareName -
     * @param options -
     * @returns Share creation response and the corresponding share client.
     */
    createShare(shareName: string, options?: ShareCreateOptions): Promise<{
        shareCreateResponse: ShareCreateResponse;
        shareClient: ShareClient;
    }>;
    /**
     * Deletes a Share.
     *
     * @param shareName -
     * @param options -
     * @returns Share deletion response and the corresponding share client.
     */
    deleteShare(shareName: string, options?: ShareDeleteMethodOptions): Promise<ShareDeleteResponse>;
    /**
     * Gets the properties of a storage account’s file service, including properties
     * for Storage Analytics and CORS (Cross-Origin Resource Sharing) rules.
     * @see https://learn.microsoft.com/rest/api/storageservices/get-file-service-properties
     *
     * @param options - Options to Get Properties operation.
     * @returns Response data for the Get Properties operation.
     */
    getProperties(options?: ServiceGetPropertiesOptions): Promise<ServiceGetPropertiesResponse>;
    /**
     * Sets properties for a storage account’s file service endpoint, including properties
     * for Storage Analytics, CORS (Cross-Origin Resource Sharing) rules and soft delete settings.
     * @see https://learn.microsoft.com/rest/api/storageservices/set-file-service-properties
     *
     * @param properties -
     * @param options - Options to Set Properties operation.
     * @returns Response data for the Set Properties operation.
     */
    setProperties(properties: FileServiceProperties, options?: ServiceSetPropertiesOptions): Promise<ServiceSetPropertiesResponse>;
    /**
     * Returns an AsyncIterableIterator for {@link ServiceListSharesSegmentResponse} objects
     *
     * @param marker - A string value that identifies the portion of
     *                          the list of shares to be returned with the next listing operation. The
     *                          operation returns the ContinuationToken value within the response body if the
     *                          listing operation did not return all shares remaining to be listed
     *                          with the current page. The ContinuationToken value can be used as the value for
     *                          the marker parameter in a subsequent call to request the next page of list
     *                          items. The marker value is opaque to the client.
     * @param options - Options to list shares operation.
     */
    private listSegments;
    /**
     * Returns an AsyncIterableIterator for share items
     *
     * @param options - Options to list shares operation.
     */
    private listItems;
    /**
     * Returns an async iterable iterator to list all the shares
     * under the specified account.
     *
     * .byPage() returns an async iterable iterator to list the shares in pages.
     *
     * Example using `for await` syntax:
     *
     * ```ts snippet:ReadmeSampleListShares
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * let i = 1;
     * for await (const share of serviceClient.listShares()) {
     *   console.log(`Share${i++}: ${share.name}`);
     * }
     * ```
     *
     * Example using `iter.next()`:
     *
     * ```ts snippet:ReadmeSampleListShares_Iterator
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * const shareIter = serviceClient.listShares();
     * let i = 1;
     * let { value, done } = await shareIter.next();
     * while (!done) {
     *   console.log(`Share ${i++}: ${value.name}`);
     *   ({ value, done } = await shareIter.next());
     * }
     * ```
     *
     * Example using `byPage()`:
     *
     * ```ts snippet:ReadmeSampleListShares_ByPage
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * let i = 1;
     * for await (const response of serviceClient.listShares().byPage({ maxPageSize: 20 })) {
     *   console.log(`Page ${i++}:`);
     *   for (const share of response.shareItems || []) {
     *     console.log(`\tShare: ${share.name}`);
     *   }
     * }
     * ```
     *
     * Example using paging with a marker:
     *
     * ```ts snippet:ReadmeSampleListShares_Continuation
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * let iterator = serviceClient.listShares().byPage({ maxPageSize: 2 });
     * let response = (await iterator.next()).value;
     *
     * for await (const share of response.shareItems || []) {
     *   console.log(`\tShare: ${share.name}`);
     * }
     *
     * // Gets next marker
     * let marker = response.continuationToken;
     *
     * // Passing next marker as continuationToken
     * iterator = serviceClient.listShares().byPage({ continuationToken: marker, maxPageSize: 10 });
     * response = (await iterator.next()).value;
     *
     * for await (const share of response.shareItems || []) {
     *   console.log(`\tShare: ${share.name}`);
     * }
     * ```
     *
     * @param options - Options to list shares operation.
     *
     * An asyncIterableIterator that supports paging.
     */
    listShares(options?: ServiceListSharesOptions): PagedAsyncIterableIterator<ShareItem, ServiceListSharesSegmentResponse>;
    /**
     * Gets the properties of a storage account's File service, including properties for Storage
     * Analytics metrics and CORS (Cross-Origin Resource Sharing) rules.
     *
     * @param marker - A string value that identifies the portion of
     *                          the list to be returned with the next list operation. The operation
     *                          returns a marker value within the response body if the list returned was
     *                          not complete. The marker value may then be used in a subsequent call to
     *                          request the next set of list items. The marker value is opaque to the
     *                          client.
     * @param options - Options to List Shares Segment operation.
     * @returns Response data for the List Shares Segment operation.
     */
    private listSharesSegment;
    /**
     * Restores a previously deleted share.
     * This API is only functional if Share Soft Delete is enabled
     * for the storage account associated with the share.
     *
     * @param deletedShareName - The name of the previously deleted share.
     * @param deletedShareVersion - The version of the previously deleted share.
     * @param options - Options to Share undelete operation.
     * @returns Restored share.
     */
    undeleteShare(deletedShareName: string, deletedShareVersion: string, options?: ServiceUndeleteShareOptions): Promise<ShareClient>;
    /**
     * Only available for ShareServiceClient constructed with a shared key credential.
     *
     * Generates an account Shared Access Signature (SAS) URI based on the client properties
     * and parameters passed in. The SAS is signed by the shared key credential of the client.
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/create-account-sas
     *
     * @param expiresOn - Optional. The time at which the shared access signature becomes invalid. Default to an hour later if not specified.
     * @param permissions - Specifies the list of permissions to be associated with the SAS.
     * @param resourceTypes - Specifies the resource types associated with the shared access signature.
     * @param options - Optional parameters.
     * @returns An account SAS URI consisting of the URI to the resource represented by this client, followed by the generated SAS token.
     */
    generateAccountSasUrl(expiresOn?: Date, permissions?: AccountSASPermissions, resourceTypes?: string, options?: ServiceGenerateAccountSasUrlOptions): string;
    /**
     * Only available for ShareServiceClient constructed with a shared key credential.
     *
     * Generates string to sign for an account Shared Access Signature (SAS) URI based on the client properties
     * and parameters passed in. The SAS is signed by the shared key credential of the client.
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/create-account-sas
     *
     * @param expiresOn - Optional. The time at which the shared access signature becomes invalid. Default to an hour later if not specified.
     * @param permissions - Specifies the list of permissions to be associated with the SAS.
     * @param resourceTypes - Specifies the resource types associated with the shared access signature.
     * @param options - Optional parameters.
     * @returns An account SAS URI consisting of the URI to the resource represented by this client, followed by the generated SAS token.
     */
    generateSasStringToSign(expiresOn?: Date, permissions?: AccountSASPermissions, resourceTypes?: string, options?: ServiceGenerateAccountSasUrlOptions): string;
}
//# sourceMappingURL=ShareServiceClient.d.ts.map