import * as coreClient from "@azure/core-client";
import * as coreRestPipeline from "@azure/core-rest-pipeline";
import * as coreHttpCompat from "@azure/core-http-compat";
/** Storage service properties. */
export interface FileServiceProperties {
    /** A summary of request statistics grouped by API in hourly aggregates for files. */
    hourMetrics?: Metrics;
    /** A summary of request statistics grouped by API in minute aggregates for files. */
    minuteMetrics?: Metrics;
    /** The set of CORS rules. */
    cors?: CorsRule[];
    /** Protocol settings */
    protocol?: ShareProtocolSettings;
}
/** Storage Analytics metrics for file service. */
export interface Metrics {
    /** The version of Storage Analytics to configure. */
    version: string;
    /** Indicates whether metrics are enabled for the File service. */
    enabled: boolean;
    /** Indicates whether metrics should generate summary statistics for called API operations. */
    includeAPIs?: boolean;
    /** The retention policy. */
    retentionPolicy?: RetentionPolicy;
}
/** The retention policy. */
export interface RetentionPolicy {
    /** Indicates whether a retention policy is enabled for the File service. If false, metrics data is retained, and the user is responsible for deleting it. */
    enabled: boolean;
    /** Indicates the number of days that metrics data should be retained. All data older than this value will be deleted. Metrics data is deleted on a best-effort basis after the retention period expires. */
    days?: number;
}
/** CORS is an HTTP feature that enables a web application running under one domain to access resources in another domain. Web browsers implement a security restriction known as same-origin policy that prevents a web page from calling APIs in a different domain; CORS provides a secure way to allow one domain (the origin domain) to call APIs in another domain. */
export interface CorsRule {
    /** The origin domains that are permitted to make a request against the storage service via CORS. The origin domain is the domain from which the request originates. Note that the origin must be an exact case-sensitive match with the origin that the user age sends to the service. You can also use the wildcard character '*' to allow all origin domains to make requests via CORS. */
    allowedOrigins: string;
    /** The methods (HTTP request verbs) that the origin domain may use for a CORS request. (comma separated) */
    allowedMethods: string;
    /** The request headers that the origin domain may specify on the CORS request. */
    allowedHeaders: string;
    /** The response headers that may be sent in the response to the CORS request and exposed by the browser to the request issuer. */
    exposedHeaders: string;
    /** The maximum amount time that a browser should cache the preflight OPTIONS request. */
    maxAgeInSeconds: number;
}
/** Protocol settings */
export interface ShareProtocolSettings {
    /** Settings for SMB protocol. */
    smb?: ShareSmbSettings;
}
/** Settings for SMB protocol. */
export interface ShareSmbSettings {
    /** Settings for SMB Multichannel. */
    multichannel?: SmbMultichannel;
}
/** Settings for SMB multichannel */
export interface SmbMultichannel {
    /** If SMB multichannel is enabled. */
    enabled?: boolean;
}
export interface StorageError {
    message?: string;
    copySourceStatusCode?: number;
    copySourceErrorCode?: string;
    copySourceErrorMessage?: string;
    authenticationErrorDetail?: string;
    code?: string;
}
/** An enumeration of shares. */
export interface ListSharesResponse {
    serviceEndpoint: string;
    prefix?: string;
    marker?: string;
    maxResults?: number;
    shareItems?: ShareItemInternal[];
    continuationToken: string;
}
/** A listed Azure Storage share item. */
export interface ShareItemInternal {
    name: string;
    snapshot?: string;
    deleted?: boolean;
    version?: string;
    /** Properties of a share. */
    properties: SharePropertiesInternal;
    /** Dictionary of <string> */
    metadata?: {
        [propertyName: string]: string;
    };
}
/** Properties of a share. */
export interface SharePropertiesInternal {
    lastModified: Date;
    etag: string;
    quota: number;
    provisionedIops?: number;
    provisionedIngressMBps?: number;
    provisionedEgressMBps?: number;
    provisionedBandwidthMiBps?: number;
    nextAllowedQuotaDowngradeTime?: Date;
    deletedTime?: Date;
    remainingRetentionDays?: number;
    accessTier?: string;
    accessTierChangeTime?: Date;
    accessTierTransitionState?: string;
    /** The current lease status of the share. */
    leaseStatus?: LeaseStatusType;
    /** Lease state of the share. */
    leaseState?: LeaseStateType;
    /** When a share is leased, specifies whether the lease is of infinite or fixed duration. */
    leaseDuration?: LeaseDurationType;
    enabledProtocols?: string;
    rootSquash?: ShareRootSquash;
    enableSnapshotVirtualDirectoryAccess?: boolean;
    paidBurstingEnabled?: boolean;
    paidBurstingMaxIops?: number;
    paidBurstingMaxBandwidthMibps?: number;
    includedBurstIops?: number;
    maxBurstCreditsForIops?: number;
    nextAllowedProvisionedIopsDowngradeTime?: Date;
    nextAllowedProvisionedBandwidthDowngradeTime?: Date;
}
/** A permission (a security descriptor) at the share level. */
export interface SharePermission {
    /** The permission in the Security Descriptor Definition Language (SDDL). */
    permission: string;
    format?: FilePermissionFormat;
}
/** Signed identifier. */
export interface SignedIdentifier {
    /** A unique id. */
    id: string;
    /** The access policy. */
    accessPolicy?: AccessPolicy;
}
/** An Access policy. */
export interface AccessPolicy {
    /** The date-time the policy is active. */
    startsOn?: string;
    /** The date-time the policy expires. */
    expiresOn?: string;
    /** The permissions for the ACL policy. */
    permissions?: string;
}
/** Stats for the share. */
export interface ShareStats {
    /** The approximate size of the data stored in bytes. Note that this value may not include all recently created or recently resized files. */
    shareUsageBytes: number;
}
/** An enumeration of directories and files. */
export interface ListFilesAndDirectoriesSegmentResponse {
    serviceEndpoint: string;
    shareName: string;
    shareSnapshot?: string;
    encoded?: boolean;
    directoryPath: string;
    prefix: StringEncoded;
    marker?: string;
    maxResults?: number;
    /** Abstract for entries that can be listed from Directory. */
    segment: FilesAndDirectoriesListSegment;
    continuationToken: string;
    directoryId?: string;
}
export interface StringEncoded {
    encoded?: boolean;
    content?: string;
}
/** Abstract for entries that can be listed from Directory. */
export interface FilesAndDirectoriesListSegment {
    directoryItems: DirectoryItem[];
    fileItems: FileItem[];
}
/** A listed directory item. */
export interface DirectoryItem {
    name: StringEncoded;
    fileId?: string;
    /** File properties. */
    properties?: FileProperty;
    attributes?: string;
    permissionKey?: string;
}
/** File properties. */
export interface FileProperty {
    /** Content length of the file. This value may not be up-to-date since an SMB client may have modified the file locally. The value of Content-Length may not reflect that fact until the handle is closed or the op-lock is broken. To retrieve current property values, call Get File Properties. */
    contentLength: number;
    creationTime?: Date;
    lastAccessTime?: Date;
    lastWriteTime?: Date;
    changeTime?: Date;
    lastModified?: Date;
    etag?: string;
}
/** A listed file item. */
export interface FileItem {
    name: StringEncoded;
    fileId?: string;
    /** File properties. */
    properties: FileProperty;
    attributes?: string;
    permissionKey?: string;
}
/** An enumeration of handles. */
export interface ListHandlesResponse {
    handleList?: HandleItem[];
    continuationToken: string;
}
/** A listed Azure Storage handle item. */
export interface HandleItem {
    /** XSMB service handle ID */
    handleId: string;
    path: StringEncoded;
    /** FileId uniquely identifies the file or directory. */
    fileId: string;
    /** ParentId uniquely identifies the parent directory of the object. */
    parentId?: string;
    /** SMB session ID in context of which the file handle was opened */
    sessionId: string;
    /** Client IP that opened the handle */
    clientIp: string;
    /** Name of the client machine where the share is being mounted */
    clientName: string;
    /** Time when the session that previously opened the handle has last been reconnected. (UTC) */
    openTime: Date;
    /** Time handle was last connected to (UTC) */
    lastReconnectTime?: Date;
    accessRightList?: AccessRight[];
}
/** The list of file ranges */
export interface ShareFileRangeList {
    ranges?: FileRange[];
    clearRanges?: ClearRange[];
}
/** An Azure Storage file range. */
export interface FileRange {
    /** Start of the range. */
    start: number;
    /** End of the range. */
    end: number;
}
export interface ClearRange {
    start: number;
    end: number;
}
/** Defines headers for Service_setProperties operation. */
export interface ServiceSetPropertiesHeaders {
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for Service_setProperties operation. */
export interface ServiceSetPropertiesExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for Service_getProperties operation. */
export interface ServiceGetPropertiesHeaders {
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for Service_getProperties operation. */
export interface ServiceGetPropertiesExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for Service_listSharesSegment operation. */
export interface ServiceListSharesSegmentHeaders {
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for Service_listSharesSegment operation. */
export interface ServiceListSharesSegmentExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for Share_create operation. */
export interface ShareCreateHeaders {
    /** The ETag contains a value which represents the version of the share, in quotes. */
    etag?: string;
    /** Returns the date and time the share was last modified. Any operation that modifies the share or its properties or metadata updates the last modified time. Operations on files do not affect the last modified time of the share. */
    lastModified?: Date;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** Returns the current share quota in GB. */
    quota?: number;
    /** The provisioned IOPS of the share. */
    shareProvisionedIops?: number;
    /** The provisioned throughput of the share. */
    shareProvisionedBandwidthMibps?: number;
    /** Returns the calculated burst IOPS of the share. */
    shareIncludedBurstIops?: number;
    /** Returned the calculated maximum burst credits. */
    maxBurstCreditsForIops?: number;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for Share_create operation. */
export interface ShareCreateExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for Share_getProperties operation. */
export interface ShareGetPropertiesHeaders {
    /** A set of name-value pairs that contain the user-defined metadata of the share. */
    metadata?: {
        [propertyName: string]: string;
    };
    /** The ETag contains a value that you can use to perform operations conditionally, in quotes. */
    etag?: string;
    /** Returns the date and time the share was last modified. Any operation that modifies the share or its properties updates the last modified time. Operations on files do not affect the last modified time of the share. */
    lastModified?: Date;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** Returns the current share quota in GB. */
    quota?: number;
    /** Returns the current share provisioned ipos. */
    provisionedIops?: number;
    /** Returns the current share provisioned ingress in megabytes per second. */
    provisionedIngressMBps?: number;
    /** Returns the current share provisioned egress in megabytes per second. */
    provisionedEgressMBps?: number;
    /** Returns the current share next allowed quota downgrade time. */
    nextAllowedQuotaDowngradeTime?: Date;
    /** Returns the current share provisioned bandwidth in mebibytes per second. */
    provisionedBandwidthMibps?: number;
    /** When a share is leased, specifies whether the lease is of infinite or fixed duration. */
    leaseDuration?: LeaseDurationType;
    /** Lease state of the share. */
    leaseState?: LeaseStateType;
    /** The current lease status of the share. */
    leaseStatus?: LeaseStatusType;
    /** Returns the access tier set on the share. */
    accessTier?: string;
    /** Returns the last modified time (in UTC) of the access tier of the share. */
    accessTierChangeTime?: Date;
    /** Returns the transition state between access tiers, when present. */
    accessTierTransitionState?: string;
    /** The protocols that have been enabled on the share. */
    enabledProtocols?: string;
    /** Valid for NFS shares only. */
    rootSquash?: ShareRootSquash;
    /** Version 2023-08-03 and newer. Specifies whether the snapshot virtual directory should be accessible at the root of share mount point when NFS is enabled. This header is only returned for shares, not for snapshots. */
    enableSnapshotVirtualDirectoryAccess?: boolean;
    /** Optional. Boolean. Default if not specified is false. This property enables paid bursting. */
    paidBurstingEnabled?: boolean;
    /** Optional. Integer. Default if not specified is the maximum IOPS the file share can support. Current maximum for a file share is 102,400 IOPS. */
    paidBurstingMaxIops?: number;
    /** Optional. Integer. Default if not specified is the maximum throughput the file share can support. Current maximum for a file share is 10,340 MiB/sec. */
    paidBurstingMaxBandwidthMibps?: number;
    /** Return the calculated burst IOPS of the share. */
    includedBurstIops?: number;
    /** Returned the calculated maximum burst credits. This is not the current burst credit level, but the maximum burst credits the share can have. */
    maxBurstCreditsForIops?: number;
    /** Returns the current share next allowed provisioned iops downgrade time. */
    nextAllowedProvisionedIopsDowngradeTime?: Date;
    /** Returns the current share next allowed provisioned bandwidth downgrade time. */
    nextAllowedProvisionedBandwidthDowngradeTime?: Date;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for Share_getProperties operation. */
export interface ShareGetPropertiesExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for Share_delete operation. */
export interface ShareDeleteHeaders {
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** Returned only for provisioned v2 file shares. Returns an approximate used storage size of the share, in bytes. */
    usageBytes?: number;
    /** Returned only for provisioned v2 file shares. Returns an approximate used snapshot storage size of the share, in bytes. */
    snapshotUsageBytes?: number;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for Share_delete operation. */
export interface ShareDeleteExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for Share_acquireLease operation. */
export interface ShareAcquireLeaseHeaders {
    /** The ETag contains a value that you can use to perform operations conditionally, in quotes. */
    etag?: string;
    /** Returns the date and time the share was last modified. Any operation that modifies the share or its properties updates the last modified time. Operations on files do not affect the last modified time of the share. */
    lastModified?: Date;
    /** Uniquely identifies a share's lease */
    leaseId?: string;
    /** If a client request id header is sent in the request, this header will be present in the response with the same value. */
    clientRequestId?: string;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** UTC date/time value generated by the service that indicates the time at which the response was initiated */
    date?: Date;
}
/** Defines headers for Share_acquireLease operation. */
export interface ShareAcquireLeaseExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for Share_releaseLease operation. */
export interface ShareReleaseLeaseHeaders {
    /** The ETag contains a value that you can use to perform operations conditionally, in quotes. */
    etag?: string;
    /** Returns the date and time the share was last modified. Any operation that modifies the share or its properties updates the last modified time. Operations on files do not affect the last modified time of the share. */
    lastModified?: Date;
    /** If a client request id header is sent in the request, this header will be present in the response with the same value. */
    clientRequestId?: string;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** UTC date/time value generated by the service that indicates the time at which the response was initiated */
    date?: Date;
}
/** Defines headers for Share_releaseLease operation. */
export interface ShareReleaseLeaseExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for Share_changeLease operation. */
export interface ShareChangeLeaseHeaders {
    /** The ETag contains a value that you can use to perform operations conditionally, in quotes. */
    etag?: string;
    /** Returns the date and time the share was last modified. Any operation that modifies the share or its properties updates the last modified time. Operations on files do not affect the last modified time of the share. */
    lastModified?: Date;
    /** Uniquely identifies a share's lease */
    leaseId?: string;
    /** If a client request id header is sent in the request, this header will be present in the response with the same value. */
    clientRequestId?: string;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** UTC date/time value generated by the service that indicates the time at which the response was initiated */
    date?: Date;
}
/** Defines headers for Share_changeLease operation. */
export interface ShareChangeLeaseExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for Share_renewLease operation. */
export interface ShareRenewLeaseHeaders {
    /** The ETag contains a value that you can use to perform operations conditionally, in quotes. */
    etag?: string;
    /** Returns the date and time the share was last modified. Any operation that modifies the share or its properties updates the last modified time. Operations on files do not affect the last modified time of the share. */
    lastModified?: Date;
    /** Uniquely identifies a share's lease */
    leaseId?: string;
    /** If a client request id header is sent in the request, this header will be present in the response with the same value. */
    clientRequestId?: string;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** Returns the current share next allowed quota downgrade time. */
    date?: Date;
}
/** Defines headers for Share_renewLease operation. */
export interface ShareRenewLeaseExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for Share_breakLease operation. */
export interface ShareBreakLeaseHeaders {
    /** The ETag contains a value that you can use to perform operations conditionally, in quotes. */
    etag?: string;
    /** Returns the date and time the share was last modified. Any operation that modifies the share or its properties updates the last modified time. Operations on files do not affect the last modified time of the share. */
    lastModified?: Date;
    /** Approximate time remaining in the lease period, in seconds. */
    leaseTimeInSeconds?: number;
    /** Uniquely identifies a share's lease */
    leaseId?: string;
    /** If a client request id header is sent in the request, this header will be present in the response with the same value. */
    clientRequestId?: string;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** UTC date/time value generated by the service that indicates the time at which the response was initiated */
    date?: Date;
}
/** Defines headers for Share_breakLease operation. */
export interface ShareBreakLeaseExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for Share_createSnapshot operation. */
export interface ShareCreateSnapshotHeaders {
    /** This header is a DateTime value that uniquely identifies the share snapshot. The value of this header may be used in subsequent requests to access the share snapshot. This value is opaque. */
    snapshot?: string;
    /** The ETag contains a value which represents the version of the share snapshot, in quotes. A share snapshot cannot be modified, so the ETag of a given share snapshot never changes. However, if new metadata was supplied with the Snapshot Share request then the ETag of the share snapshot differs from that of the base share. If no metadata was specified with the request, the ETag of the share snapshot is identical to that of the base share at the time the share snapshot was taken. */
    etag?: string;
    /** Returns the date and time the share was last modified. A share snapshot cannot be modified, so the last modified time of a given share snapshot never changes. However, if new metadata was supplied with the Snapshot Share request then the last modified time of the share snapshot differs from that of the base share. If no metadata was specified with the request, the last modified time of the share snapshot is identical to that of the base share at the time the share snapshot was taken. */
    lastModified?: Date;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for Share_createSnapshot operation. */
export interface ShareCreateSnapshotExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for Share_createPermission operation. */
export interface ShareCreatePermissionHeaders {
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** Key of the permission set for the directory/file. */
    filePermissionKey?: string;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for Share_createPermission operation. */
export interface ShareCreatePermissionExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for Share_getPermission operation. */
export interface ShareGetPermissionHeaders {
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for Share_getPermission operation. */
export interface ShareGetPermissionExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for Share_setProperties operation. */
export interface ShareSetPropertiesHeaders {
    /** The ETag contains a value that you can use to perform operations conditionally, in quotes. */
    etag?: string;
    /** Returns the date and time the share was last modified. Any operation that modifies the share or its properties updates the last modified time. Operations on files do not affect the last modified time of the share. */
    lastModified?: Date;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** Returns the current share quota in GB. */
    quota?: number;
    /** Returns the current share provisioned ipos. */
    provisionedIops?: number;
    /** Returns the current share provisioned bandwidth in mebibytes per second. */
    provisionedBandwidthMibps?: number;
    /** Return the calculated burst IOPS of the share. */
    includedBurstIops?: number;
    /** Returned the calculated maximum burst credits. This is not the current burst credit level, but the maximum burst credits the share can have. */
    maxBurstCreditsForIops?: number;
    /** Returns the current share next allowed quota downgrade time. */
    nextAllowedQuotaDowngradeTime?: Date;
    /** Returns the current share next allowed provisioned iops downgrade time. */
    nextAllowedProvisionedIopsDowngradeTime?: Date;
    /** Returns the current share next allowed provisioned bandwidth downgrade time. */
    nextAllowedProvisionedBandwidthDowngradeTime?: Date;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for Share_setProperties operation. */
export interface ShareSetPropertiesExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for Share_setMetadata operation. */
export interface ShareSetMetadataHeaders {
    /** The ETag contains a value that you can use to perform operations conditionally, in quotes. */
    etag?: string;
    /** Returns the date and time the share was last modified. Any operation that modifies the share or its properties updates the last modified time. Operations on files do not affect the last modified time of the share. */
    lastModified?: Date;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for Share_setMetadata operation. */
export interface ShareSetMetadataExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for Share_getAccessPolicy operation. */
export interface ShareGetAccessPolicyHeaders {
    /** The ETag contains a value that you can use to perform operations conditionally, in quotes. */
    etag?: string;
    /** Returns the date and time the share was last modified. Any operation that modifies the share or its properties updates the last modified time. Operations on files do not affect the last modified time of the share. */
    lastModified?: Date;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for Share_getAccessPolicy operation. */
export interface ShareGetAccessPolicyExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for Share_setAccessPolicy operation. */
export interface ShareSetAccessPolicyHeaders {
    /** The ETag contains a value that you can use to perform operations conditionally, in quotes. */
    etag?: string;
    /** Returns the date and time the share was last modified. Any operation that modifies the share or its properties updates the last modified time. Operations on files do not affect the last modified time of the share. */
    lastModified?: Date;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for Share_setAccessPolicy operation. */
export interface ShareSetAccessPolicyExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for Share_getStatistics operation. */
export interface ShareGetStatisticsHeaders {
    /** The ETag contains a value that you can use to perform operations conditionally, in quotes. */
    etag?: string;
    /** Returns the date and time the share was last modified. Any operation that modifies the share or its properties updates the last modified time. Operations on files do not affect the last modified time of the share. */
    lastModified?: Date;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for Share_getStatistics operation. */
export interface ShareGetStatisticsExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for Share_restore operation. */
export interface ShareRestoreHeaders {
    /** The ETag contains a value that you can use to perform operations conditionally, in quotes. */
    etag?: string;
    /** Returns the date and time the share was last modified. Any operation that modifies the share or its properties updates the last modified time. Operations on files do not affect the last modified time of the share. */
    lastModified?: Date;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** If a client request id header is sent in the request, this header will be present in the response with the same value. */
    clientRequestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** Returns the current share quota in GB. */
    quota?: number;
    /** Returns the current share provisioned ipos. */
    provisionedIops?: number;
    /** Returns the current share provisioned bandwidth in mebibytes per second. */
    provisionedBandwidthMibps?: number;
    /** Return the calculated burst IOPS of the share. */
    includedBurstIops?: number;
    /** Returned the calculated maximum burst credits. This is not the current burst credit level, but the maximum burst credits the share can have. */
    maxBurstCreditsForIops?: number;
}
/** Defines headers for Share_restore operation. */
export interface ShareRestoreExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for Directory_create operation. */
export interface DirectoryCreateHeaders {
    /** The ETag contains a value which represents the version of the directory, in quotes. */
    etag?: string;
    /** Returns the date and time the share was last modified. Any operation that modifies the directory or its properties updates the last modified time. Operations on files do not affect the last modified time of the directory. */
    lastModified?: Date;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** The value of this header is set to true if the contents of the request are successfully encrypted using the specified algorithm, and false otherwise. */
    isServerEncrypted?: boolean;
    /** Key of the permission set for the directory. */
    filePermissionKey?: string;
    /** Attributes set for the directory. */
    fileAttributes?: string;
    /** Creation time for the directory. */
    fileCreatedOn?: Date;
    /** Last write time for the directory. */
    fileLastWriteOn?: Date;
    /** Change time for the directory. */
    fileChangeOn?: Date;
    /** The fileId of the directory. */
    fileId?: string;
    /** The parent fileId of the directory. */
    fileParentId?: string;
    /** NFS only. The mode of the file or directory. */
    fileMode?: string;
    /** NFS only. The owner of the file or directory. */
    owner?: string;
    /** NFS only. The owning group of the file or directory. */
    group?: string;
    /** NFS only. Type of the file or directory. */
    nfsFileType?: NfsFileType;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for Directory_create operation. */
export interface DirectoryCreateExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for Directory_getProperties operation. */
export interface DirectoryGetPropertiesHeaders {
    /** A set of name-value pairs that contain metadata for the directory. */
    metadata?: {
        [propertyName: string]: string;
    };
    /** The ETag contains a value that you can use to perform operations conditionally, in quotes. */
    etag?: string;
    /** Returns the date and time the Directory was last modified. Operations on files within the directory do not affect the last modified time of the directory. */
    lastModified?: Date;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** The value of this header is set to true if the directory metadata is completely encrypted using the specified algorithm. Otherwise, the value is set to false. */
    isServerEncrypted?: boolean;
    /** Attributes set for the directory. */
    fileAttributes?: string;
    /** Creation time for the directory. */
    fileCreatedOn?: Date;
    /** Last write time for the directory. */
    fileLastWriteOn?: Date;
    /** Change time for the directory. */
    fileChangeOn?: Date;
    /** Key of the permission set for the directory. */
    filePermissionKey?: string;
    /** The fileId of the directory. */
    fileId?: string;
    /** The parent fileId of the directory. */
    fileParentId?: string;
    /** NFS only. The mode of the file or directory. */
    fileMode?: string;
    /** NFS only. The owner of the file or directory. */
    owner?: string;
    /** NFS only. The owning group of the file or directory. */
    group?: string;
    /** NFS only. Type of the file or directory. */
    nfsFileType?: NfsFileType;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for Directory_getProperties operation. */
export interface DirectoryGetPropertiesExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for Directory_delete operation. */
export interface DirectoryDeleteHeaders {
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for Directory_delete operation. */
export interface DirectoryDeleteExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for Directory_setProperties operation. */
export interface DirectorySetPropertiesHeaders {
    /** The ETag contains a value which represents the version of the file, in quotes. */
    etag?: string;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Returns the date and time the directory was last modified. Any operation that modifies the directory or its properties updates the last modified time. Operations on files do not affect the last modified time of the directory. */
    lastModified?: Date;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** The value of this header is set to true if the contents of the request are successfully encrypted using the specified algorithm, and false otherwise. */
    isServerEncrypted?: boolean;
    /** Key of the permission set for the directory. */
    filePermissionKey?: string;
    /** Attributes set for the directory. */
    fileAttributes?: string;
    /** Creation time for the directory. */
    fileCreatedOn?: Date;
    /** Last write time for the directory. */
    fileLastWriteOn?: Date;
    /** Change time for the directory. */
    fileChangeOn?: Date;
    /** The fileId of the directory. */
    fileId?: string;
    /** The parent fileId of the directory. */
    fileParentId?: string;
    /** NFS only. The mode of the file or directory. */
    fileMode?: string;
    /** NFS only. The owner of the file or directory. */
    owner?: string;
    /** NFS only. The owning group of the file or directory. */
    group?: string;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for Directory_setProperties operation. */
export interface DirectorySetPropertiesExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for Directory_setMetadata operation. */
export interface DirectorySetMetadataHeaders {
    /** The ETag contains a value which represents the version of the directory, in quotes. */
    etag?: string;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** The value of this header is set to true if the contents of the request are successfully encrypted using the specified algorithm, and false otherwise. */
    isServerEncrypted?: boolean;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for Directory_setMetadata operation. */
export interface DirectorySetMetadataExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for Directory_listFilesAndDirectoriesSegment operation. */
export interface DirectoryListFilesAndDirectoriesSegmentHeaders {
    /** Specifies the format in which the results are returned. Currently this value is 'application/xml'. */
    contentType?: string;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for Directory_listFilesAndDirectoriesSegment operation. */
export interface DirectoryListFilesAndDirectoriesSegmentExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for Directory_listHandles operation. */
export interface DirectoryListHandlesHeaders {
    /** Specifies the format in which the results are returned. Currently this value is 'application/xml'. */
    contentType?: string;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for Directory_listHandles operation. */
export interface DirectoryListHandlesExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for Directory_forceCloseHandles operation. */
export interface DirectoryForceCloseHandlesHeaders {
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** A string describing next handle to be closed. It is returned when more handles need to be closed to complete the request. */
    marker?: string;
    /** Contains count of number of handles closed. */
    numberOfHandlesClosed?: number;
    /** Contains count of number of handles that failed to close. */
    numberOfHandlesFailedToClose?: number;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for Directory_forceCloseHandles operation. */
export interface DirectoryForceCloseHandlesExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for Directory_rename operation. */
export interface DirectoryRenameHeaders {
    /** The ETag contains a value which represents the version of the file, in quotes. */
    etag?: string;
    /** Returns the date and time the share was last modified. Any operation that modifies the directory or its properties updates the last modified time. Operations on files do not affect the last modified time of the directory. */
    lastModified?: Date;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** The value of this header is set to true if the contents of the request are successfully encrypted using the specified algorithm, and false otherwise. */
    isServerEncrypted?: boolean;
    /** Key of the permission set for the file. */
    filePermissionKey?: string;
    /** Attributes set for the file. */
    fileAttributes?: string;
    /** Creation time for the file. */
    fileCreationTime?: Date;
    /** Last write time for the file. */
    fileLastWriteTime?: Date;
    /** Change time for the file. */
    fileChangeTime?: Date;
    /** The fileId of the file. */
    fileId?: string;
    /** The parent fileId of the directory. */
    fileParentId?: string;
}
/** Defines headers for Directory_rename operation. */
export interface DirectoryRenameExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for File_create operation. */
export interface FileCreateHeaders {
    /** The ETag contains a value which represents the version of the file, in quotes. */
    etag?: string;
    /** Returns the date and time the share was last modified. Any operation that modifies the directory or its properties updates the last modified time. Operations on files do not affect the last modified time of the directory. */
    lastModified?: Date;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** The value of this header is set to true if the contents of the request are successfully encrypted using the specified algorithm, and false otherwise. */
    isServerEncrypted?: boolean;
    /** Key of the permission set for the file. */
    filePermissionKey?: string;
    /** Attributes set for the file. */
    fileAttributes?: string;
    /** Creation time for the file. */
    fileCreatedOn?: Date;
    /** Last write time for the file. */
    fileLastWriteOn?: Date;
    /** Change time for the file. */
    fileChangeOn?: Date;
    /** The fileId of the file. */
    fileId?: string;
    /** The parent fileId of the file. */
    fileParentId?: string;
    /** NFS only. The mode of the file or directory. */
    fileMode?: string;
    /** NFS only. The owner of the file or directory. */
    owner?: string;
    /** NFS only. The owning group of the file or directory. */
    group?: string;
    /** NFS only. Type of the file or directory. */
    nfsFileType?: NfsFileType;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for File_create operation. */
export interface FileCreateExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for File_download operation. */
export interface FileDownloadHeaders {
    /** Returns the date and time the file was last modified. Any operation that modifies the file or its properties updates the last modified time. */
    lastModified?: Date;
    /** A set of name-value pairs associated with this file as user-defined metadata. */
    metadata?: {
        [propertyName: string]: string;
    };
    /** The number of bytes present in the response body. */
    contentLength?: number;
    /** The content type specified for the file. The default content type is 'application/octet-stream' */
    contentType?: string;
    /** Indicates the range of bytes returned if the client requested a subset of the file by setting the Range request header. */
    contentRange?: string;
    /** The ETag contains a value that you can use to perform operations conditionally, in quotes. */
    etag?: string;
    /** If the file has an MD5 hash and the request is to read the full file, this response header is returned so that the client can check for message content integrity. If the request is to read a specified range and the 'x-ms-range-get-content-md5' is set to true, then the request returns an MD5 hash for the range, as long as the range size is less than or equal to 4 MB. If neither of these sets of conditions is true, then no value is returned for the 'Content-MD5' header. */
    contentMD5?: Uint8Array;
    /** Returns the value that was specified for the Content-Encoding request header. */
    contentEncoding?: string;
    /** Returned if it was previously specified for the file. */
    cacheControl?: string;
    /** Returns the value that was specified for the 'x-ms-content-disposition' header and specifies how to process the response. */
    contentDisposition?: string;
    /** Returns the value that was specified for the Content-Language request header. */
    contentLanguage?: string;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** Indicates that the service supports requests for partial file content. */
    acceptRanges?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** Conclusion time of the last attempted Copy File operation where this file was the destination file. This value can specify the time of a completed, aborted, or failed copy attempt. */
    copyCompletedOn?: Date;
    /** Only appears when x-ms-copy-status is failed or pending. Describes cause of fatal or non-fatal copy operation failure. */
    copyStatusDescription?: string;
    /** String identifier for the last attempted Copy File operation where this file was the destination file. */
    copyId?: string;
    /** Contains the number of bytes copied and the total bytes in the source in the last attempted Copy File operation where this file was the destination file. Can show between 0 and Content-Length bytes copied. */
    copyProgress?: string;
    /** URL up to 2KB in length that specifies the source file used in the last attempted Copy File operation where this file was the destination file. */
    copySource?: string;
    /** State of the copy operation identified by 'x-ms-copy-id'. */
    copyStatus?: CopyStatusType;
    /** If the file has a MD5 hash, and if request contains range header (Range or x-ms-range), this response header is returned with the value of the whole file's MD5 value. This value may or may not be equal to the value returned in Content-MD5 header, with the latter calculated from the requested range. */
    fileContentMD5?: Uint8Array;
    /** The value of this header is set to true if the file data and application metadata are completely encrypted using the specified algorithm. Otherwise, the value is set to false (when the file is unencrypted, or if only parts of the file/application metadata are encrypted). */
    isServerEncrypted?: boolean;
    /** Attributes set for the file. */
    fileAttributes?: string;
    /** Creation time for the file. */
    fileCreatedOn?: Date;
    /** Last write time for the file. */
    fileLastWriteOn?: Date;
    /** Change time for the file. */
    fileChangeOn?: Date;
    /** Key of the permission set for the file. */
    filePermissionKey?: string;
    /** The fileId of the file. */
    fileId?: string;
    /** The parent fileId of the file. */
    fileParentId?: string;
    /** When a file is leased, specifies whether the lease is of infinite or fixed duration. */
    leaseDuration?: LeaseDurationType;
    /** Lease state of the file. */
    leaseState?: LeaseStateType;
    /** The current lease status of the file. */
    leaseStatus?: LeaseStatusType;
    /** NFS only. The mode of the file or directory. */
    fileMode?: string;
    /** NFS only. The owner of the file or directory. */
    owner?: string;
    /** NFS only. The owning group of the file or directory. */
    group?: string;
    /** NFS only. The link count of the file or directory. */
    linkCount?: number;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for File_download operation. */
export interface FileDownloadExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for File_getProperties operation. */
export interface FileGetPropertiesHeaders {
    /** Returns the date and time the file was last modified. The date format follows RFC 1123. Any operation that modifies the file or its properties updates the last modified time. */
    lastModified?: Date;
    /** A set of name-value pairs associated with this file as user-defined metadata. */
    metadata?: {
        [propertyName: string]: string;
    };
    /** Returns the type File. Reserved for future use. */
    fileType?: string;
    /** The size of the file in bytes. This header returns the value of the 'x-ms-content-length' header that is stored with the file. */
    contentLength?: number;
    /** The content type specified for the file. The default content type is 'application/octet-stream' */
    contentType?: string;
    /** The ETag contains a value that you can use to perform operations conditionally, in quotes. */
    etag?: string;
    /** If the Content-MD5 header has been set for the file, the Content-MD5 response header is returned so that the client can check for message content integrity. */
    contentMD5?: Uint8Array;
    /** If the Content-Encoding request header has previously been set for the file, the Content-Encoding value is returned in this header. */
    contentEncoding?: string;
    /** If the Cache-Control request header has previously been set for the file, the Cache-Control value is returned in this header. */
    cacheControl?: string;
    /** Returns the value that was specified for the 'x-ms-content-disposition' header and specifies how to process the response. */
    contentDisposition?: string;
    /** Returns the value that was specified for the Content-Language request header. */
    contentLanguage?: string;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** Conclusion time of the last attempted Copy File operation where this file was the destination file. This value can specify the time of a completed, aborted, or failed copy attempt. */
    copyCompletedOn?: Date;
    /** Only appears when x-ms-copy-status is failed or pending. Describes cause of fatal or non-fatal copy operation failure. */
    copyStatusDescription?: string;
    /** String identifier for the last attempted Copy File operation where this file was the destination file. */
    copyId?: string;
    /** Contains the number of bytes copied and the total bytes in the source in the last attempted Copy File operation where this file was the destination file. Can show between 0 and Content-Length bytes copied. */
    copyProgress?: string;
    /** URL up to 2KB in length that specifies the source file used in the last attempted Copy File operation where this file was the destination file. */
    copySource?: string;
    /** State of the copy operation identified by 'x-ms-copy-id'. */
    copyStatus?: CopyStatusType;
    /** The value of this header is set to true if the file data and application metadata are completely encrypted using the specified algorithm. Otherwise, the value is set to false (when the file is unencrypted, or if only parts of the file/application metadata are encrypted). */
    isServerEncrypted?: boolean;
    /** Attributes set for the file. */
    fileAttributes?: string;
    /** Creation time for the file. */
    fileCreatedOn?: Date;
    /** Last write time for the file. */
    fileLastWriteOn?: Date;
    /** Change time for the file. */
    fileChangeOn?: Date;
    /** Key of the permission set for the file. */
    filePermissionKey?: string;
    /** The fileId of the file. */
    fileId?: string;
    /** The parent fileId of the file. */
    fileParentId?: string;
    /** When a file is leased, specifies whether the lease is of infinite or fixed duration. */
    leaseDuration?: LeaseDurationType;
    /** Lease state of the file. */
    leaseState?: LeaseStateType;
    /** The current lease status of the file. */
    leaseStatus?: LeaseStatusType;
    /** NFS only. The mode of the file or directory. */
    fileMode?: string;
    /** NFS only. The owner of the file or directory. */
    owner?: string;
    /** NFS only. The owning group of the file or directory. */
    group?: string;
    /** NFS only. The link count of the file or directory. */
    linkCount?: number;
    /** NFS only. Type of the file or directory. */
    nfsFileType?: NfsFileType;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for File_getProperties operation. */
export interface FileGetPropertiesExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for File_delete operation. */
export interface FileDeleteHeaders {
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** NFS only. The link count of the file or directory. */
    linkCount?: number;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for File_delete operation. */
export interface FileDeleteExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for File_setHttpHeaders operation. */
export interface FileSetHttpHeadersHeaders {
    /** The ETag contains a value which represents the version of the file, in quotes. */
    etag?: string;
    /** Returns the date and time the directory was last modified. Any operation that modifies the directory or its properties updates the last modified time. Operations on files do not affect the last modified time of the directory. */
    lastModified?: Date;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** The value of this header is set to true if the contents of the request are successfully encrypted using the specified algorithm, and false otherwise. */
    isServerEncrypted?: boolean;
    /** Key of the permission set for the file. */
    filePermissionKey?: string;
    /** Attributes set for the file. */
    fileAttributes?: string;
    /** Creation time for the file. */
    fileCreatedOn?: Date;
    /** Last write time for the file. */
    fileLastWriteOn?: Date;
    /** Change time for the file. */
    fileChangeOn?: Date;
    /** The fileId of the directory. */
    fileId?: string;
    /** The parent fileId of the directory. */
    fileParentId?: string;
    /** NFS only. The mode of the file or directory. */
    fileMode?: string;
    /** NFS only. The owner of the file or directory. */
    owner?: string;
    /** NFS only. The owning group of the file or directory. */
    group?: string;
    /** NFS only. The link count of the file or directory. */
    linkCount?: number;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for File_setHttpHeaders operation. */
export interface FileSetHttpHeadersExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for File_setMetadata operation. */
export interface FileSetMetadataHeaders {
    /** The ETag contains a value which represents the version of the file, in quotes. */
    etag?: string;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** The value of this header is set to true if the contents of the request are successfully encrypted using the specified algorithm, and false otherwise. */
    isServerEncrypted?: boolean;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for File_setMetadata operation. */
export interface FileSetMetadataExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for File_acquireLease operation. */
export interface FileAcquireLeaseHeaders {
    /** The ETag contains a value that you can use to perform operations conditionally. If the request version is 2011-08-18 or newer, the ETag value will be in quotes. */
    etag?: string;
    /** Returns the date and time the file was last modified. Any operation that modifies the file, including an update of the file's metadata or properties, changes the last-modified time of the file. */
    lastModified?: Date;
    /** Uniquely identifies a file's lease */
    leaseId?: string;
    /** If a client request id header is sent in the request, this header will be present in the response with the same value. */
    clientRequestId?: string;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** UTC date/time value generated by the service that indicates the time at which the response was initiated */
    date?: Date;
}
/** Defines headers for File_acquireLease operation. */
export interface FileAcquireLeaseExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for File_releaseLease operation. */
export interface FileReleaseLeaseHeaders {
    /** The ETag contains a value that you can use to perform operations conditionally. If the request version is 2011-08-18 or newer, the ETag value will be in quotes. */
    etag?: string;
    /** Returns the date and time the file was last modified. Any operation that modifies the file, including an update of the file's metadata or properties, changes the last-modified time of the file. */
    lastModified?: Date;
    /** If a client request id header is sent in the request, this header will be present in the response with the same value. */
    clientRequestId?: string;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** UTC date/time value generated by the service that indicates the time at which the response was initiated */
    date?: Date;
}
/** Defines headers for File_releaseLease operation. */
export interface FileReleaseLeaseExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for File_changeLease operation. */
export interface FileChangeLeaseHeaders {
    /** The ETag contains a value that you can use to perform operations conditionally. If the request version is 2011-08-18 or newer, the ETag value will be in quotes. */
    etag?: string;
    /** Returns the date and time the file was last modified. Any operation that modifies the file, including an update of the file's metadata or properties, changes the last-modified time of the file. */
    lastModified?: Date;
    /** Uniquely identifies a file's lease */
    leaseId?: string;
    /** If a client request id header is sent in the request, this header will be present in the response with the same value. */
    clientRequestId?: string;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** UTC date/time value generated by the service that indicates the time at which the response was initiated */
    date?: Date;
}
/** Defines headers for File_changeLease operation. */
export interface FileChangeLeaseExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for File_breakLease operation. */
export interface FileBreakLeaseHeaders {
    /** The ETag contains a value that you can use to perform operations conditionally. If the request version is 2011-08-18 or newer, the ETag value will be in quotes. */
    etag?: string;
    /** Returns the date and time the file was last modified. Any operation that modifies the file, including an update of the file's metadata or properties, changes the last-modified time of the file. */
    lastModified?: Date;
    /** Uniquely identifies a file's lease */
    leaseId?: string;
    /** If a client request id header is sent in the request, this header will be present in the response with the same value. */
    clientRequestId?: string;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** UTC date/time value generated by the service that indicates the time at which the response was initiated */
    date?: Date;
}
/** Defines headers for File_breakLease operation. */
export interface FileBreakLeaseExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for File_uploadRange operation. */
export interface FileUploadRangeHeaders {
    /** The ETag contains a value which represents the version of the file, in quotes. */
    etag?: string;
    /** Returns the date and time the directory was last modified. Any operation that modifies the share or its properties or metadata updates the last modified time. Operations on files do not affect the last modified time of the share. */
    lastModified?: Date;
    /** This header is returned so that the client can check for message content integrity. The value of this header is computed by the File service; it is not necessarily the same value as may have been specified in the request headers. */
    contentMD5?: Uint8Array;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** The value of this header is set to true if the contents of the request are successfully encrypted using the specified algorithm, and false otherwise. */
    isServerEncrypted?: boolean;
    /** Last write time for the file. */
    fileLastWriteTime?: Date;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for File_uploadRange operation. */
export interface FileUploadRangeExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for File_uploadRangeFromURL operation. */
export interface FileUploadRangeFromURLHeaders {
    /** The ETag contains a value which represents the version of the file, in quotes. */
    etag?: string;
    /** Returns the date and time the directory was last modified. Any operation that modifies the share or its properties or metadata updates the last modified time. Operations on files do not affect the last modified time of the share. */
    lastModified?: Date;
    /** This header is returned so that the client can check for message content integrity. The value of this header is computed by the File service; it is not necessarily the same value as may have been specified in the request headers. */
    xMsContentCrc64?: Uint8Array;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** The value of this header is set to true if the contents of the request are successfully encrypted using the specified algorithm, and false otherwise. */
    isServerEncrypted?: boolean;
    /** Last write time for the file. */
    fileLastWriteTime?: Date;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for File_uploadRangeFromURL operation. */
export interface FileUploadRangeFromURLExceptionHeaders {
    errorCode?: string;
    copySourceErrorCode?: string;
    copySourceStatusCode?: number;
}
/** Defines headers for File_getRangeList operation. */
export interface FileGetRangeListHeaders {
    /** The date/time that the file was last modified. Any operation that modifies the file, including an update of the file's metadata or properties, changes the file's last modified time. */
    lastModified?: Date;
    /** The ETag contains a value which represents the version of the file, in quotes. */
    etag?: string;
    /** The size of the file in bytes. */
    fileContentLength?: number;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for File_getRangeList operation. */
export interface FileGetRangeListExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for File_startCopy operation. */
export interface FileStartCopyHeaders {
    /** If the copy is completed, contains the ETag of the destination file. If the copy is not complete, contains the ETag of the empty file created at the start of the copy. */
    etag?: string;
    /** Returns the date/time that the copy operation to the destination file completed. */
    lastModified?: Date;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** String identifier for this copy operation. Use with Get File or Get File Properties to check the status of this copy operation, or pass to Abort Copy File to abort a pending copy. */
    copyId?: string;
    /** State of the copy operation identified by x-ms-copy-id. */
    copyStatus?: CopyStatusType;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for File_startCopy operation. */
export interface FileStartCopyExceptionHeaders {
    errorCode?: string;
    copySourceErrorCode?: string;
    copySourceStatusCode?: number;
}
/** Defines headers for File_abortCopy operation. */
export interface FileAbortCopyHeaders {
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for File_abortCopy operation. */
export interface FileAbortCopyExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for File_listHandles operation. */
export interface FileListHandlesHeaders {
    /** Specifies the format in which the results are returned. Currently this value is 'application/xml'. */
    contentType?: string;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for File_listHandles operation. */
export interface FileListHandlesExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for File_forceCloseHandles operation. */
export interface FileForceCloseHandlesHeaders {
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** A string describing next handle to be closed. It is returned when more handles need to be closed to complete the request. */
    marker?: string;
    /** Contains count of number of handles closed. */
    numberOfHandlesClosed?: number;
    /** Contains count of number of handles that failed to close. */
    numberOfHandlesFailedToClose?: number;
    /** Error Code */
    errorCode?: string;
}
/** Defines headers for File_forceCloseHandles operation. */
export interface FileForceCloseHandlesExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for File_rename operation. */
export interface FileRenameHeaders {
    /** The ETag contains a value which represents the version of the file, in quotes. */
    etag?: string;
    /** Returns the date and time the share was last modified. Any operation that modifies the directory or its properties updates the last modified time. Operations on files do not affect the last modified time of the directory. */
    lastModified?: Date;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** The value of this header is set to true if the contents of the request are successfully encrypted using the specified algorithm, and false otherwise. */
    isServerEncrypted?: boolean;
    /** Key of the permission set for the file. */
    filePermissionKey?: string;
    /** Attributes set for the file. */
    fileAttributes?: string;
    /** Creation time for the file. */
    fileCreationTime?: Date;
    /** Last write time for the file. */
    fileLastWriteTime?: Date;
    /** Change time for the file. */
    fileChangeTime?: Date;
    /** The fileId of the file. */
    fileId?: string;
    /** The parent fileId of the directory. */
    fileParentId?: string;
}
/** Defines headers for File_rename operation. */
export interface FileRenameExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for File_createSymbolicLink operation. */
export interface FileCreateSymbolicLinkHeaders {
    /** The ETag contains a value which represents the version of the file, in quotes. */
    etag?: string;
    /** Returns the date and time the share was last modified. Any operation that modifies the directory or its properties updates the last modified time. Operations on files do not affect the last modified time of the directory. */
    lastModified?: Date;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** Creation time for the file. */
    fileCreationTime?: Date;
    /** Last write time for the file. */
    fileLastWriteTime?: Date;
    /** Change time for the file. */
    fileChangeTime?: Date;
    /** The fileId of the file. */
    fileId?: string;
    /** The parent fileId of the directory. */
    fileParentId?: string;
    /** If a client request id header is sent in the request, this header will be present in the response with the same value. */
    clientRequestId?: string;
    /** NFS only. The mode of the file or directory. */
    fileMode?: string;
    /** NFS only. The owner of the file or directory. */
    owner?: string;
    /** NFS only. The owning group of the file or directory. */
    group?: string;
    /** NFS only. Type of the file or directory. */
    nfsFileType?: NfsFileType;
}
/** Defines headers for File_createSymbolicLink operation. */
export interface FileCreateSymbolicLinkExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for File_getSymbolicLink operation. */
export interface FileGetSymbolicLinkHeaders {
    /** The ETag contains a value which represents the version of the file, in quotes. */
    etag?: string;
    /** Returns the date and time the share was last modified. Any operation that modifies the directory or its properties updates the last modified time. Operations on files do not affect the last modified time of the directory. */
    lastModified?: Date;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** The path to the original file, the symbolic link is pointing to. The path is of type string which is not resolved and is stored as is. The path can be absolute path or the relative path depending on the content stored in the symbolic link file. */
    linkText?: string;
    /** If a client request id header is sent in the request, this header will be present in the response with the same value. */
    clientRequestId?: string;
}
/** Defines headers for File_getSymbolicLink operation. */
export interface FileGetSymbolicLinkExceptionHeaders {
    errorCode?: string;
}
/** Defines headers for File_createHardLink operation. */
export interface FileCreateHardLinkHeaders {
    /** The ETag contains a value which represents the version of the file, in quotes. */
    etag?: string;
    /** Returns the date and time the share was last modified. Any operation that modifies the directory or its properties updates the last modified time. Operations on files do not affect the last modified time of the directory. */
    lastModified?: Date;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the File service used to execute the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** Creation time for the file. */
    fileCreationTime?: Date;
    /** Last write time for the file. */
    fileLastWriteTime?: Date;
    /** Change time for the file. */
    fileChangeTime?: Date;
    /** The fileId of the file. */
    fileId?: string;
    /** The parent fileId of the directory. */
    fileParentId?: string;
    /** If a client request id header is sent in the request, this header will be present in the response with the same value. */
    clientRequestId?: string;
    /** NFS only. The link count of the file or directory. */
    linkCount?: number;
    /** NFS only. The mode of the file or directory. */
    fileMode?: string;
    /** NFS only. The owner of the file or directory. */
    owner?: string;
    /** NFS only. The owning group of the file or directory. */
    group?: string;
    /** NFS only. Type of the file or directory. */
    nfsFileType?: NfsFileType;
}
/** Defines headers for File_createHardLink operation. */
export interface FileCreateHardLinkExceptionHeaders {
    errorCode?: string;
}
/** Parameter group */
export interface LeaseAccessConditions {
    /** If specified, the operation only succeeds if the resource's lease is active and matches this ID. */
    leaseId?: string;
}
/** Parameter group */
export interface SourceLeaseAccessConditions {
    /** Required if the source file has an active infinite lease. */
    sourceLeaseId?: string;
}
/** Parameter group */
export interface DestinationLeaseAccessConditions {
    /** Required if the destination file has an active infinite lease. The lease ID specified for this header must match the lease ID of the destination file. If the request does not include the lease ID or it is not valid, the operation fails with status code 412 (Precondition Failed). If this header is specified and the destination file does not currently have an active lease, the operation will also fail with status code 412 (Precondition Failed). */
    destinationLeaseId?: string;
}
/** Parameter group */
export interface CopyFileSmbInfo {
    /** Specifies either the option to copy file attributes from a source file(source) to a target file or a list of attributes to set on a target file. */
    fileAttributes?: string;
    /** Specifies either the option to copy file creation time from a source file(source) to a target file or a time value in ISO 8601 format to set as creation time on a target file. */
    fileCreationTime?: string;
    /** Specifies either the option to copy file last write time from a source file(source) to a target file or a time value in ISO 8601 format to set as last write time on a target file. */
    fileLastWriteTime?: string;
    /** Specifies either the option to copy file last write time from a source file(source) to a target file or a time value in ISO 8601 format to set as last write time on a target file. */
    fileChangeTime?: string;
    /** Specifies the option to copy file security descriptor from source file or to set it using the value which is defined by the header value of x-ms-file-permission or x-ms-file-permission-key. */
    filePermissionCopyMode?: PermissionCopyModeType;
    /** Specifies the option to overwrite the target file if it already exists and has read-only attribute set. */
    ignoreReadOnly?: boolean;
    /** Specifies the option to set archive attribute on a target file. True means archive attribute will be set on a target file despite attribute overrides or a source file state. */
    setArchiveAttribute?: boolean;
}
/** Parameter group */
export interface FileHttpHeaders {
    /** Sets the MIME content type of the file. The default type is 'application/octet-stream'. */
    fileContentType?: string;
    /** Specifies which content encodings have been applied to the file. */
    fileContentEncoding?: string;
    /** Specifies the natural languages used by this resource. */
    fileContentLanguage?: string;
    /** Sets the file's cache control. The File service stores this value but does not use or modify it. */
    fileCacheControl?: string;
    /** Sets the file's MD5 hash. */
    fileContentMD5?: Uint8Array;
    /** Sets the file's Content-Disposition header. */
    fileContentDisposition?: string;
}
/** Parameter group */
export interface SourceModifiedAccessConditions {
    /** Specify the crc64 value to operate only on range with a matching crc64 checksum. */
    sourceIfMatchCrc64?: Uint8Array;
    /** Specify the crc64 value to operate only on range without a matching crc64 checksum. */
    sourceIfNoneMatchCrc64?: Uint8Array;
}
/** Known values of {@link ShareTokenIntent} that the service accepts. */
export declare enum KnownShareTokenIntent {
    /** Backup */
    Backup = "backup"
}
/**
 * Defines values for ShareTokenIntent. \
 * {@link KnownShareTokenIntent} can be used interchangeably with ShareTokenIntent,
 *  this enum contains the known values that the service supports.
 * ### Known values supported by the service
 * **backup**
 */
export type ShareTokenIntent = string;
/** Known values of {@link NfsFileType} that the service accepts. */
export declare enum KnownNfsFileType {
    /** Regular */
    Regular = "Regular",
    /** Directory */
    Directory = "Directory",
    /** SymLink */
    SymLink = "SymLink"
}
/**
 * Defines values for NfsFileType. \
 * {@link KnownNfsFileType} can be used interchangeably with NfsFileType,
 *  this enum contains the known values that the service supports.
 * ### Known values supported by the service
 * **Regular** \
 * **Directory** \
 * **SymLink**
 */
export type NfsFileType = string;
/** Known values of {@link StorageErrorCode} that the service accepts. */
export declare enum KnownStorageErrorCode {
    /** AccountAlreadyExists */
    AccountAlreadyExists = "AccountAlreadyExists",
    /** AccountBeingCreated */
    AccountBeingCreated = "AccountBeingCreated",
    /** AccountIsDisabled */
    AccountIsDisabled = "AccountIsDisabled",
    /** AuthenticationFailed */
    AuthenticationFailed = "AuthenticationFailed",
    /** AuthorizationFailure */
    AuthorizationFailure = "AuthorizationFailure",
    /** ConditionHeadersNotSupported */
    ConditionHeadersNotSupported = "ConditionHeadersNotSupported",
    /** ConditionNotMet */
    ConditionNotMet = "ConditionNotMet",
    /** EmptyMetadataKey */
    EmptyMetadataKey = "EmptyMetadataKey",
    /** FileShareProvisionedBandwidthDowngradeNotAllowed */
    FileShareProvisionedBandwidthDowngradeNotAllowed = "FileShareProvisionedBandwidthDowngradeNotAllowed",
    /** FileShareProvisionedIopsDowngradeNotAllowed */
    FileShareProvisionedIopsDowngradeNotAllowed = "FileShareProvisionedIopsDowngradeNotAllowed",
    /** InsufficientAccountPermissions */
    InsufficientAccountPermissions = "InsufficientAccountPermissions",
    /** InternalError */
    InternalError = "InternalError",
    /** InvalidAuthenticationInfo */
    InvalidAuthenticationInfo = "InvalidAuthenticationInfo",
    /** InvalidHeaderValue */
    InvalidHeaderValue = "InvalidHeaderValue",
    /** InvalidHttpVerb */
    InvalidHttpVerb = "InvalidHttpVerb",
    /** InvalidInput */
    InvalidInput = "InvalidInput",
    /** InvalidMd5 */
    InvalidMd5 = "InvalidMd5",
    /** InvalidMetadata */
    InvalidMetadata = "InvalidMetadata",
    /** InvalidQueryParameterValue */
    InvalidQueryParameterValue = "InvalidQueryParameterValue",
    /** InvalidRange */
    InvalidRange = "InvalidRange",
    /** InvalidResourceName */
    InvalidResourceName = "InvalidResourceName",
    /** InvalidUri */
    InvalidUri = "InvalidUri",
    /** InvalidXmlDocument */
    InvalidXmlDocument = "InvalidXmlDocument",
    /** InvalidXmlNodeValue */
    InvalidXmlNodeValue = "InvalidXmlNodeValue",
    /** Md5Mismatch */
    Md5Mismatch = "Md5Mismatch",
    /** MetadataTooLarge */
    MetadataTooLarge = "MetadataTooLarge",
    /** MissingContentLengthHeader */
    MissingContentLengthHeader = "MissingContentLengthHeader",
    /** MissingRequiredQueryParameter */
    MissingRequiredQueryParameter = "MissingRequiredQueryParameter",
    /** MissingRequiredHeader */
    MissingRequiredHeader = "MissingRequiredHeader",
    /** MissingRequiredXmlNode */
    MissingRequiredXmlNode = "MissingRequiredXmlNode",
    /** MultipleConditionHeadersNotSupported */
    MultipleConditionHeadersNotSupported = "MultipleConditionHeadersNotSupported",
    /** OperationTimedOut */
    OperationTimedOut = "OperationTimedOut",
    /** OutOfRangeInput */
    OutOfRangeInput = "OutOfRangeInput",
    /** OutOfRangeQueryParameterValue */
    OutOfRangeQueryParameterValue = "OutOfRangeQueryParameterValue",
    /** RequestBodyTooLarge */
    RequestBodyTooLarge = "RequestBodyTooLarge",
    /** ResourceTypeMismatch */
    ResourceTypeMismatch = "ResourceTypeMismatch",
    /** RequestUrlFailedToParse */
    RequestUrlFailedToParse = "RequestUrlFailedToParse",
    /** ResourceAlreadyExists */
    ResourceAlreadyExists = "ResourceAlreadyExists",
    /** ResourceNotFound */
    ResourceNotFound = "ResourceNotFound",
    /** ServerBusy */
    ServerBusy = "ServerBusy",
    /** UnsupportedHeader */
    UnsupportedHeader = "UnsupportedHeader",
    /** UnsupportedXmlNode */
    UnsupportedXmlNode = "UnsupportedXmlNode",
    /** UnsupportedQueryParameter */
    UnsupportedQueryParameter = "UnsupportedQueryParameter",
    /** UnsupportedHttpVerb */
    UnsupportedHttpVerb = "UnsupportedHttpVerb",
    /** CannotDeleteFileOrDirectory */
    CannotDeleteFileOrDirectory = "CannotDeleteFileOrDirectory",
    /** ClientCacheFlushDelay */
    ClientCacheFlushDelay = "ClientCacheFlushDelay",
    /** DeletePending */
    DeletePending = "DeletePending",
    /** DirectoryNotEmpty */
    DirectoryNotEmpty = "DirectoryNotEmpty",
    /** FileLockConflict */
    FileLockConflict = "FileLockConflict",
    /** InvalidFileOrDirectoryPathName */
    InvalidFileOrDirectoryPathName = "InvalidFileOrDirectoryPathName",
    /** ParentNotFound */
    ParentNotFound = "ParentNotFound",
    /** ReadOnlyAttribute */
    ReadOnlyAttribute = "ReadOnlyAttribute",
    /** ShareAlreadyExists */
    ShareAlreadyExists = "ShareAlreadyExists",
    /** ShareBeingDeleted */
    ShareBeingDeleted = "ShareBeingDeleted",
    /** ShareDisabled */
    ShareDisabled = "ShareDisabled",
    /** ShareNotFound */
    ShareNotFound = "ShareNotFound",
    /** SharingViolation */
    SharingViolation = "SharingViolation",
    /** ShareSnapshotInProgress */
    ShareSnapshotInProgress = "ShareSnapshotInProgress",
    /** ShareSnapshotCountExceeded */
    ShareSnapshotCountExceeded = "ShareSnapshotCountExceeded",
    /** ShareSnapshotOperationNotSupported */
    ShareSnapshotOperationNotSupported = "ShareSnapshotOperationNotSupported",
    /** ShareHasSnapshots */
    ShareHasSnapshots = "ShareHasSnapshots",
    /** PreviousSnapshotNotFound */
    PreviousSnapshotNotFound = "PreviousSnapshotNotFound",
    /** ContainerQuotaDowngradeNotAllowed */
    ContainerQuotaDowngradeNotAllowed = "ContainerQuotaDowngradeNotAllowed",
    /** AuthorizationSourceIPMismatch */
    AuthorizationSourceIPMismatch = "AuthorizationSourceIPMismatch",
    /** AuthorizationProtocolMismatch */
    AuthorizationProtocolMismatch = "AuthorizationProtocolMismatch",
    /** AuthorizationPermissionMismatch */
    AuthorizationPermissionMismatch = "AuthorizationPermissionMismatch",
    /** AuthorizationServiceMismatch */
    AuthorizationServiceMismatch = "AuthorizationServiceMismatch",
    /** AuthorizationResourceTypeMismatch */
    AuthorizationResourceTypeMismatch = "AuthorizationResourceTypeMismatch",
    /** FeatureVersionMismatch */
    FeatureVersionMismatch = "FeatureVersionMismatch",
    /** ShareSnapshotNotFound */
    ShareSnapshotNotFound = "ShareSnapshotNotFound"
}
/**
 * Defines values for StorageErrorCode. \
 * {@link KnownStorageErrorCode} can be used interchangeably with StorageErrorCode,
 *  this enum contains the known values that the service supports.
 * ### Known values supported by the service
 * **AccountAlreadyExists** \
 * **AccountBeingCreated** \
 * **AccountIsDisabled** \
 * **AuthenticationFailed** \
 * **AuthorizationFailure** \
 * **ConditionHeadersNotSupported** \
 * **ConditionNotMet** \
 * **EmptyMetadataKey** \
 * **FileShareProvisionedBandwidthDowngradeNotAllowed** \
 * **FileShareProvisionedIopsDowngradeNotAllowed** \
 * **InsufficientAccountPermissions** \
 * **InternalError** \
 * **InvalidAuthenticationInfo** \
 * **InvalidHeaderValue** \
 * **InvalidHttpVerb** \
 * **InvalidInput** \
 * **InvalidMd5** \
 * **InvalidMetadata** \
 * **InvalidQueryParameterValue** \
 * **InvalidRange** \
 * **InvalidResourceName** \
 * **InvalidUri** \
 * **InvalidXmlDocument** \
 * **InvalidXmlNodeValue** \
 * **Md5Mismatch** \
 * **MetadataTooLarge** \
 * **MissingContentLengthHeader** \
 * **MissingRequiredQueryParameter** \
 * **MissingRequiredHeader** \
 * **MissingRequiredXmlNode** \
 * **MultipleConditionHeadersNotSupported** \
 * **OperationTimedOut** \
 * **OutOfRangeInput** \
 * **OutOfRangeQueryParameterValue** \
 * **RequestBodyTooLarge** \
 * **ResourceTypeMismatch** \
 * **RequestUrlFailedToParse** \
 * **ResourceAlreadyExists** \
 * **ResourceNotFound** \
 * **ServerBusy** \
 * **UnsupportedHeader** \
 * **UnsupportedXmlNode** \
 * **UnsupportedQueryParameter** \
 * **UnsupportedHttpVerb** \
 * **CannotDeleteFileOrDirectory** \
 * **ClientCacheFlushDelay** \
 * **DeletePending** \
 * **DirectoryNotEmpty** \
 * **FileLockConflict** \
 * **InvalidFileOrDirectoryPathName** \
 * **ParentNotFound** \
 * **ReadOnlyAttribute** \
 * **ShareAlreadyExists** \
 * **ShareBeingDeleted** \
 * **ShareDisabled** \
 * **ShareNotFound** \
 * **SharingViolation** \
 * **ShareSnapshotInProgress** \
 * **ShareSnapshotCountExceeded** \
 * **ShareSnapshotOperationNotSupported** \
 * **ShareHasSnapshots** \
 * **PreviousSnapshotNotFound** \
 * **ContainerQuotaDowngradeNotAllowed** \
 * **AuthorizationSourceIPMismatch** \
 * **AuthorizationProtocolMismatch** \
 * **AuthorizationPermissionMismatch** \
 * **AuthorizationServiceMismatch** \
 * **AuthorizationResourceTypeMismatch** \
 * **FeatureVersionMismatch** \
 * **ShareSnapshotNotFound**
 */
export type StorageErrorCode = string;
/** Defines values for ListSharesIncludeType. */
export type ListSharesIncludeType = "snapshots" | "metadata" | "deleted";
/** Defines values for LeaseStatusType. */
export type LeaseStatusType = "locked" | "unlocked";
/** Defines values for LeaseStateType. */
export type LeaseStateType = "available" | "leased" | "expired" | "breaking" | "broken";
/** Defines values for LeaseDurationType. */
export type LeaseDurationType = "infinite" | "fixed";
/** Defines values for ShareRootSquash. */
export type ShareRootSquash = "NoRootSquash" | "RootSquash" | "AllSquash";
/** Defines values for ShareAccessTier. */
export type ShareAccessTier = "TransactionOptimized" | "Hot" | "Cool" | "Premium";
/** Defines values for DeleteSnapshotsOptionType. */
export type DeleteSnapshotsOptionType = "include" | "include-leased";
/** Defines values for FilePermissionFormat. */
export type FilePermissionFormat = "Sddl" | "Binary";
/** Defines values for ListFilesIncludeType. */
export type ListFilesIncludeType = "Timestamps" | "Etag" | "Attributes" | "PermissionKey";
/** Defines values for AccessRight. */
export type AccessRight = "Read" | "Write" | "Delete";
/** Defines values for CopyStatusType. */
export type CopyStatusType = "pending" | "success" | "aborted" | "failed";
/** Defines values for FileRangeWriteType. */
export type FileRangeWriteType = "update" | "clear";
/** Defines values for FileLastWrittenMode. */
export type FileLastWrittenMode = "Now" | "Preserve";
/** Defines values for PermissionCopyModeType. */
export type PermissionCopyModeType = "source" | "override";
/** Defines values for ModeCopyMode. */
export type ModeCopyMode = "source" | "override";
/** Defines values for OwnerCopyMode. */
export type OwnerCopyMode = "source" | "override";
/** Optional parameters. */
export interface ServiceSetPropertiesOptionalParams extends coreClient.OperationOptions {
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
}
/** Contains response data for the setProperties operation. */
export type ServiceSetPropertiesResponse = ServiceSetPropertiesHeaders;
/** Optional parameters. */
export interface ServiceGetPropertiesOptionalParams extends coreClient.OperationOptions {
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
}
/** Contains response data for the getProperties operation. */
export type ServiceGetPropertiesResponse = ServiceGetPropertiesHeaders & FileServiceProperties;
/** Optional parameters. */
export interface ServiceListSharesSegmentOptionalParams extends coreClient.OperationOptions {
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** Filters the results to return only entries whose name begins with the specified prefix. */
    prefix?: string;
    /** A string value that identifies the portion of the list to be returned with the next list operation. The operation returns a marker value within the response body if the list returned was not complete. The marker value may then be used in a subsequent call to request the next set of list items. The marker value is opaque to the client. */
    marker?: string;
    /** Specifies the maximum number of entries to return. If the request does not specify maxresults, or specifies a value greater than 5,000, the server will return up to 5,000 items. */
    maxResults?: number;
    /** Include this parameter to specify one or more datasets to include in the response. */
    include?: ListSharesIncludeType[];
}
/** Contains response data for the listSharesSegment operation. */
export type ServiceListSharesSegmentResponse = ServiceListSharesSegmentHeaders & ListSharesResponse;
/** Optional parameters. */
export interface ShareCreateOptionalParams extends coreClient.OperationOptions {
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** A name-value pair to associate with a file storage object. */
    metadata?: {
        [propertyName: string]: string;
    };
    /** Specifies the maximum size of the share, in gigabytes. */
    quota?: number;
    /** Specifies the access tier of the share. */
    accessTier?: ShareAccessTier;
    /** Protocols to enable on the share. */
    enabledProtocols?: string;
    /** Root squash to set on the share.  Only valid for NFS shares. */
    rootSquash?: ShareRootSquash;
    enableSnapshotVirtualDirectoryAccess?: boolean;
    /** Optional. Boolean. Default if not specified is false. This property enables paid bursting. */
    paidBurstingEnabled?: boolean;
    /** Optional. Integer. Default if not specified is the maximum throughput the file share can support. Current maximum for a file share is 10,340  MiB/sec. */
    paidBurstingMaxBandwidthMibps?: number;
    /** Optional. Integer. Default if not specified is the maximum IOPS the file share can support. Current maximum for a file share is 102,400 IOPS. */
    paidBurstingMaxIops?: number;
    /** Optional. Supported in version 2025-01-05 and later. Only allowed for provisioned v2 file shares. Specifies the provisioned number of input/output operations per second (IOPS) of the share. If this is not specified, the provisioned IOPS is set to value calculated based on recommendation formula. */
    shareProvisionedIops?: number;
    /** Optional. Supported in version 2025-01-05 and later. Only allowed for provisioned v2 file shares. Specifies the provisioned bandwidth of the share, in mebibytes per second (MiBps). If this is not specified, the provisioned bandwidth is set to value calculated based on recommendation formula. */
    shareProvisionedBandwidthMibps?: number;
}
/** Contains response data for the create operation. */
export type ShareCreateResponse = ShareCreateHeaders;
/** Optional parameters. */
export interface ShareGetPropertiesOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    leaseAccessConditions?: LeaseAccessConditions;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** The snapshot parameter is an opaque DateTime value that, when present, specifies the share snapshot to query. */
    shareSnapshot?: string;
}
/** Contains response data for the getProperties operation. */
export type ShareGetPropertiesResponse = ShareGetPropertiesHeaders;
/** Optional parameters. */
export interface ShareDeleteOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    leaseAccessConditions?: LeaseAccessConditions;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** The snapshot parameter is an opaque DateTime value that, when present, specifies the share snapshot to query. */
    shareSnapshot?: string;
    /** Specifies the option include to delete the base share and all of its snapshots. */
    deleteSnapshots?: DeleteSnapshotsOptionType;
}
/** Contains response data for the delete operation. */
export type ShareDeleteResponse = ShareDeleteHeaders;
/** Optional parameters. */
export interface ShareAcquireLeaseOptionalParams extends coreClient.OperationOptions {
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** The snapshot parameter is an opaque DateTime value that, when present, specifies the share snapshot to query. */
    shareSnapshot?: string;
    /** Specifies the duration of the lease, in seconds, or negative one (-1) for a lease that never expires. A non-infinite lease can be between 15 and 60 seconds. A lease duration cannot be changed using renew or change. */
    duration?: number;
    /** Proposed lease ID, in a GUID string format. The File service returns 400 (Invalid request) if the proposed lease ID is not in the correct format. See Guid Constructor (String) for a list of valid GUID string formats. */
    proposedLeaseId?: string;
    /** Provides a client-generated, opaque value with a 1 KB character limit that is recorded in the analytics logs when storage analytics logging is enabled. */
    requestId?: string;
}
/** Contains response data for the acquireLease operation. */
export type ShareAcquireLeaseResponse = ShareAcquireLeaseHeaders;
/** Optional parameters. */
export interface ShareReleaseLeaseOptionalParams extends coreClient.OperationOptions {
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** The snapshot parameter is an opaque DateTime value that, when present, specifies the share snapshot to query. */
    shareSnapshot?: string;
    /** Provides a client-generated, opaque value with a 1 KB character limit that is recorded in the analytics logs when storage analytics logging is enabled. */
    requestId?: string;
}
/** Contains response data for the releaseLease operation. */
export type ShareReleaseLeaseResponse = ShareReleaseLeaseHeaders;
/** Optional parameters. */
export interface ShareChangeLeaseOptionalParams extends coreClient.OperationOptions {
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** The snapshot parameter is an opaque DateTime value that, when present, specifies the share snapshot to query. */
    shareSnapshot?: string;
    /** Proposed lease ID, in a GUID string format. The File service returns 400 (Invalid request) if the proposed lease ID is not in the correct format. See Guid Constructor (String) for a list of valid GUID string formats. */
    proposedLeaseId?: string;
    /** Provides a client-generated, opaque value with a 1 KB character limit that is recorded in the analytics logs when storage analytics logging is enabled. */
    requestId?: string;
}
/** Contains response data for the changeLease operation. */
export type ShareChangeLeaseResponse = ShareChangeLeaseHeaders;
/** Optional parameters. */
export interface ShareRenewLeaseOptionalParams extends coreClient.OperationOptions {
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** The snapshot parameter is an opaque DateTime value that, when present, specifies the share snapshot to query. */
    shareSnapshot?: string;
    /** Provides a client-generated, opaque value with a 1 KB character limit that is recorded in the analytics logs when storage analytics logging is enabled. */
    requestId?: string;
}
/** Contains response data for the renewLease operation. */
export type ShareRenewLeaseResponse = ShareRenewLeaseHeaders;
/** Optional parameters. */
export interface ShareBreakLeaseOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    leaseAccessConditions?: LeaseAccessConditions;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** The snapshot parameter is an opaque DateTime value that, when present, specifies the share snapshot to query. */
    shareSnapshot?: string;
    /** Provides a client-generated, opaque value with a 1 KB character limit that is recorded in the analytics logs when storage analytics logging is enabled. */
    requestId?: string;
    /** For a break operation, proposed duration the lease should continue before it is broken, in seconds, between 0 and 60. This break period is only used if it is shorter than the time remaining on the lease. If longer, the time remaining on the lease is used. A new lease will not be available before the break period has expired, but the lease may be held for longer than the break period. If this header does not appear with a break operation, a fixed-duration lease breaks after the remaining lease period elapses, and an infinite lease breaks immediately. */
    breakPeriod?: number;
}
/** Contains response data for the breakLease operation. */
export type ShareBreakLeaseResponse = ShareBreakLeaseHeaders;
/** Optional parameters. */
export interface ShareCreateSnapshotOptionalParams extends coreClient.OperationOptions {
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** A name-value pair to associate with a file storage object. */
    metadata?: {
        [propertyName: string]: string;
    };
}
/** Contains response data for the createSnapshot operation. */
export type ShareCreateSnapshotResponse = ShareCreateSnapshotHeaders;
/** Optional parameters. */
export interface ShareCreatePermissionOptionalParams extends coreClient.OperationOptions {
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
}
/** Contains response data for the createPermission operation. */
export type ShareCreatePermissionResponse = ShareCreatePermissionHeaders;
/** Optional parameters. */
export interface ShareGetPermissionOptionalParams extends coreClient.OperationOptions {
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** Optional. Available for version 2023-06-01 and later. Specifies the format in which the permission is returned. Acceptable values are SDDL or binary. If x-ms-file-permission-format is unspecified or explicitly set to SDDL, the permission is returned in SDDL format. If x-ms-file-permission-format is explicitly set to binary, the permission is returned as a base64 string representing the binary encoding of the permission */
    filePermissionFormat?: FilePermissionFormat;
}
/** Contains response data for the getPermission operation. */
export type ShareGetPermissionResponse = ShareGetPermissionHeaders & SharePermission;
/** Optional parameters. */
export interface ShareSetPropertiesOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    leaseAccessConditions?: LeaseAccessConditions;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** Specifies the maximum size of the share, in gigabytes. */
    quota?: number;
    /** Specifies the access tier of the share. */
    accessTier?: ShareAccessTier;
    /** Root squash to set on the share.  Only valid for NFS shares. */
    rootSquash?: ShareRootSquash;
    enableSnapshotVirtualDirectoryAccess?: boolean;
    /** Optional. Boolean. Default if not specified is false. This property enables paid bursting. */
    paidBurstingEnabled?: boolean;
    /** Optional. Integer. Default if not specified is the maximum throughput the file share can support. Current maximum for a file share is 10,340  MiB/sec. */
    paidBurstingMaxBandwidthMibps?: number;
    /** Optional. Integer. Default if not specified is the maximum IOPS the file share can support. Current maximum for a file share is 102,400 IOPS. */
    paidBurstingMaxIops?: number;
    /** Optional. Supported in version 2025-01-05 and later. Only allowed for provisioned v2 file shares. Specifies the provisioned number of input/output operations per second (IOPS) of the share. If this is not specified, the provisioned IOPS is set to value calculated based on recommendation formula. */
    shareProvisionedIops?: number;
    /** Optional. Supported in version 2025-01-05 and later. Only allowed for provisioned v2 file shares. Specifies the provisioned bandwidth of the share, in mebibytes per second (MiBps). If this is not specified, the provisioned bandwidth is set to value calculated based on recommendation formula. */
    shareProvisionedBandwidthMibps?: number;
}
/** Contains response data for the setProperties operation. */
export type ShareSetPropertiesResponse = ShareSetPropertiesHeaders;
/** Optional parameters. */
export interface ShareSetMetadataOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    leaseAccessConditions?: LeaseAccessConditions;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** A name-value pair to associate with a file storage object. */
    metadata?: {
        [propertyName: string]: string;
    };
}
/** Contains response data for the setMetadata operation. */
export type ShareSetMetadataResponse = ShareSetMetadataHeaders;
/** Optional parameters. */
export interface ShareGetAccessPolicyOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    leaseAccessConditions?: LeaseAccessConditions;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
}
/** Contains response data for the getAccessPolicy operation. */
export type ShareGetAccessPolicyResponse = ShareGetAccessPolicyHeaders & SignedIdentifier[];
/** Optional parameters. */
export interface ShareSetAccessPolicyOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    leaseAccessConditions?: LeaseAccessConditions;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** The ACL for the share. */
    shareAcl?: SignedIdentifier[];
}
/** Contains response data for the setAccessPolicy operation. */
export type ShareSetAccessPolicyResponse = ShareSetAccessPolicyHeaders;
/** Optional parameters. */
export interface ShareGetStatisticsOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    leaseAccessConditions?: LeaseAccessConditions;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
}
/** Contains response data for the getStatistics operation. */
export type ShareGetStatisticsResponse = ShareGetStatisticsHeaders & ShareStats;
/** Optional parameters. */
export interface ShareRestoreOptionalParams extends coreClient.OperationOptions {
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** Provides a client-generated, opaque value with a 1 KB character limit that is recorded in the analytics logs when storage analytics logging is enabled. */
    requestId?: string;
    /** Specifies the name of the previously-deleted share. */
    deletedShareName?: string;
    /** Specifies the version of the previously-deleted share. */
    deletedShareVersion?: string;
}
/** Contains response data for the restore operation. */
export type ShareRestoreResponse = ShareRestoreHeaders;
/** Optional parameters. */
export interface DirectoryCreateOptionalParams extends coreClient.OperationOptions {
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** A name-value pair to associate with a file storage object. */
    metadata?: {
        [propertyName: string]: string;
    };
    /** Optional. Available for version 2023-06-01 and later. Specifies the format in which the permission is returned. Acceptable values are SDDL or binary. If x-ms-file-permission-format is unspecified or explicitly set to SDDL, the permission is returned in SDDL format. If x-ms-file-permission-format is explicitly set to binary, the permission is returned as a base64 string representing the binary encoding of the permission */
    filePermissionFormat?: FilePermissionFormat;
    /** If true, the trailing dot will not be trimmed from the target URI. */
    allowTrailingDot?: boolean;
    /** If specified the permission (security descriptor) shall be set for the directory/file. This header can be used if Permission size is <= 8KB, else x-ms-file-permission-key header shall be used. Default value: Inherit. If SDDL is specified as input, it must have owner, group and dacl. Note: Only one of the x-ms-file-permission or x-ms-file-permission-key should be specified. */
    filePermission?: string;
    /** Key of the permission to be set for the directory/file. Note: Only one of the x-ms-file-permission or x-ms-file-permission-key should be specified. */
    filePermissionKey?: string;
    /** If specified, the provided file attributes shall be set. Default value: ‘Archive’ for file and ‘Directory’ for directory. ‘None’ can also be specified as default. */
    fileAttributes?: string;
    /** Creation time for the file/directory. Default value: Now. */
    fileCreatedOn?: string;
    /** Last write time for the file/directory. Default value: Now. */
    fileLastWriteOn?: string;
    /** Change time for the file/directory. Default value: Now. */
    fileChangeOn?: string;
    /** Optional, NFS only. The owner of the file or directory. */
    owner?: string;
    /** Optional, NFS only. The owning group of the file or directory. */
    group?: string;
    /** Optional, NFS only. The file mode of the file or directory */
    fileMode?: string;
}
/** Contains response data for the create operation. */
export type DirectoryCreateResponse = DirectoryCreateHeaders;
/** Optional parameters. */
export interface DirectoryGetPropertiesOptionalParams extends coreClient.OperationOptions {
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** The snapshot parameter is an opaque DateTime value that, when present, specifies the share snapshot to query. */
    shareSnapshot?: string;
    /** If true, the trailing dot will not be trimmed from the target URI. */
    allowTrailingDot?: boolean;
}
/** Contains response data for the getProperties operation. */
export type DirectoryGetPropertiesResponse = DirectoryGetPropertiesHeaders;
/** Optional parameters. */
export interface DirectoryDeleteOptionalParams extends coreClient.OperationOptions {
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** If true, the trailing dot will not be trimmed from the target URI. */
    allowTrailingDot?: boolean;
}
/** Contains response data for the delete operation. */
export type DirectoryDeleteResponse = DirectoryDeleteHeaders;
/** Optional parameters. */
export interface DirectorySetPropertiesOptionalParams extends coreClient.OperationOptions {
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** Optional. Available for version 2023-06-01 and later. Specifies the format in which the permission is returned. Acceptable values are SDDL or binary. If x-ms-file-permission-format is unspecified or explicitly set to SDDL, the permission is returned in SDDL format. If x-ms-file-permission-format is explicitly set to binary, the permission is returned as a base64 string representing the binary encoding of the permission */
    filePermissionFormat?: FilePermissionFormat;
    /** If true, the trailing dot will not be trimmed from the target URI. */
    allowTrailingDot?: boolean;
    /** If specified the permission (security descriptor) shall be set for the directory/file. This header can be used if Permission size is <= 8KB, else x-ms-file-permission-key header shall be used. Default value: Inherit. If SDDL is specified as input, it must have owner, group and dacl. Note: Only one of the x-ms-file-permission or x-ms-file-permission-key should be specified. */
    filePermission?: string;
    /** Key of the permission to be set for the directory/file. Note: Only one of the x-ms-file-permission or x-ms-file-permission-key should be specified. */
    filePermissionKey?: string;
    /** If specified, the provided file attributes shall be set. Default value: ‘Archive’ for file and ‘Directory’ for directory. ‘None’ can also be specified as default. */
    fileAttributes?: string;
    /** Creation time for the file/directory. Default value: Now. */
    fileCreatedOn?: string;
    /** Last write time for the file/directory. Default value: Now. */
    fileLastWriteOn?: string;
    /** Change time for the file/directory. Default value: Now. */
    fileChangeOn?: string;
    /** Optional, NFS only. The owner of the file or directory. */
    owner?: string;
    /** Optional, NFS only. The owning group of the file or directory. */
    group?: string;
    /** Optional, NFS only. The file mode of the file or directory */
    fileMode?: string;
}
/** Contains response data for the setProperties operation. */
export type DirectorySetPropertiesResponse = DirectorySetPropertiesHeaders;
/** Optional parameters. */
export interface DirectorySetMetadataOptionalParams extends coreClient.OperationOptions {
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** A name-value pair to associate with a file storage object. */
    metadata?: {
        [propertyName: string]: string;
    };
    /** If true, the trailing dot will not be trimmed from the target URI. */
    allowTrailingDot?: boolean;
}
/** Contains response data for the setMetadata operation. */
export type DirectorySetMetadataResponse = DirectorySetMetadataHeaders;
/** Optional parameters. */
export interface DirectoryListFilesAndDirectoriesSegmentOptionalParams extends coreClient.OperationOptions {
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** Filters the results to return only entries whose name begins with the specified prefix. */
    prefix?: string;
    /** A string value that identifies the portion of the list to be returned with the next list operation. The operation returns a marker value within the response body if the list returned was not complete. The marker value may then be used in a subsequent call to request the next set of list items. The marker value is opaque to the client. */
    marker?: string;
    /** Specifies the maximum number of entries to return. If the request does not specify maxresults, or specifies a value greater than 5,000, the server will return up to 5,000 items. */
    maxResults?: number;
    /** The snapshot parameter is an opaque DateTime value that, when present, specifies the share snapshot to query. */
    shareSnapshot?: string;
    /** If true, the trailing dot will not be trimmed from the target URI. */
    allowTrailingDot?: boolean;
    /** Include this parameter to specify one or more datasets to include in the response. */
    include?: ListFilesIncludeType[];
    /** Include extended information. */
    includeExtendedInfo?: boolean;
}
/** Contains response data for the listFilesAndDirectoriesSegment operation. */
export type DirectoryListFilesAndDirectoriesSegmentResponse = DirectoryListFilesAndDirectoriesSegmentHeaders & ListFilesAndDirectoriesSegmentResponse;
/** Optional parameters. */
export interface DirectoryListHandlesOptionalParams extends coreClient.OperationOptions {
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** A string value that identifies the portion of the list to be returned with the next list operation. The operation returns a marker value within the response body if the list returned was not complete. The marker value may then be used in a subsequent call to request the next set of list items. The marker value is opaque to the client. */
    marker?: string;
    /** Specifies the maximum number of entries to return. If the request does not specify maxresults, or specifies a value greater than 5,000, the server will return up to 5,000 items. */
    maxResults?: number;
    /** The snapshot parameter is an opaque DateTime value that, when present, specifies the share snapshot to query. */
    shareSnapshot?: string;
    /** If true, the trailing dot will not be trimmed from the target URI. */
    allowTrailingDot?: boolean;
    /** Specifies operation should apply to the directory specified in the URI, its files, its subdirectories and their files. */
    recursive?: boolean;
}
/** Contains response data for the listHandles operation. */
export type DirectoryListHandlesResponse = DirectoryListHandlesHeaders & ListHandlesResponse;
/** Optional parameters. */
export interface DirectoryForceCloseHandlesOptionalParams extends coreClient.OperationOptions {
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** A string value that identifies the portion of the list to be returned with the next list operation. The operation returns a marker value within the response body if the list returned was not complete. The marker value may then be used in a subsequent call to request the next set of list items. The marker value is opaque to the client. */
    marker?: string;
    /** The snapshot parameter is an opaque DateTime value that, when present, specifies the share snapshot to query. */
    shareSnapshot?: string;
    /** If true, the trailing dot will not be trimmed from the target URI. */
    allowTrailingDot?: boolean;
    /** Specifies operation should apply to the directory specified in the URI, its files, its subdirectories and their files. */
    recursive?: boolean;
}
/** Contains response data for the forceCloseHandles operation. */
export type DirectoryForceCloseHandlesResponse = DirectoryForceCloseHandlesHeaders;
/** Optional parameters. */
export interface DirectoryRenameOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    sourceLeaseAccessConditions?: SourceLeaseAccessConditions;
    /** Parameter group */
    destinationLeaseAccessConditions?: DestinationLeaseAccessConditions;
    /** Parameter group */
    copyFileSmbInfo?: CopyFileSmbInfo;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** A name-value pair to associate with a file storage object. */
    metadata?: {
        [propertyName: string]: string;
    };
    /** Optional. Available for version 2023-06-01 and later. Specifies the format in which the permission is returned. Acceptable values are SDDL or binary. If x-ms-file-permission-format is unspecified or explicitly set to SDDL, the permission is returned in SDDL format. If x-ms-file-permission-format is explicitly set to binary, the permission is returned as a base64 string representing the binary encoding of the permission */
    filePermissionFormat?: FilePermissionFormat;
    /** If true, the trailing dot will not be trimmed from the target URI. */
    allowTrailingDot?: boolean;
    /** If specified the permission (security descriptor) shall be set for the directory/file. This header can be used if Permission size is <= 8KB, else x-ms-file-permission-key header shall be used. Default value: Inherit. If SDDL is specified as input, it must have owner, group and dacl. Note: Only one of the x-ms-file-permission or x-ms-file-permission-key should be specified. */
    filePermission?: string;
    /** Key of the permission to be set for the directory/file. Note: Only one of the x-ms-file-permission or x-ms-file-permission-key should be specified. */
    filePermissionKey?: string;
    /** Optional. A boolean value for if the destination file already exists, whether this request will overwrite the file or not. If true, the rename will succeed and will overwrite the destination file. If not provided or if false and the destination file does exist, the request will not overwrite the destination file. If provided and the destination file doesn’t exist, the rename will succeed. Note: This value does not override the x-ms-file-copy-ignore-read-only header value. */
    replaceIfExists?: boolean;
    /** Optional. A boolean value that specifies whether the ReadOnly attribute on a preexisting destination file should be respected. If true, the rename will succeed, otherwise, a previous file at the destination with the ReadOnly attribute set will cause the rename to fail. */
    ignoreReadOnly?: boolean;
    /** If true, the trailing dot will not be trimmed from the source URI. */
    allowSourceTrailingDot?: boolean;
}
/** Contains response data for the rename operation. */
export type DirectoryRenameResponse = DirectoryRenameHeaders;
/** Optional parameters. */
export interface FileCreateOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    leaseAccessConditions?: LeaseAccessConditions;
    /** Parameter group */
    fileHttpHeaders?: FileHttpHeaders;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** A name-value pair to associate with a file storage object. */
    metadata?: {
        [propertyName: string]: string;
    };
    /** Optional. Available for version 2023-06-01 and later. Specifies the format in which the permission is returned. Acceptable values are SDDL or binary. If x-ms-file-permission-format is unspecified or explicitly set to SDDL, the permission is returned in SDDL format. If x-ms-file-permission-format is explicitly set to binary, the permission is returned as a base64 string representing the binary encoding of the permission */
    filePermissionFormat?: FilePermissionFormat;
    /** If true, the trailing dot will not be trimmed from the target URI. */
    allowTrailingDot?: boolean;
    /** If specified the permission (security descriptor) shall be set for the directory/file. This header can be used if Permission size is <= 8KB, else x-ms-file-permission-key header shall be used. Default value: Inherit. If SDDL is specified as input, it must have owner, group and dacl. Note: Only one of the x-ms-file-permission or x-ms-file-permission-key should be specified. */
    filePermission?: string;
    /** Key of the permission to be set for the directory/file. Note: Only one of the x-ms-file-permission or x-ms-file-permission-key should be specified. */
    filePermissionKey?: string;
    /** If specified, the provided file attributes shall be set. Default value: ‘Archive’ for file and ‘Directory’ for directory. ‘None’ can also be specified as default. */
    fileAttributes?: string;
    /** Creation time for the file/directory. Default value: Now. */
    fileCreatedOn?: string;
    /** Last write time for the file/directory. Default value: Now. */
    fileLastWriteOn?: string;
    /** Change time for the file/directory. Default value: Now. */
    fileChangeOn?: string;
    /** Optional, NFS only. The owner of the file or directory. */
    owner?: string;
    /** Optional, NFS only. The owning group of the file or directory. */
    group?: string;
    /** Optional, NFS only. The file mode of the file or directory */
    fileMode?: string;
    /** Optional, NFS only. Type of the file or directory. */
    nfsFileType?: NfsFileType;
}
/** Contains response data for the create operation. */
export type FileCreateResponse = FileCreateHeaders;
/** Optional parameters. */
export interface FileDownloadOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    leaseAccessConditions?: LeaseAccessConditions;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** If true, the trailing dot will not be trimmed from the target URI. */
    allowTrailingDot?: boolean;
    /** Return file data only from the specified byte range. */
    range?: string;
    /** When this header is set to true and specified together with the Range header, the service returns the MD5 hash for the range, as long as the range is less than or equal to 4 MB in size. */
    rangeGetContentMD5?: boolean;
}
/** Contains response data for the download operation. */
export type FileDownloadResponse = FileDownloadHeaders & {
    /**
     * BROWSER ONLY
     *
     * The response body as a browser Blob.
     * Always `undefined` in node.js.
     */
    blobBody?: Promise<Blob>;
    /**
     * NODEJS ONLY
     *
     * The response body as a node.js Readable stream.
     * Always `undefined` in the browser.
     */
    readableStreamBody?: NodeJS.ReadableStream;
};
/** Optional parameters. */
export interface FileGetPropertiesOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    leaseAccessConditions?: LeaseAccessConditions;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** The snapshot parameter is an opaque DateTime value that, when present, specifies the share snapshot to query. */
    shareSnapshot?: string;
    /** If true, the trailing dot will not be trimmed from the target URI. */
    allowTrailingDot?: boolean;
}
/** Contains response data for the getProperties operation. */
export type FileGetPropertiesResponse = FileGetPropertiesHeaders;
/** Optional parameters. */
export interface FileDeleteOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    leaseAccessConditions?: LeaseAccessConditions;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** If true, the trailing dot will not be trimmed from the target URI. */
    allowTrailingDot?: boolean;
}
/** Contains response data for the delete operation. */
export type FileDeleteResponse = FileDeleteHeaders;
/** Optional parameters. */
export interface FileSetHttpHeadersOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    leaseAccessConditions?: LeaseAccessConditions;
    /** Parameter group */
    fileHttpHeaders?: FileHttpHeaders;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** Optional. Available for version 2023-06-01 and later. Specifies the format in which the permission is returned. Acceptable values are SDDL or binary. If x-ms-file-permission-format is unspecified or explicitly set to SDDL, the permission is returned in SDDL format. If x-ms-file-permission-format is explicitly set to binary, the permission is returned as a base64 string representing the binary encoding of the permission */
    filePermissionFormat?: FilePermissionFormat;
    /** If true, the trailing dot will not be trimmed from the target URI. */
    allowTrailingDot?: boolean;
    /** If specified the permission (security descriptor) shall be set for the directory/file. This header can be used if Permission size is <= 8KB, else x-ms-file-permission-key header shall be used. Default value: Inherit. If SDDL is specified as input, it must have owner, group and dacl. Note: Only one of the x-ms-file-permission or x-ms-file-permission-key should be specified. */
    filePermission?: string;
    /** Key of the permission to be set for the directory/file. Note: Only one of the x-ms-file-permission or x-ms-file-permission-key should be specified. */
    filePermissionKey?: string;
    /** If specified, the provided file attributes shall be set. Default value: ‘Archive’ for file and ‘Directory’ for directory. ‘None’ can also be specified as default. */
    fileAttributes?: string;
    /** Creation time for the file/directory. Default value: Now. */
    fileCreatedOn?: string;
    /** Last write time for the file/directory. Default value: Now. */
    fileLastWriteOn?: string;
    /** Change time for the file/directory. Default value: Now. */
    fileChangeOn?: string;
    /** Optional, NFS only. The owner of the file or directory. */
    owner?: string;
    /** Optional, NFS only. The owning group of the file or directory. */
    group?: string;
    /** Optional, NFS only. The file mode of the file or directory */
    fileMode?: string;
    /** Resizes a file to the specified size. If the specified byte value is less than the current size of the file, then all ranges above the specified byte value are cleared. */
    fileContentLength?: number;
}
/** Contains response data for the setHttpHeaders operation. */
export type FileSetHttpHeadersResponse = FileSetHttpHeadersHeaders;
/** Optional parameters. */
export interface FileSetMetadataOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    leaseAccessConditions?: LeaseAccessConditions;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** A name-value pair to associate with a file storage object. */
    metadata?: {
        [propertyName: string]: string;
    };
    /** If true, the trailing dot will not be trimmed from the target URI. */
    allowTrailingDot?: boolean;
}
/** Contains response data for the setMetadata operation. */
export type FileSetMetadataResponse = FileSetMetadataHeaders;
/** Optional parameters. */
export interface FileAcquireLeaseOptionalParams extends coreClient.OperationOptions {
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** Specifies the duration of the lease, in seconds, or negative one (-1) for a lease that never expires. A non-infinite lease can be between 15 and 60 seconds. A lease duration cannot be changed using renew or change. */
    duration?: number;
    /** Proposed lease ID, in a GUID string format. The File service returns 400 (Invalid request) if the proposed lease ID is not in the correct format. See Guid Constructor (String) for a list of valid GUID string formats. */
    proposedLeaseId?: string;
    /** Provides a client-generated, opaque value with a 1 KB character limit that is recorded in the analytics logs when storage analytics logging is enabled. */
    requestId?: string;
    /** If true, the trailing dot will not be trimmed from the target URI. */
    allowTrailingDot?: boolean;
}
/** Contains response data for the acquireLease operation. */
export type FileAcquireLeaseResponse = FileAcquireLeaseHeaders;
/** Optional parameters. */
export interface FileReleaseLeaseOptionalParams extends coreClient.OperationOptions {
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** Provides a client-generated, opaque value with a 1 KB character limit that is recorded in the analytics logs when storage analytics logging is enabled. */
    requestId?: string;
    /** If true, the trailing dot will not be trimmed from the target URI. */
    allowTrailingDot?: boolean;
}
/** Contains response data for the releaseLease operation. */
export type FileReleaseLeaseResponse = FileReleaseLeaseHeaders;
/** Optional parameters. */
export interface FileChangeLeaseOptionalParams extends coreClient.OperationOptions {
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** Proposed lease ID, in a GUID string format. The File service returns 400 (Invalid request) if the proposed lease ID is not in the correct format. See Guid Constructor (String) for a list of valid GUID string formats. */
    proposedLeaseId?: string;
    /** Provides a client-generated, opaque value with a 1 KB character limit that is recorded in the analytics logs when storage analytics logging is enabled. */
    requestId?: string;
    /** If true, the trailing dot will not be trimmed from the target URI. */
    allowTrailingDot?: boolean;
}
/** Contains response data for the changeLease operation. */
export type FileChangeLeaseResponse = FileChangeLeaseHeaders;
/** Optional parameters. */
export interface FileBreakLeaseOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    leaseAccessConditions?: LeaseAccessConditions;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** Provides a client-generated, opaque value with a 1 KB character limit that is recorded in the analytics logs when storage analytics logging is enabled. */
    requestId?: string;
    /** If true, the trailing dot will not be trimmed from the target URI. */
    allowTrailingDot?: boolean;
}
/** Contains response data for the breakLease operation. */
export type FileBreakLeaseResponse = FileBreakLeaseHeaders;
/** Optional parameters. */
export interface FileUploadRangeOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    leaseAccessConditions?: LeaseAccessConditions;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** If true, the trailing dot will not be trimmed from the target URI. */
    allowTrailingDot?: boolean;
    /** Initial data. */
    body?: coreRestPipeline.RequestBodyType;
    /** An MD5 hash of the content. This hash is used to verify the integrity of the data during transport. When the Content-MD5 header is specified, the File service compares the hash of the content that has arrived with the header value that was sent. If the two hashes do not match, the operation will fail with error code 400 (Bad Request). */
    contentMD5?: Uint8Array;
    /** If the file last write time should be preserved or overwritten */
    fileLastWrittenMode?: FileLastWrittenMode;
}
/** Contains response data for the uploadRange operation. */
export type FileUploadRangeResponse = FileUploadRangeHeaders;
/** Optional parameters. */
export interface FileUploadRangeFromURLOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    leaseAccessConditions?: LeaseAccessConditions;
    /** Parameter group */
    sourceModifiedAccessConditions?: SourceModifiedAccessConditions;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** If true, the trailing dot will not be trimmed from the target URI. */
    allowTrailingDot?: boolean;
    /** If true, the trailing dot will not be trimmed from the source URI. */
    allowSourceTrailingDot?: boolean;
    /** If the file last write time should be preserved or overwritten */
    fileLastWrittenMode?: FileLastWrittenMode;
    /** Bytes of source data in the specified range. */
    sourceRange?: string;
    /** Specify the crc64 calculated for the range of bytes that must be read from the copy source. */
    sourceContentCrc64?: Uint8Array;
    /** Only Bearer type is supported. Credentials should be a valid OAuth access token to copy source. */
    copySourceAuthorization?: string;
}
/** Contains response data for the uploadRangeFromURL operation. */
export type FileUploadRangeFromURLResponse = FileUploadRangeFromURLHeaders;
/** Optional parameters. */
export interface FileGetRangeListOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    leaseAccessConditions?: LeaseAccessConditions;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** The snapshot parameter is an opaque DateTime value that, when present, specifies the share snapshot to query. */
    shareSnapshot?: string;
    /** If true, the trailing dot will not be trimmed from the target URI. */
    allowTrailingDot?: boolean;
    /** Specifies the range of bytes over which to list ranges, inclusively. */
    range?: string;
    /** The previous snapshot parameter is an opaque DateTime value that, when present, specifies the previous snapshot. */
    prevsharesnapshot?: string;
    /** This header is allowed only when PrevShareSnapshot query parameter is set. Determines whether the changed ranges for a file that has been renamed or moved between the target snapshot (or the live file) and the previous snapshot should be listed. If the value is true, the valid changed ranges for the file will be returned. If the value is false, the operation will result in a failure with 409 (Conflict) response. The default value is false. */
    supportRename?: boolean;
}
/** Contains response data for the getRangeList operation. */
export type FileGetRangeListResponse = FileGetRangeListHeaders & ShareFileRangeList;
/** Optional parameters. */
export interface FileStartCopyOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    leaseAccessConditions?: LeaseAccessConditions;
    /** Parameter group */
    copyFileSmbInfo?: CopyFileSmbInfo;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** A name-value pair to associate with a file storage object. */
    metadata?: {
        [propertyName: string]: string;
    };
    /** Optional. Available for version 2023-06-01 and later. Specifies the format in which the permission is returned. Acceptable values are SDDL or binary. If x-ms-file-permission-format is unspecified or explicitly set to SDDL, the permission is returned in SDDL format. If x-ms-file-permission-format is explicitly set to binary, the permission is returned as a base64 string representing the binary encoding of the permission */
    filePermissionFormat?: FilePermissionFormat;
    /** If true, the trailing dot will not be trimmed from the target URI. */
    allowTrailingDot?: boolean;
    /** If specified the permission (security descriptor) shall be set for the directory/file. This header can be used if Permission size is <= 8KB, else x-ms-file-permission-key header shall be used. Default value: Inherit. If SDDL is specified as input, it must have owner, group and dacl. Note: Only one of the x-ms-file-permission or x-ms-file-permission-key should be specified. */
    filePermission?: string;
    /** Key of the permission to be set for the directory/file. Note: Only one of the x-ms-file-permission or x-ms-file-permission-key should be specified. */
    filePermissionKey?: string;
    /** Optional, NFS only. The owner of the file or directory. */
    owner?: string;
    /** Optional, NFS only. The owning group of the file or directory. */
    group?: string;
    /** Optional, NFS only. The file mode of the file or directory */
    fileMode?: string;
    /** If true, the trailing dot will not be trimmed from the source URI. */
    allowSourceTrailingDot?: boolean;
    /** NFS only. Applicable only when the copy source is a File. Determines the copy behavior of the mode bits of the file. source: The mode on the destination file is copied from the source file. override: The mode on the destination file is determined via the x-ms-mode header. */
    fileModeCopyMode?: ModeCopyMode;
    /** NFS only. Determines the copy behavior of the owner user identifier (UID) and group identifier (GID) of the file. source: The owner user identifier (UID) and group identifier (GID) on the destination file is copied from the source file. override: The owner user identifier (UID) and group identifier (GID) on the destination file is determined via the x-ms-owner and x-ms-group  headers. */
    fileOwnerCopyMode?: OwnerCopyMode;
}
/** Contains response data for the startCopy operation. */
export type FileStartCopyResponse = FileStartCopyHeaders;
/** Optional parameters. */
export interface FileAbortCopyOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    leaseAccessConditions?: LeaseAccessConditions;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** If true, the trailing dot will not be trimmed from the target URI. */
    allowTrailingDot?: boolean;
}
/** Contains response data for the abortCopy operation. */
export type FileAbortCopyResponse = FileAbortCopyHeaders;
/** Optional parameters. */
export interface FileListHandlesOptionalParams extends coreClient.OperationOptions {
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** A string value that identifies the portion of the list to be returned with the next list operation. The operation returns a marker value within the response body if the list returned was not complete. The marker value may then be used in a subsequent call to request the next set of list items. The marker value is opaque to the client. */
    marker?: string;
    /** Specifies the maximum number of entries to return. If the request does not specify maxresults, or specifies a value greater than 5,000, the server will return up to 5,000 items. */
    maxResults?: number;
    /** The snapshot parameter is an opaque DateTime value that, when present, specifies the share snapshot to query. */
    shareSnapshot?: string;
    /** If true, the trailing dot will not be trimmed from the target URI. */
    allowTrailingDot?: boolean;
}
/** Contains response data for the listHandles operation. */
export type FileListHandlesResponse = FileListHandlesHeaders & ListHandlesResponse;
/** Optional parameters. */
export interface FileForceCloseHandlesOptionalParams extends coreClient.OperationOptions {
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** A string value that identifies the portion of the list to be returned with the next list operation. The operation returns a marker value within the response body if the list returned was not complete. The marker value may then be used in a subsequent call to request the next set of list items. The marker value is opaque to the client. */
    marker?: string;
    /** The snapshot parameter is an opaque DateTime value that, when present, specifies the share snapshot to query. */
    shareSnapshot?: string;
    /** If true, the trailing dot will not be trimmed from the target URI. */
    allowTrailingDot?: boolean;
}
/** Contains response data for the forceCloseHandles operation. */
export type FileForceCloseHandlesResponse = FileForceCloseHandlesHeaders;
/** Optional parameters. */
export interface FileRenameOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    sourceLeaseAccessConditions?: SourceLeaseAccessConditions;
    /** Parameter group */
    destinationLeaseAccessConditions?: DestinationLeaseAccessConditions;
    /** Parameter group */
    copyFileSmbInfo?: CopyFileSmbInfo;
    /** Parameter group */
    fileHttpHeaders?: FileHttpHeaders;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** A name-value pair to associate with a file storage object. */
    metadata?: {
        [propertyName: string]: string;
    };
    /** Optional. Available for version 2023-06-01 and later. Specifies the format in which the permission is returned. Acceptable values are SDDL or binary. If x-ms-file-permission-format is unspecified or explicitly set to SDDL, the permission is returned in SDDL format. If x-ms-file-permission-format is explicitly set to binary, the permission is returned as a base64 string representing the binary encoding of the permission */
    filePermissionFormat?: FilePermissionFormat;
    /** If true, the trailing dot will not be trimmed from the target URI. */
    allowTrailingDot?: boolean;
    /** If specified the permission (security descriptor) shall be set for the directory/file. This header can be used if Permission size is <= 8KB, else x-ms-file-permission-key header shall be used. Default value: Inherit. If SDDL is specified as input, it must have owner, group and dacl. Note: Only one of the x-ms-file-permission or x-ms-file-permission-key should be specified. */
    filePermission?: string;
    /** Key of the permission to be set for the directory/file. Note: Only one of the x-ms-file-permission or x-ms-file-permission-key should be specified. */
    filePermissionKey?: string;
    /** Optional. A boolean value for if the destination file already exists, whether this request will overwrite the file or not. If true, the rename will succeed and will overwrite the destination file. If not provided or if false and the destination file does exist, the request will not overwrite the destination file. If provided and the destination file doesn’t exist, the rename will succeed. Note: This value does not override the x-ms-file-copy-ignore-read-only header value. */
    replaceIfExists?: boolean;
    /** Optional. A boolean value that specifies whether the ReadOnly attribute on a preexisting destination file should be respected. If true, the rename will succeed, otherwise, a previous file at the destination with the ReadOnly attribute set will cause the rename to fail. */
    ignoreReadOnly?: boolean;
    /** If true, the trailing dot will not be trimmed from the source URI. */
    allowSourceTrailingDot?: boolean;
}
/** Contains response data for the rename operation. */
export type FileRenameResponse = FileRenameHeaders;
/** Optional parameters. */
export interface FileCreateSymbolicLinkOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    leaseAccessConditions?: LeaseAccessConditions;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** A name-value pair to associate with a file storage object. */
    metadata?: {
        [propertyName: string]: string;
    };
    /** Provides a client-generated, opaque value with a 1 KB character limit that is recorded in the analytics logs when storage analytics logging is enabled. */
    requestId?: string;
    /** Creation time for the file/directory. Default value: Now. */
    fileCreatedOn?: string;
    /** Last write time for the file/directory. Default value: Now. */
    fileLastWriteOn?: string;
    /** Optional, NFS only. The owner of the file or directory. */
    owner?: string;
    /** Optional, NFS only. The owning group of the file or directory. */
    group?: string;
}
/** Contains response data for the createSymbolicLink operation. */
export type FileCreateSymbolicLinkResponse = FileCreateSymbolicLinkHeaders;
/** Optional parameters. */
export interface FileGetSymbolicLinkOptionalParams extends coreClient.OperationOptions {
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** The snapshot parameter is an opaque DateTime value that, when present, specifies the share snapshot to query. */
    shareSnapshot?: string;
    /** Provides a client-generated, opaque value with a 1 KB character limit that is recorded in the analytics logs when storage analytics logging is enabled. */
    requestId?: string;
}
/** Contains response data for the getSymbolicLink operation. */
export type FileGetSymbolicLinkResponse = FileGetSymbolicLinkHeaders;
/** Optional parameters. */
export interface FileCreateHardLinkOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    leaseAccessConditions?: LeaseAccessConditions;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://learn.microsoft.com/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations">Setting Timeouts for File Service Operations.</a> */
    timeoutInSeconds?: number;
    /** Valid value is backup */
    fileRequestIntent?: ShareTokenIntent;
    /** Provides a client-generated, opaque value with a 1 KB character limit that is recorded in the analytics logs when storage analytics logging is enabled. */
    requestId?: string;
}
/** Contains response data for the createHardLink operation. */
export type FileCreateHardLinkResponse = FileCreateHardLinkHeaders;
/** Optional parameters. */
export interface StorageClientOptionalParams extends coreHttpCompat.ExtendedServiceClientOptions {
    /** Specifies the version of the operation to use for this request. */
    version?: string;
    /** Only update is supported: - Update: Writes the bytes downloaded from the source url into the specified range. */
    fileRangeWriteFromUrl?: string;
    /** Overrides client endpoint. */
    endpoint?: string;
}
//# sourceMappingURL=index.d.ts.map