var async = require('async');

var helpers = require('../../../helpers/azure/');

module.exports = {
    title: 'VMSS Windows AntiMalware Extension',
    category: 'Virtual Machine Scale Set',
    domain: 'Compute',
    severity: 'Medium',
    description: 'Ensures that Virtual Machine Scale Set windows instances have IaaS AntiMalware extension installed',
    more_info: 'The VM Scale Set Windows AntiMalware Extension provides real-time protection against viruses, spyware, and other malicious software for virtual machine scale sets running on the Windows operating system.',
    recommended_action: 'Modify Virtual Machine Scale Set and install IaaS AntiMalware extension.',
    link: 'https://learn.microsoft.com/en-us/azure/virtual-machines/extensions/iaas-antimalware-windows',
    apis: ['virtualMachineScaleSets:listAll'],
    realtime_triggers: ['microsoftcompute:virtualmachinescalesets:write', 'microsoftcompute:virtualmachinescalesets:delete', 'microsoftcompute:virtualmachinescalesets:extensions:write', 'microsoftcompute:virtualmachinescalesets:extensions:delete'],

    run: function(cache, settings, callback) {
        var results = [];
        var source = {};
        var locations = helpers.locations(settings.govcloud);

        async.each(locations.virtualMachineScaleSets, function(location, rcb) {
            var virtualMachineScaleSets = helpers.addSource(cache, source,
                ['virtualMachineScaleSets', 'listAll', location]);

            if (!virtualMachineScaleSets) return rcb();

            if (virtualMachineScaleSets.err || !virtualMachineScaleSets.data) {
                helpers.addResult(results, 3, 'Unable to query for Virtual Machine Scale Sets: ' + helpers.addError(virtualMachineScaleSets), location);
                return rcb();
            }

            if (!virtualMachineScaleSets.data.length) {
                helpers.addResult(results, 0, 'No existing Virtual Machine Scale Sets found', location);
                return rcb();
            }

            for (let scaleSet of virtualMachineScaleSets.data) {
                if (!scaleSet.id ||
                    (scaleSet.virtualMachineProfile &&
                        scaleSet.virtualMachineProfile.storageProfile &&
                        scaleSet.virtualMachineProfile.storageProfile.osDisk &&
                        scaleSet.virtualMachineProfile.storageProfile.osDisk.osType &&
                        scaleSet.virtualMachineProfile.storageProfile.osDisk.osType.toLowerCase() != 'windows')) continue;

                const scaleSetExtensions = scaleSet.virtualMachineProfile &&
                    scaleSet.virtualMachineProfile.extensionProfile &&
                    scaleSet.virtualMachineProfile.extensionProfile.extensions ?
                    scaleSet.virtualMachineProfile.extensionProfile.extensions : [];

                if (!scaleSetExtensions.length) {
                    helpers.addResult(results, 2, 'No VMSS Extensions found', location, scaleSet.id);
                } else {
                    let found = scaleSetExtensions.find(vmExt => vmExt.name && vmExt.name.toLowerCase() === 'iaasantimalware');
                    if (found) {
                        helpers.addResult(results, 0, 'Windows Virtual Machine Scale Set has IaaS Antimalware extension installed', location, scaleSet.id);
                    } else {
                        helpers.addResult(results, 2, 'Windows Virtual Machine Scale Set does not have IaaS Antimalware extension installed', location, scaleSet.id);
                    }
                }
            }

            rcb();
        }, function() {
            callback(null, results, source);
        });
    }
};