﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Runtime.InteropServices;
using System.Threading;

namespace Debugger
{
    public delegate void Callback(DebuggerMessage Value);

    public class NativeDebugger
    {
        const string DllPath = @"DebuggerComponents.dll";

        public event EventHandler<ExceptionEventArgs> ExceptionCaught;

        private int _id;

        public bool Running { get; private set; }        

        public NativeDebugger()
        {
            _id = Environment.TickCount;
        }

        #region Function Imports
        [DllImport(DllPath)]
        private static extern void StartProcess(string Path,
            [MarshalAs(UnmanagedType.FunctionPtr)]Callback Callback,
            int Id);

        [DllImport(DllPath)]
        private static extern void Attach(int ProcessId,
            [MarshalAs(UnmanagedType.FunctionPtr)]Callback Callback,
            int Id);

        [DllImport(DllPath)]
        private static extern void Stop(int Id);
        #endregion

        private void RaiseExceptionCaught(ExceptionInfo Exception)
        {
            var handle = ExceptionCaught;

            if (handle != null)
                handle(this, new ExceptionEventArgs(Exception));
        }

        private void SetRunning(bool Value)
        {
            if (Running == Value)
                throw new InvalidOperationException("The debugger is " + 
                    (Value ? "already" : "not") + "  running.");

            Running = Value;
        }

        private void DebugCallback(DebuggerMessage Message)
        {
            if (Message.MessageType == DebugMessageType.Exception)
            {
                RaiseExceptionCaught(Message.Exception);
            }
        }

        public void StartProcess(string Path)
        {
            SetRunning(true);

            ThreadPool.QueueUserWorkItem(x => StartProcess(Path, DebugCallback, _id));
        }

        public void Attach(int ProcessId)
        {
            SetRunning(true);

            ThreadPool.QueueUserWorkItem(x => Attach(ProcessId, DebugCallback, _id));
        }

        public void Stop()
        {
            SetRunning(false);

            Stop(_id);

            Running = false;
        }
    }
}
