// tacacs.hpp

#ifndef TACACS_HPP
#define TACACS_HPP

#include "datum.h"
#include "protocol.h"
#include "json_object.h"
#include "match.h"

namespace tacacs {

#include "tacacs_plus_params.hpp"

    //  5.1. The Authentication START Packet Body
    //
    //  1 2 3 4 5 6 7 8  1 2 3 4 5 6 7 8  1 2 3 4 5 6 7 8  1 2 3 4 5 6 7 8
    // +----------------+----------------+----------------+----------------+
    // |    action      |    priv_lvl    |  authen_type   | authen_service |
    // +----------------+----------------+----------------+----------------+
    // |    user_len    |    port_len    |  rem_addr_len  |    data_len    |
    // +----------------+----------------+----------------+----------------+
    // |    user ...
    // +----------------+----------------+----------------+----------------+
    // |    port ...
    // +----------------+----------------+----------------+----------------+
    // |    rem_addr ...
    // +----------------+----------------+----------------+----------------+
    // |    data...
    // +----------------+----------------+----------------+----------------+
    //
    class authentication_start {
        authen_action<uint8_t> action;
        privilege_level<uint8_t> priv_lvl;
        authen_type<uint8_t> type;
        authen_service<uint8_t> service;
        encoded<uint8_t> user_len;
        encoded<uint8_t> port_len;
        encoded<uint8_t> rem_addr_len;
        encoded<uint8_t> data_len;
        datum user;
        datum port;
        datum rem_addr;
        datum data;

    public:

        authentication_start(datum &d) :
            action{d},
            priv_lvl{d},
            type{d},
            service{d},
            user_len{d},
            port_len{d},
            rem_addr_len{d},
            data_len{d},
            user{d, user_len},
            port{d, port_len},
            rem_addr{d, rem_addr_len},
            data{d, data_len}
        { }

        bool is_valid() const { return data.is_not_null(); }

        void write_json(json_object &o, bool metadata=false) const {
            (void)metadata;
            if (!is_valid()) {
                return;
            }
            json_object auth_start_json{o, "authentication_start"};
            action.write_json(auth_start_json);
            priv_lvl.write_json(auth_start_json);
            type.write_json(auth_start_json);
            service.write_json(auth_start_json);
            auth_start_json.print_key_json_string("user", user);
            auth_start_json.print_key_json_string("port", port);
            auth_start_json.print_key_json_string("rem_addr", rem_addr);
            auth_start_json.print_key_hex("data", data);
            auth_start_json.close();
        }
    };

    // The TACACS+ server authentication reply packet format:
    //
    //  1 2 3 4 5 6 7 8  1 2 3 4 5 6 7 8  1 2 3 4 5 6 7 8  1 2 3 4 5 6 7 8
    // +----------------+----------------+----------------+----------------+
    // |     status     |      flags     |        server_msg_len           |
    // +----------------+----------------+----------------+----------------+
    // |           data_len              |        server_msg ...
    // +----------------+----------------+----------------+----------------+
    // |           data ...
    // +----------------+----------------+
    //
    class auth_reply {
        authen_status<uint8_t> status;
        encoded<uint8_t> flags;
        encoded<uint16_t> server_msg_len;
        encoded<uint16_t> data_len;
        datum server_msg;
        datum data;
        bool valid;

    public:

        auth_reply(datum &d) :
            status{d},
            flags{d},
            server_msg_len{d},
            data_len{d},
            server_msg{d, server_msg_len.value()},
            data{d, data_len.value()},
            valid{d.is_not_null()}
        { }

        void write_json(json_object &o, bool metadata=false) const {
            (void)metadata;
            if (!valid) { return; }
            json_object json{o, "authentication_reply"};
            status.write_json(json);

            json_array_bitflags flags_array{json, "flags", flags};
            flags_array.flag<7>("noecho");
            flags_array.check_for_unknown_flags<7>();
            flags_array.close();

            json.print_key_json_string("server_msg", server_msg);
            json.print_key_hex("data", data);
            json.close();
        }

    };

    // 5.3. The Authentication CONTINUE Packet Body
    //
    // This packet is sent from the client to the server following the receipt of a REPLY packet.
    //
    //  1 2 3 4 5 6 7 8  1 2 3 4 5 6 7 8  1 2 3 4 5 6 7 8  1 2 3 4 5 6 7 8
    // +----------------+----------------+----------------+----------------+
    // |          user_msg len           |            data_len             |
    // +----------------+----------------+----------------+----------------+
    // |     flags      |  user_msg ...
    // +----------------+----------------+----------------+----------------+
    // |    data ...
    // +----------------+
    //
    class auth_continue {
        encoded<uint16_t> user_msg_len;
        encoded<uint16_t> data_len;
        encoded<uint8_t> flags;
        datum user_msg;
        datum data;
        bool valid;

    public:

        auth_continue(datum &d) :
            user_msg_len{d},
            data_len{d},
            flags{d},
            user_msg{d, user_msg_len.value()},
            data{d, data_len.value()},
            valid{d.is_not_null()}
        {}

        void write_json(json_object &o, bool metadata=false) const {
            (void)metadata;
            if (!valid) { return; }
            json_object json{o, "authentication_continue"};

            json_array_bitflags flags_array{json, "flags", flags};
            flags_array.flag<7>("abort");
            flags_array.check_for_unknown_flags<7>();
            flags_array.close();

            json.print_key_json_string("user", user_msg);
            json.print_key_hex("data", data);
            json.close();
        }

    };

    // 4.1. The TACACS+ Packet Header
    //
    // All TACACS+ packets begin with the following 12-byte header. The header describes the remainder of the packet:
    //
    //  1 2 3 4 5 6 7 8  1 2 3 4 5 6 7 8  1 2 3 4 5 6 7 8  1 2 3 4 5 6 7 8
    // +----------------+----------------+----------------+----------------+
    // |major  | minor  |                |                |                |
    // |version| version|      type      |     seq_no     |   flags        |
    // +----------------+----------------+----------------+----------------+
    // |                                                                   |
    // |                            session_id                             |
    // +----------------+----------------+----------------+----------------+
    // |                                                                   |
    // |                              length                               |
    // +----------------+----------------+----------------+----------------+
    //
    class packet : public base_protocol {
        encoded<uint8_t> version;
        encoded<uint8_t> type;
        encoded<uint8_t> seq_no;
        encoded<uint8_t> flags;
        encoded<uint32_t> session_id;
        encoded<uint32_t> length;
        datum body;

    public:

        packet(datum &d) :
            version{d},
            type{d},
            seq_no{d},
            flags{d},
            session_id{d},
            length{d},
            body{d}
        { }

        bool is_not_empty() const {
            //
            // consistency checks, as per RFC 8907:
            //
            //    - accept versions 12.0 and 12.1
            //    - accept type = 0, 1, 2, or 3
            //    - require that body.lenth() == length.value()
            //
            return ((version.value() & 0xfe) == 0xc0) and ((type.value() & 0xfc) == 0x00) and body.length() == length.value();
        }

        enum msg_type { request, reply };

        enum msg_type direction() const {
            if (seq_no.value() & 1) {
                return msg_type::request;
            }
            return msg_type::reply;
        }

        void write_json(json_object &o, bool) const {
            if (!is_not_empty()) {
                return;
            }
            json_object tacacs_json{o, "tacacs_plus"};
            tacacs_json.print_key_uint("major_version", version.slice<0,4>());
            tacacs_json.print_key_uint("minor_version", version.slice<4,8>());
            print_type_code(tacacs_json);
            tacacs_json.print_key_uint("seq_no", seq_no);

            json_array_bitflags flags_array{tacacs_json, "flags", flags};
            flags_array.flag<5>("single_connect");
            flags_array.flag<7>("unencrypted");
            flags_array.check_for_unknown_flags<5,7>();
            flags_array.close();

            tacacs_json.print_key_uint_hex("session_id", session_id.value());

            if (flags.bit<7>()) {             // unencrypted
                if (type.value() == 0x01) {
                    if (direction() == msg_type::request) {
                        if (seq_no.value() == 1) {
                            if (lookahead<authentication_start> as{body}) {
                                as.value.write_json(tacacs_json);
                            }
                        } else {
                            if (lookahead<auth_continue> ac{body}) {
                                ac.value.write_json(tacacs_json);
                            }
                        }
                    } else {
                        if (lookahead<auth_reply> ar{body}) {
                            ar.value.write_json(tacacs_json);
                        }
                    }
                } else {
                    //
                    // message body is unencrypted
                    //
                    const char *message_type = direction() == msg_type::request ? "unencrypted_request" : "unencrypted_reply";
                    tacacs_json.print_key_hex(message_type, body);
                }
            } else {
                //
                // message body is encrypted
                //
                const char *message_type = direction() == msg_type::request ? "encrypted_request" : "encrypted_reply";
                tacacs_json.print_key_hex(message_type, body);
            }
            tacacs_json.close();
        }

        void write_l7_metadata(cbor_object &o, bool) {
            cbor_array protocols{o, "protocols"};
            protocols.print_string("tacacs");
            protocols.close();
        }

        void print_type_code(json_object &o) const {
            const char *result = nullptr;
            switch(type.value()) {
            case 0x01:
                result = "authentication";  // TAC_PLUS_AUTHEN := 0x01 (Authentication)
                break;
            case 0x02:
                result = "authorization";   // TAC_PLUS_AUTHOR := 0x02 (Authorization)
                break;
            case 0x03:
                result = "accounting";      // TAC_PLUS_ACCT := 0x03 (Accounting)
                break;
            default:
                ;
            }
            if (result) {
                o.print_key_string("type", result);
            } else {
                o.print_key_unknown_code("type", type);
            }
        }

    };

    [[maybe_unused]] static bool unit_test()  {

        FILE *output = nullptr;   // set this to obtain verbose output

        bool passed = true;

        uint8_t ref_dat_1[] = {
            0xc0, 0x01, 0x01, 0x01, 0x6d, 0x0e, 0x16, 0x31, 0x00, 0x00, 0x00, 0x11,
            0x01, 0x01, 0x01, 0x01, 0x00, 0x04, 0x05, 0x00, 0x74, 0x74, 0x79, 0x30,
            0x61, 0x73, 0x79, 0x6e, 0x63
        };
        uint8_t ref_out_1[] = {
            0x7b, 0x22, 0x74, 0x61, 0x63, 0x61, 0x63, 0x73, 0x5f, 0x70, 0x6c, 0x75,
            0x73, 0x22, 0x3a, 0x7b, 0x22, 0x6d, 0x61, 0x6a, 0x6f, 0x72, 0x5f, 0x76,
            0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x22, 0x3a, 0x31, 0x32, 0x2c, 0x22,
            0x6d, 0x69, 0x6e, 0x6f, 0x72, 0x5f, 0x76, 0x65, 0x72, 0x73, 0x69, 0x6f,
            0x6e, 0x22, 0x3a, 0x30, 0x2c, 0x22, 0x74, 0x79, 0x70, 0x65, 0x22, 0x3a,
            0x22, 0x61, 0x75, 0x74, 0x68, 0x65, 0x6e, 0x74, 0x69, 0x63, 0x61, 0x74,
            0x69, 0x6f, 0x6e, 0x22, 0x2c, 0x22, 0x73, 0x65, 0x71, 0x5f, 0x6e, 0x6f,
            0x22, 0x3a, 0x31, 0x2c, 0x22, 0x66, 0x6c, 0x61, 0x67, 0x73, 0x22, 0x3a,
            0x5b, 0x22, 0x75, 0x6e, 0x65, 0x6e, 0x63, 0x72, 0x79, 0x70, 0x74, 0x65,
            0x64, 0x22, 0x5d, 0x2c, 0x22, 0x73, 0x65, 0x73, 0x73, 0x69, 0x6f, 0x6e,
            0x5f, 0x69, 0x64, 0x22, 0x3a, 0x22, 0x36, 0x64, 0x30, 0x65, 0x31, 0x36,
            0x33, 0x31, 0x22, 0x2c, 0x22, 0x61, 0x75, 0x74, 0x68, 0x65, 0x6e, 0x74,
            0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x5f, 0x73, 0x74, 0x61, 0x72,
            0x74, 0x22, 0x3a, 0x7b, 0x22, 0x61, 0x75, 0x74, 0x68, 0x65, 0x6e, 0x5f,
            0x61, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x22, 0x3a, 0x22, 0x4c, 0x4f, 0x47,
            0x49, 0x4e, 0x22, 0x2c, 0x22, 0x70, 0x72, 0x69, 0x76, 0x69, 0x6c, 0x65,
            0x67, 0x65, 0x5f, 0x6c, 0x65, 0x76, 0x65, 0x6c, 0x22, 0x3a, 0x22, 0x55,
            0x53, 0x45, 0x52, 0x22, 0x2c, 0x22, 0x61, 0x75, 0x74, 0x68, 0x65, 0x6e,
            0x5f, 0x74, 0x79, 0x70, 0x65, 0x22, 0x3a, 0x22, 0x41, 0x53, 0x43, 0x49,
            0x49, 0x22, 0x2c, 0x22, 0x61, 0x75, 0x74, 0x68, 0x65, 0x6e, 0x5f, 0x73,
            0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x22, 0x3a, 0x22, 0x4c, 0x4f, 0x47,
            0x49, 0x4e, 0x22, 0x2c, 0x22, 0x70, 0x6f, 0x72, 0x74, 0x22, 0x3a, 0x22,
            0x74, 0x74, 0x79, 0x30, 0x22, 0x2c, 0x22, 0x72, 0x65, 0x6d, 0x5f, 0x61,
            0x64, 0x64, 0x72, 0x22, 0x3a, 0x22, 0x61, 0x73, 0x79, 0x6e, 0x63, 0x22,
            0x2c, 0x22, 0x64, 0x61, 0x74, 0x61, 0x22, 0x3a, 0x22, 0x22, 0x7d, 0x7d,
            0x7d
        };
        passed &= test_json_output<tacacs::packet>(ref_dat_1, sizeof(ref_dat_1), ref_out_1, sizeof(ref_out_1), output);

        uint8_t ref_dat_2[] = {
            0xc0, 0x01, 0x02, 0x01, 0x6d, 0x0e, 0x16, 0x31, 0x00, 0x00, 0x00, 0x2b,
            0x04, 0x00, 0x00, 0x25, 0x00, 0x00, 0x0a, 0x55, 0x73, 0x65, 0x72, 0x20,
            0x41, 0x63, 0x63, 0x65, 0x73, 0x73, 0x20, 0x56, 0x65, 0x72, 0x69, 0x66,
            0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x0a, 0x0a, 0x55, 0x73, 0x65,
            0x72, 0x6e, 0x61, 0x6d, 0x65, 0x3a, 0x20
        };
        uint8_t ref_out_2[] = {
            0x7b, 0x22, 0x74, 0x61, 0x63, 0x61, 0x63, 0x73, 0x5f, 0x70, 0x6c, 0x75,
            0x73, 0x22, 0x3a, 0x7b, 0x22, 0x6d, 0x61, 0x6a, 0x6f, 0x72, 0x5f, 0x76,
            0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x22, 0x3a, 0x31, 0x32, 0x2c, 0x22,
            0x6d, 0x69, 0x6e, 0x6f, 0x72, 0x5f, 0x76, 0x65, 0x72, 0x73, 0x69, 0x6f,
            0x6e, 0x22, 0x3a, 0x30, 0x2c, 0x22, 0x74, 0x79, 0x70, 0x65, 0x22, 0x3a,
            0x22, 0x61, 0x75, 0x74, 0x68, 0x65, 0x6e, 0x74, 0x69, 0x63, 0x61, 0x74,
            0x69, 0x6f, 0x6e, 0x22, 0x2c, 0x22, 0x73, 0x65, 0x71, 0x5f, 0x6e, 0x6f,
            0x22, 0x3a, 0x32, 0x2c, 0x22, 0x66, 0x6c, 0x61, 0x67, 0x73, 0x22, 0x3a,
            0x5b, 0x22, 0x75, 0x6e, 0x65, 0x6e, 0x63, 0x72, 0x79, 0x70, 0x74, 0x65,
            0x64, 0x22, 0x5d, 0x2c, 0x22, 0x73, 0x65, 0x73, 0x73, 0x69, 0x6f, 0x6e,
            0x5f, 0x69, 0x64, 0x22, 0x3a, 0x22, 0x36, 0x64, 0x30, 0x65, 0x31, 0x36,
            0x33, 0x31, 0x22, 0x2c, 0x22, 0x61, 0x75, 0x74, 0x68, 0x65, 0x6e, 0x74,
            0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x5f, 0x72, 0x65, 0x70, 0x6c,
            0x79, 0x22, 0x3a, 0x7b, 0x22, 0x61, 0x75, 0x74, 0x68, 0x65, 0x6e, 0x5f,
            0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x22, 0x3a, 0x22, 0x47, 0x45, 0x54,
            0x55, 0x53, 0x45, 0x52, 0x22, 0x2c, 0x22, 0x66, 0x6c, 0x61, 0x67, 0x73,
            0x22, 0x3a, 0x5b, 0x5d, 0x2c, 0x22, 0x73, 0x65, 0x72, 0x76, 0x65, 0x72,
            0x5f, 0x6d, 0x73, 0x67, 0x22, 0x3a, 0x22, 0x5c, 0x75, 0x30, 0x30, 0x30,
            0x61, 0x55, 0x73, 0x65, 0x72, 0x20, 0x41, 0x63, 0x63, 0x65, 0x73, 0x73,
            0x20, 0x56, 0x65, 0x72, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f,
            0x6e, 0x5c, 0x75, 0x30, 0x30, 0x30, 0x61, 0x5c, 0x75, 0x30, 0x30, 0x30,
            0x61, 0x55, 0x73, 0x65, 0x72, 0x6e, 0x61, 0x6d, 0x65, 0x3a, 0x20, 0x22,
            0x2c, 0x22, 0x64, 0x61, 0x74, 0x61, 0x22, 0x3a, 0x22, 0x22, 0x7d, 0x7d,
            0x7d
        };
        passed &= test_json_output<tacacs::packet>(ref_dat_2, sizeof(ref_dat_2), ref_out_2, sizeof(ref_out_2), output);

        uint8_t ref_dat_3[] = {
            0xc0, 0x01, 0x03, 0x01, 0x6d, 0x0e, 0x16, 0x31, 0x00, 0x00, 0x00, 0x0a,
            0x00, 0x05, 0x00, 0x00, 0x00, 0x61, 0x64, 0x6d, 0x69, 0x6e
        };
        uint8_t ref_out_3[] = {
            0x7b, 0x22, 0x74, 0x61, 0x63, 0x61, 0x63, 0x73, 0x5f, 0x70, 0x6c, 0x75,
            0x73, 0x22, 0x3a, 0x7b, 0x22, 0x6d, 0x61, 0x6a, 0x6f, 0x72, 0x5f, 0x76,
            0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x22, 0x3a, 0x31, 0x32, 0x2c, 0x22,
            0x6d, 0x69, 0x6e, 0x6f, 0x72, 0x5f, 0x76, 0x65, 0x72, 0x73, 0x69, 0x6f,
            0x6e, 0x22, 0x3a, 0x30, 0x2c, 0x22, 0x74, 0x79, 0x70, 0x65, 0x22, 0x3a,
            0x22, 0x61, 0x75, 0x74, 0x68, 0x65, 0x6e, 0x74, 0x69, 0x63, 0x61, 0x74,
            0x69, 0x6f, 0x6e, 0x22, 0x2c, 0x22, 0x73, 0x65, 0x71, 0x5f, 0x6e, 0x6f,
            0x22, 0x3a, 0x33, 0x2c, 0x22, 0x66, 0x6c, 0x61, 0x67, 0x73, 0x22, 0x3a,
            0x5b, 0x22, 0x75, 0x6e, 0x65, 0x6e, 0x63, 0x72, 0x79, 0x70, 0x74, 0x65,
            0x64, 0x22, 0x5d, 0x2c, 0x22, 0x73, 0x65, 0x73, 0x73, 0x69, 0x6f, 0x6e,
            0x5f, 0x69, 0x64, 0x22, 0x3a, 0x22, 0x36, 0x64, 0x30, 0x65, 0x31, 0x36,
            0x33, 0x31, 0x22, 0x2c, 0x22, 0x61, 0x75, 0x74, 0x68, 0x65, 0x6e, 0x74,
            0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x5f, 0x63, 0x6f, 0x6e, 0x74,
            0x69, 0x6e, 0x75, 0x65, 0x22, 0x3a, 0x7b, 0x22, 0x66, 0x6c, 0x61, 0x67,
            0x73, 0x22, 0x3a, 0x5b, 0x5d, 0x2c, 0x22, 0x75, 0x73, 0x65, 0x72, 0x22,
            0x3a, 0x22, 0x61, 0x64, 0x6d, 0x69, 0x6e, 0x22, 0x2c, 0x22, 0x64, 0x61,
            0x74, 0x61, 0x22, 0x3a, 0x22, 0x22, 0x7d, 0x7d, 0x7d
        };
        passed &= test_json_output<tacacs::packet>(ref_dat_3, sizeof(ref_dat_3), ref_out_3, sizeof(ref_out_3), output);

        uint8_t ref_dat_4[] = {
            0xc0, 0x01, 0x04, 0x01, 0x6d, 0x0e, 0x16, 0x31, 0x00, 0x00, 0x00, 0x10,
            0x05, 0x01, 0x00, 0x0a, 0x00, 0x00, 0x50, 0x61, 0x73, 0x73, 0x77, 0x6f,
            0x72, 0x64, 0x3a, 0x20
        };
        uint8_t ref_out_4[] = {
            0x7b, 0x22, 0x74, 0x61, 0x63, 0x61, 0x63, 0x73, 0x5f, 0x70, 0x6c, 0x75,
            0x73, 0x22, 0x3a, 0x7b, 0x22, 0x6d, 0x61, 0x6a, 0x6f, 0x72, 0x5f, 0x76,
            0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x22, 0x3a, 0x31, 0x32, 0x2c, 0x22,
            0x6d, 0x69, 0x6e, 0x6f, 0x72, 0x5f, 0x76, 0x65, 0x72, 0x73, 0x69, 0x6f,
            0x6e, 0x22, 0x3a, 0x30, 0x2c, 0x22, 0x74, 0x79, 0x70, 0x65, 0x22, 0x3a,
            0x22, 0x61, 0x75, 0x74, 0x68, 0x65, 0x6e, 0x74, 0x69, 0x63, 0x61, 0x74,
            0x69, 0x6f, 0x6e, 0x22, 0x2c, 0x22, 0x73, 0x65, 0x71, 0x5f, 0x6e, 0x6f,
            0x22, 0x3a, 0x34, 0x2c, 0x22, 0x66, 0x6c, 0x61, 0x67, 0x73, 0x22, 0x3a,
            0x5b, 0x22, 0x75, 0x6e, 0x65, 0x6e, 0x63, 0x72, 0x79, 0x70, 0x74, 0x65,
            0x64, 0x22, 0x5d, 0x2c, 0x22, 0x73, 0x65, 0x73, 0x73, 0x69, 0x6f, 0x6e,
            0x5f, 0x69, 0x64, 0x22, 0x3a, 0x22, 0x36, 0x64, 0x30, 0x65, 0x31, 0x36,
            0x33, 0x31, 0x22, 0x2c, 0x22, 0x61, 0x75, 0x74, 0x68, 0x65, 0x6e, 0x74,
            0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x5f, 0x72, 0x65, 0x70, 0x6c,
            0x79, 0x22, 0x3a, 0x7b, 0x22, 0x61, 0x75, 0x74, 0x68, 0x65, 0x6e, 0x5f,
            0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x22, 0x3a, 0x22, 0x47, 0x45, 0x54,
            0x50, 0x41, 0x53, 0x53, 0x22, 0x2c, 0x22, 0x66, 0x6c, 0x61, 0x67, 0x73,
            0x22, 0x3a, 0x5b, 0x22, 0x6e, 0x6f, 0x65, 0x63, 0x68, 0x6f, 0x22, 0x5d,
            0x2c, 0x22, 0x73, 0x65, 0x72, 0x76, 0x65, 0x72, 0x5f, 0x6d, 0x73, 0x67,
            0x22, 0x3a, 0x22, 0x50, 0x61, 0x73, 0x73, 0x77, 0x6f, 0x72, 0x64, 0x3a,
            0x20, 0x22, 0x2c, 0x22, 0x64, 0x61, 0x74, 0x61, 0x22, 0x3a, 0x22, 0x22,
            0x7d, 0x7d, 0x7d
        };
        passed &= test_json_output<tacacs::packet>(ref_dat_4, sizeof(ref_dat_4), ref_out_4, sizeof(ref_out_4), output);

        uint8_t ref_dat_5[] = {
            0xc0, 0x01, 0x05, 0x01, 0x6d, 0x0e, 0x16, 0x31, 0x00, 0x00, 0x00, 0x0a,
            0x00, 0x05, 0x00, 0x00, 0x00, 0x61, 0x64, 0x6d, 0x69, 0x6e

        };
        uint8_t ref_out_5[] = {
            0x7b, 0x22, 0x74, 0x61, 0x63, 0x61, 0x63, 0x73, 0x5f, 0x70, 0x6c, 0x75,
            0x73, 0x22, 0x3a, 0x7b, 0x22, 0x6d, 0x61, 0x6a, 0x6f, 0x72, 0x5f, 0x76,
            0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x22, 0x3a, 0x31, 0x32, 0x2c, 0x22,
            0x6d, 0x69, 0x6e, 0x6f, 0x72, 0x5f, 0x76, 0x65, 0x72, 0x73, 0x69, 0x6f,
            0x6e, 0x22, 0x3a, 0x30, 0x2c, 0x22, 0x74, 0x79, 0x70, 0x65, 0x22, 0x3a,
            0x22, 0x61, 0x75, 0x74, 0x68, 0x65, 0x6e, 0x74, 0x69, 0x63, 0x61, 0x74,
            0x69, 0x6f, 0x6e, 0x22, 0x2c, 0x22, 0x73, 0x65, 0x71, 0x5f, 0x6e, 0x6f,
            0x22, 0x3a, 0x35, 0x2c, 0x22, 0x66, 0x6c, 0x61, 0x67, 0x73, 0x22, 0x3a,
            0x5b, 0x22, 0x75, 0x6e, 0x65, 0x6e, 0x63, 0x72, 0x79, 0x70, 0x74, 0x65,
            0x64, 0x22, 0x5d, 0x2c, 0x22, 0x73, 0x65, 0x73, 0x73, 0x69, 0x6f, 0x6e,
            0x5f, 0x69, 0x64, 0x22, 0x3a, 0x22, 0x36, 0x64, 0x30, 0x65, 0x31, 0x36,
            0x33, 0x31, 0x22, 0x2c, 0x22, 0x61, 0x75, 0x74, 0x68, 0x65, 0x6e, 0x74,
            0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x5f, 0x63, 0x6f, 0x6e, 0x74,
            0x69, 0x6e, 0x75, 0x65, 0x22, 0x3a, 0x7b, 0x22, 0x66, 0x6c, 0x61, 0x67,
            0x73, 0x22, 0x3a, 0x5b, 0x5d, 0x2c, 0x22, 0x75, 0x73, 0x65, 0x72, 0x22,
            0x3a, 0x22, 0x61, 0x64, 0x6d, 0x69, 0x6e, 0x22, 0x2c, 0x22, 0x64, 0x61,
            0x74, 0x61, 0x22, 0x3a, 0x22, 0x22, 0x7d, 0x7d, 0x7d
        };
        passed &= test_json_output<tacacs::packet>(ref_dat_5, sizeof(ref_dat_5), ref_out_5, sizeof(ref_out_5), output);

        uint8_t ref_dat_6[] = {
            0xc0, 0x01, 0x06, 0x01, 0x6d, 0x0e, 0x16, 0x31, 0x00, 0x00, 0x00, 0x06,
            0x01, 0x00, 0x00, 0x00, 0x00, 0x00
        };
        uint8_t ref_out_6[] = {
            0x7b, 0x22, 0x74, 0x61, 0x63, 0x61, 0x63, 0x73, 0x5f, 0x70, 0x6c, 0x75,
            0x73, 0x22, 0x3a, 0x7b, 0x22, 0x6d, 0x61, 0x6a, 0x6f, 0x72, 0x5f, 0x76,
            0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x22, 0x3a, 0x31, 0x32, 0x2c, 0x22,
            0x6d, 0x69, 0x6e, 0x6f, 0x72, 0x5f, 0x76, 0x65, 0x72, 0x73, 0x69, 0x6f,
            0x6e, 0x22, 0x3a, 0x30, 0x2c, 0x22, 0x74, 0x79, 0x70, 0x65, 0x22, 0x3a,
            0x22, 0x61, 0x75, 0x74, 0x68, 0x65, 0x6e, 0x74, 0x69, 0x63, 0x61, 0x74,
            0x69, 0x6f, 0x6e, 0x22, 0x2c, 0x22, 0x73, 0x65, 0x71, 0x5f, 0x6e, 0x6f,
            0x22, 0x3a, 0x36, 0x2c, 0x22, 0x66, 0x6c, 0x61, 0x67, 0x73, 0x22, 0x3a,
            0x5b, 0x22, 0x75, 0x6e, 0x65, 0x6e, 0x63, 0x72, 0x79, 0x70, 0x74, 0x65,
            0x64, 0x22, 0x5d, 0x2c, 0x22, 0x73, 0x65, 0x73, 0x73, 0x69, 0x6f, 0x6e,
            0x5f, 0x69, 0x64, 0x22, 0x3a, 0x22, 0x36, 0x64, 0x30, 0x65, 0x31, 0x36,
            0x33, 0x31, 0x22, 0x2c, 0x22, 0x61, 0x75, 0x74, 0x68, 0x65, 0x6e, 0x74,
            0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x5f, 0x72, 0x65, 0x70, 0x6c,
            0x79, 0x22, 0x3a, 0x7b, 0x22, 0x61, 0x75, 0x74, 0x68, 0x65, 0x6e, 0x5f,
            0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x22, 0x3a, 0x22, 0x50, 0x41, 0x53,
            0x53, 0x22, 0x2c, 0x22, 0x66, 0x6c, 0x61, 0x67, 0x73, 0x22, 0x3a, 0x5b,
            0x5d, 0x2c, 0x22, 0x64, 0x61, 0x74, 0x61, 0x22, 0x3a, 0x22, 0x22, 0x7d,
            0x7d, 0x7d
        };
        passed &= test_json_output<tacacs::packet>(ref_dat_6, sizeof(ref_dat_6), ref_out_6, sizeof(ref_out_6), output);

        return passed;

    }

};

namespace {

    [[maybe_unused]] inline int tacacs_authentication_start_fuzz_test(const uint8_t *data, size_t size) {
        return json_output_fuzzer<tacacs::authentication_start>(data, size);
    }
    
    [[maybe_unused]] inline int tacacs_auth_reply_fuzz_test(const uint8_t *data, size_t size) {
        return json_output_fuzzer<tacacs::auth_reply>(data, size);
    }

    [[maybe_unused]] inline int tacacs_auth_continue_fuzz_test(const uint8_t *data, size_t size) {
        return json_output_fuzzer<tacacs::auth_continue>(data, size);
    }

    [[maybe_unused]] inline int tacacs_packet_fuzz_test(const uint8_t *data, size_t size) {
        return json_output_fuzzer<tacacs::packet>(data, size);
    }

};

#endif // TACACS_HPP
