#!/usr/bin/env bash
# PentestGPT Authentication Configuration
# Interactive setup for Claude Code authentication

set -e

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
BLUE='\033[0;34m'
PURPLE='\033[0;35m'
YELLOW='\033[0;33m'
NC='\033[0m'

# Get the directory where this script is located, then go up to project root
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
PROJECT_DIR="$(dirname "$SCRIPT_DIR")"
ENV_FILE="${PROJECT_DIR}/.env.auth"

echo -e "${PURPLE}"
cat << "EOF"
    ____             __            __  __________ ______
   / __ \___  ____  / /____  _____/ /_/ ____/ __ /_  __/
  / /_/ / _ \/ __ \/ __/ _ \/ ___/ __/ / __/ /_/ // /
 / ____/  __/ / / / /_/  __(__  ) /_/ /_/ / ____// /
/_/    \___/_/ /_/\__/\___/____/\__/\____/_/    /_/

    Authentication Configuration
EOF
echo -e "${NC}"

echo -e "${BLUE}Select authentication method:${NC}\n"
echo -e "  ${GREEN}[1]${NC} Claude Code Subscription ${YELLOW}(Recommended)${NC}"
echo -e "      Use your Claude Code subscription via 'claude login'"
echo ""
echo -e "  ${GREEN}[2]${NC} OpenRouter API Key"
echo -e "      Route requests through OpenRouter to use GPT-5.1, Gemini, etc."
echo ""
echo -e "  ${GREEN}[3]${NC} Anthropic API Key"
echo -e "      Use Anthropic's Claude directly with your API key"
echo ""
echo -e "  ${GREEN}[4]${NC} Local LLM (via LM Studio, Ollama, etc.)"
echo -e "      Route requests to a local LLM server on your host machine"
echo ""

read -p "Enter your choice [1-4] (default: 1): " choice
choice="${choice:-1}"

case $choice in
    1)
        # Save auth mode for Claude Code subscription (manual login)
        cat > "$ENV_FILE" << EOF
# PentestGPT Authentication Configuration
# Generated by make config
PENTESTGPT_AUTH_MODE=manual
EOF

        echo -e "${GREEN}Claude Code subscription mode selected.${NC}"
        echo -e "${BLUE}After running 'make connect', execute 'claude login' inside the container.${NC}"
        ;;

    2)
        echo ""
        echo -e "${BLUE}OpenRouter Configuration${NC}"
        echo -e "${YELLOW}Get your API key from: https://openrouter.ai/keys${NC}"
        echo ""
        read -sp "Enter your OpenRouter API key: " openrouter_key
        echo ""

        if [ -z "$openrouter_key" ]; then
            echo -e "${RED}Error: API key cannot be empty${NC}"
            exit 1
        fi

        # Save auth mode
        cat > "$ENV_FILE" << EOF
# PentestGPT Authentication Configuration
# Generated by make config
PENTESTGPT_AUTH_MODE=openrouter
OPENROUTER_API_KEY=${openrouter_key}
EOF

        # Set restrictive permissions on the file
        chmod 600 "$ENV_FILE"

        echo -e "${GREEN}OpenRouter configuration saved!${NC}"
        echo -e "${BLUE}CCR will be configured automatically when you run 'make connect'${NC}"
        ;;

    3)
        echo ""
        echo -e "${BLUE}Anthropic API Key Configuration${NC}"
        echo -e "${YELLOW}Get your API key from: https://console.anthropic.com/settings/keys${NC}"
        echo ""
        read -sp "Enter your Anthropic API key: " anthropic_key
        echo ""

        if [ -z "$anthropic_key" ]; then
            echo -e "${RED}Error: API key cannot be empty${NC}"
            exit 1
        fi

        # Save auth mode
        cat > "$ENV_FILE" << EOF
# PentestGPT Authentication Configuration
# Generated by make config
PENTESTGPT_AUTH_MODE=anthropic
ANTHROPIC_API_KEY=${anthropic_key}
EOF

        # Set restrictive permissions on the file
        chmod 600 "$ENV_FILE"

        echo -e "${GREEN}Anthropic API key saved!${NC}"
        ;;

    4)
        # Save auth mode for Local LLM
        cat > "$ENV_FILE" << EOF
# PentestGPT Authentication Configuration
# Generated by make config
PENTESTGPT_AUTH_MODE=local
EOF

        echo -e "${GREEN}Local LLM mode selected!${NC}"
        echo ""
        echo -e "${BLUE}Setup Instructions:${NC}"
        echo "  1. Start your local LLM server (e.g., LM Studio) on your host machine"
        echo "     Default expected URL: http://localhost:1234/v1/chat/completions"
        echo ""
        echo "  2. To customize models or URL, edit:"
        echo "     scripts/ccr-config-template.json"
        echo ""
        echo "  3. Run 'make connect' to start PentestGPT"
        ;;

    *)
        echo -e "${RED}Invalid choice. Exiting.${NC}"
        exit 1
        ;;
esac

echo ""
echo -e "${GREEN}Configuration complete!${NC}"
echo -e "Run ${PURPLE}make connect${NC} to start PentestGPT."
