from datetime import datetime
from unittest import mock
from uuid import uuid4

from prowler.providers.aws.services.backup.backup_service import BackupPlan

AWS_REGION = "eu-west-1"
AWS_ACCOUNT_NUMBER = "123456789012"


class Test_backup_plans_exist:
    def test_no_backup_plans(self):
        backup_client = mock.MagicMock()
        backup_client.audited_account = AWS_ACCOUNT_NUMBER
        backup_client.audited_account_arn = f"arn:aws:iam::{AWS_ACCOUNT_NUMBER}:root"
        backup_client.audited_partition = "aws"
        backup_client.region = AWS_REGION
        backup_client.backup_plan_arn_template = f"arn:{backup_client.audited_partition}:backup:{backup_client.region}:{backup_client.audited_account}:backup-plan"
        backup_client.__get_backup_plan_arn_template__ = mock.MagicMock(
            return_value=backup_client.backup_plan_arn_template
        )
        backup_client.backup_plans = []
        backup_client.backup_vaults = ["vault"]
        with (
            mock.patch(
                "prowler.providers.aws.services.backup.backup_service.Backup",
                new=backup_client,
            ),
            mock.patch(
                "prowler.providers.aws.services.backup.backup_plans_exist.backup_plans_exist.backup_client",
                new=backup_client,
            ),
        ):
            # Test Check
            from prowler.providers.aws.services.backup.backup_plans_exist.backup_plans_exist import (
                backup_plans_exist,
            )

            check = backup_plans_exist()
            result = check.execute()

            assert len(result) == 1
            assert result[0].status == "FAIL"
            assert result[0].status_extended == "No Backup Plan exist."
            assert result[0].resource_id == AWS_ACCOUNT_NUMBER
            assert (
                result[0].resource_arn
                == f"arn:aws:backup:{AWS_REGION}:{AWS_ACCOUNT_NUMBER}:backup-plan"
            )
            assert result[0].region == AWS_REGION
            assert result[0].resource_tags == []

    def test_no_backup_plans_not_vaults(self):
        backup_client = mock.MagicMock
        backup_client.audited_account = AWS_ACCOUNT_NUMBER
        backup_client.audited_account_arn = f"arn:aws:iam::{AWS_ACCOUNT_NUMBER}:root"
        backup_client.region = AWS_REGION
        backup_client.backup_plans = []
        backup_client.backup_vaults = []
        with (
            mock.patch(
                "prowler.providers.aws.services.backup.backup_service.Backup",
                new=backup_client,
            ),
            mock.patch(
                "prowler.providers.aws.services.backup.backup_plans_exist.backup_plans_exist.backup_client",
                new=backup_client,
            ),
        ):
            # Test Check
            from prowler.providers.aws.services.backup.backup_plans_exist.backup_plans_exist import (
                backup_plans_exist,
            )

            check = backup_plans_exist()
            result = check.execute()

            assert len(result) == 0

    def test_one_backup_plan(self):
        backup_client = mock.MagicMock
        backup_client.audited_account = AWS_ACCOUNT_NUMBER
        backup_client.audited_account_arn = f"arn:aws:iam::{AWS_ACCOUNT_NUMBER}:root"
        backup_client.region = AWS_REGION
        backup_plan_id = str(uuid4()).upper()
        backup_plan_arn = (
            f"arn:aws:backup:{AWS_REGION}:{AWS_ACCOUNT_NUMBER}:plan:{backup_plan_id}"
        )
        backup_client.backup_plans = [
            BackupPlan(
                arn=backup_plan_arn,
                id=backup_plan_id,
                region=AWS_REGION,
                name="MyBackupPlan",
                version_id="version_id",
                last_execution_date=datetime(2015, 1, 1),
                advanced_settings=[],
                tags=[],
            )
        ]
        with (
            mock.patch(
                "prowler.providers.aws.services.backup.backup_service.Backup",
                new=backup_client,
            ),
            mock.patch(
                "prowler.providers.aws.services.backup.backup_plans_exist.backup_plans_exist.backup_client",
                new=backup_client,
            ),
        ):
            # Test Check
            from prowler.providers.aws.services.backup.backup_plans_exist.backup_plans_exist import (
                backup_plans_exist,
            )

            check = backup_plans_exist()
            result = check.execute()

            assert len(result) == 1
            assert result[0].status == "PASS"
            assert (
                result[0].status_extended
                == f"At least one Backup Plan exists: {result[0].resource_id}."
            )
            assert result[0].resource_id == "MyBackupPlan"
            assert (
                result[0].resource_arn
                == f"arn:aws:backup:{AWS_REGION}:{AWS_ACCOUNT_NUMBER}:plan:{backup_plan_id}"
            )
            assert result[0].region == AWS_REGION
            assert result[0].resource_tags == []
