#include "xselinux.h"

#include <xcb/xcb.h>
#include <xcb/xcbext.h>

#include "base/logging.h"
#include "base/posix/eintr_wrapper.h"
#include "ui/gfx/x/xproto_internal.h"

namespace x11 {

SELinux::SELinux(x11::Connection* connection,
    const x11::QueryExtensionReply& info)
    : connection_(connection), info_(info) {}

Future<SELinux::QueryVersionReply>
SELinux::QueryVersion(
    const SELinux::QueryVersionRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& client_major = request.client_major;
  auto& client_minor = request.client_minor;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 0;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // client_major
  buf.Write(&client_major);

  // client_minor
  buf.Write(&client_minor);

  Align(&buf, 4);

  return x11::SendRequest<SELinux::QueryVersionReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<SELinux::QueryVersionReply>
ReadReply<SELinux::QueryVersionReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<SELinux::QueryVersionReply>();

  auto& sequence = (*reply).sequence;
  auto& server_major = (*reply).server_major;
  auto& server_minor = (*reply).server_minor;

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // server_major
  Read(&server_major, &buf);

  // server_minor
  Read(&server_minor, &buf);

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<void>
SELinux::SetDeviceCreateContext(
    const SELinux::SetDeviceCreateContextRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  uint32_t context_len{};
  auto& context = request.context;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 1;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_len
  context_len = context.size();
  buf.Write(&context_len);

  // context
  DCHECK_EQ(static_cast<size_t>(context_len), context.size());
  for (auto& context_elem : context) {
    // context_elem
    buf.Write(&context_elem);

  }

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<SELinux::GetDeviceCreateContextReply>
SELinux::GetDeviceCreateContext(
    const SELinux::GetDeviceCreateContextRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 2;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  Align(&buf, 4);

  return x11::SendRequest<SELinux::GetDeviceCreateContextReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<SELinux::GetDeviceCreateContextReply>
ReadReply<SELinux::GetDeviceCreateContextReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<SELinux::GetDeviceCreateContextReply>();

  auto& sequence = (*reply).sequence;
  uint32_t context_len{};
  auto& context = (*reply).context;

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // context_len
  Read(&context_len, &buf);

  // pad1
  Pad(&buf, 20);

  // context
  context.resize(context_len);
  for (auto& context_elem : context) {
    // context_elem
    Read(&context_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<void>
SELinux::SetDeviceContext(
    const SELinux::SetDeviceContextRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& device = request.device;
  uint32_t context_len{};
  auto& context = request.context;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 3;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // device
  buf.Write(&device);

  // context_len
  context_len = context.size();
  buf.Write(&context_len);

  // context
  DCHECK_EQ(static_cast<size_t>(context_len), context.size());
  for (auto& context_elem : context) {
    // context_elem
    buf.Write(&context_elem);

  }

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<SELinux::GetDeviceContextReply>
SELinux::GetDeviceContext(
    const SELinux::GetDeviceContextRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& device = request.device;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 4;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // device
  buf.Write(&device);

  Align(&buf, 4);

  return x11::SendRequest<SELinux::GetDeviceContextReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<SELinux::GetDeviceContextReply>
ReadReply<SELinux::GetDeviceContextReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<SELinux::GetDeviceContextReply>();

  auto& sequence = (*reply).sequence;
  uint32_t context_len{};
  auto& context = (*reply).context;

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // context_len
  Read(&context_len, &buf);

  // pad1
  Pad(&buf, 20);

  // context
  context.resize(context_len);
  for (auto& context_elem : context) {
    // context_elem
    Read(&context_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<void>
SELinux::SetWindowCreateContext(
    const SELinux::SetWindowCreateContextRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  uint32_t context_len{};
  auto& context = request.context;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 5;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_len
  context_len = context.size();
  buf.Write(&context_len);

  // context
  DCHECK_EQ(static_cast<size_t>(context_len), context.size());
  for (auto& context_elem : context) {
    // context_elem
    buf.Write(&context_elem);

  }

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<SELinux::GetWindowCreateContextReply>
SELinux::GetWindowCreateContext(
    const SELinux::GetWindowCreateContextRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 6;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  Align(&buf, 4);

  return x11::SendRequest<SELinux::GetWindowCreateContextReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<SELinux::GetWindowCreateContextReply>
ReadReply<SELinux::GetWindowCreateContextReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<SELinux::GetWindowCreateContextReply>();

  auto& sequence = (*reply).sequence;
  uint32_t context_len{};
  auto& context = (*reply).context;

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // context_len
  Read(&context_len, &buf);

  // pad1
  Pad(&buf, 20);

  // context
  context.resize(context_len);
  for (auto& context_elem : context) {
    // context_elem
    Read(&context_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<SELinux::GetWindowContextReply>
SELinux::GetWindowContext(
    const SELinux::GetWindowContextRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& window = request.window;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 7;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // window
  buf.Write(&window);

  Align(&buf, 4);

  return x11::SendRequest<SELinux::GetWindowContextReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<SELinux::GetWindowContextReply>
ReadReply<SELinux::GetWindowContextReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<SELinux::GetWindowContextReply>();

  auto& sequence = (*reply).sequence;
  uint32_t context_len{};
  auto& context = (*reply).context;

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // context_len
  Read(&context_len, &buf);

  // pad1
  Pad(&buf, 20);

  // context
  context.resize(context_len);
  for (auto& context_elem : context) {
    // context_elem
    Read(&context_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<void>
SELinux::SetPropertyCreateContext(
    const SELinux::SetPropertyCreateContextRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  uint32_t context_len{};
  auto& context = request.context;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 8;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_len
  context_len = context.size();
  buf.Write(&context_len);

  // context
  DCHECK_EQ(static_cast<size_t>(context_len), context.size());
  for (auto& context_elem : context) {
    // context_elem
    buf.Write(&context_elem);

  }

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<SELinux::GetPropertyCreateContextReply>
SELinux::GetPropertyCreateContext(
    const SELinux::GetPropertyCreateContextRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 9;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  Align(&buf, 4);

  return x11::SendRequest<SELinux::GetPropertyCreateContextReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<SELinux::GetPropertyCreateContextReply>
ReadReply<SELinux::GetPropertyCreateContextReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<SELinux::GetPropertyCreateContextReply>();

  auto& sequence = (*reply).sequence;
  uint32_t context_len{};
  auto& context = (*reply).context;

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // context_len
  Read(&context_len, &buf);

  // pad1
  Pad(&buf, 20);

  // context
  context.resize(context_len);
  for (auto& context_elem : context) {
    // context_elem
    Read(&context_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<void>
SELinux::SetPropertyUseContext(
    const SELinux::SetPropertyUseContextRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  uint32_t context_len{};
  auto& context = request.context;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 10;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_len
  context_len = context.size();
  buf.Write(&context_len);

  // context
  DCHECK_EQ(static_cast<size_t>(context_len), context.size());
  for (auto& context_elem : context) {
    // context_elem
    buf.Write(&context_elem);

  }

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<SELinux::GetPropertyUseContextReply>
SELinux::GetPropertyUseContext(
    const SELinux::GetPropertyUseContextRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 11;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  Align(&buf, 4);

  return x11::SendRequest<SELinux::GetPropertyUseContextReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<SELinux::GetPropertyUseContextReply>
ReadReply<SELinux::GetPropertyUseContextReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<SELinux::GetPropertyUseContextReply>();

  auto& sequence = (*reply).sequence;
  uint32_t context_len{};
  auto& context = (*reply).context;

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // context_len
  Read(&context_len, &buf);

  // pad1
  Pad(&buf, 20);

  // context
  context.resize(context_len);
  for (auto& context_elem : context) {
    // context_elem
    Read(&context_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<SELinux::GetPropertyContextReply>
SELinux::GetPropertyContext(
    const SELinux::GetPropertyContextRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& window = request.window;
  auto& property = request.property;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 12;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // window
  buf.Write(&window);

  // property
  buf.Write(&property);

  Align(&buf, 4);

  return x11::SendRequest<SELinux::GetPropertyContextReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<SELinux::GetPropertyContextReply>
ReadReply<SELinux::GetPropertyContextReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<SELinux::GetPropertyContextReply>();

  auto& sequence = (*reply).sequence;
  uint32_t context_len{};
  auto& context = (*reply).context;

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // context_len
  Read(&context_len, &buf);

  // pad1
  Pad(&buf, 20);

  // context
  context.resize(context_len);
  for (auto& context_elem : context) {
    // context_elem
    Read(&context_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<SELinux::GetPropertyDataContextReply>
SELinux::GetPropertyDataContext(
    const SELinux::GetPropertyDataContextRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& window = request.window;
  auto& property = request.property;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 13;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // window
  buf.Write(&window);

  // property
  buf.Write(&property);

  Align(&buf, 4);

  return x11::SendRequest<SELinux::GetPropertyDataContextReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<SELinux::GetPropertyDataContextReply>
ReadReply<SELinux::GetPropertyDataContextReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<SELinux::GetPropertyDataContextReply>();

  auto& sequence = (*reply).sequence;
  uint32_t context_len{};
  auto& context = (*reply).context;

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // context_len
  Read(&context_len, &buf);

  // pad1
  Pad(&buf, 20);

  // context
  context.resize(context_len);
  for (auto& context_elem : context) {
    // context_elem
    Read(&context_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<SELinux::ListPropertiesReply>
SELinux::ListProperties(
    const SELinux::ListPropertiesRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& window = request.window;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 14;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // window
  buf.Write(&window);

  Align(&buf, 4);

  return x11::SendRequest<SELinux::ListPropertiesReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<SELinux::ListPropertiesReply>
ReadReply<SELinux::ListPropertiesReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<SELinux::ListPropertiesReply>();

  auto& sequence = (*reply).sequence;
  uint32_t properties_len{};
  auto& properties = (*reply).properties;

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // properties_len
  Read(&properties_len, &buf);

  // pad1
  Pad(&buf, 20);

  // properties
  properties.resize(properties_len);
  for (auto& properties_elem : properties) {
    // properties_elem
    {
      auto& name = properties_elem.name;
      uint32_t object_context_len{};
      uint32_t data_context_len{};
      auto& object_context = properties_elem.object_context;
      auto& data_context = properties_elem.data_context;

      // name
      Read(&name, &buf);

      // object_context_len
      Read(&object_context_len, &buf);

      // data_context_len
      Read(&data_context_len, &buf);

      // object_context
      object_context.resize(object_context_len);
      for (auto& object_context_elem : object_context) {
        // object_context_elem
        Read(&object_context_elem, &buf);

      }

      // pad0
      Align(&buf, 4);

      // data_context
      data_context.resize(data_context_len);
      for (auto& data_context_elem : data_context) {
        // data_context_elem
        Read(&data_context_elem, &buf);

      }

      // pad1
      Align(&buf, 4);

    }

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<void>
SELinux::SetSelectionCreateContext(
    const SELinux::SetSelectionCreateContextRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  uint32_t context_len{};
  auto& context = request.context;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 15;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_len
  context_len = context.size();
  buf.Write(&context_len);

  // context
  DCHECK_EQ(static_cast<size_t>(context_len), context.size());
  for (auto& context_elem : context) {
    // context_elem
    buf.Write(&context_elem);

  }

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<SELinux::GetSelectionCreateContextReply>
SELinux::GetSelectionCreateContext(
    const SELinux::GetSelectionCreateContextRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 16;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  Align(&buf, 4);

  return x11::SendRequest<SELinux::GetSelectionCreateContextReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<SELinux::GetSelectionCreateContextReply>
ReadReply<SELinux::GetSelectionCreateContextReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<SELinux::GetSelectionCreateContextReply>();

  auto& sequence = (*reply).sequence;
  uint32_t context_len{};
  auto& context = (*reply).context;

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // context_len
  Read(&context_len, &buf);

  // pad1
  Pad(&buf, 20);

  // context
  context.resize(context_len);
  for (auto& context_elem : context) {
    // context_elem
    Read(&context_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<void>
SELinux::SetSelectionUseContext(
    const SELinux::SetSelectionUseContextRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  uint32_t context_len{};
  auto& context = request.context;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 17;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_len
  context_len = context.size();
  buf.Write(&context_len);

  // context
  DCHECK_EQ(static_cast<size_t>(context_len), context.size());
  for (auto& context_elem : context) {
    // context_elem
    buf.Write(&context_elem);

  }

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<SELinux::GetSelectionUseContextReply>
SELinux::GetSelectionUseContext(
    const SELinux::GetSelectionUseContextRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 18;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  Align(&buf, 4);

  return x11::SendRequest<SELinux::GetSelectionUseContextReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<SELinux::GetSelectionUseContextReply>
ReadReply<SELinux::GetSelectionUseContextReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<SELinux::GetSelectionUseContextReply>();

  auto& sequence = (*reply).sequence;
  uint32_t context_len{};
  auto& context = (*reply).context;

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // context_len
  Read(&context_len, &buf);

  // pad1
  Pad(&buf, 20);

  // context
  context.resize(context_len);
  for (auto& context_elem : context) {
    // context_elem
    Read(&context_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<SELinux::GetSelectionContextReply>
SELinux::GetSelectionContext(
    const SELinux::GetSelectionContextRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& selection = request.selection;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 19;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // selection
  buf.Write(&selection);

  Align(&buf, 4);

  return x11::SendRequest<SELinux::GetSelectionContextReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<SELinux::GetSelectionContextReply>
ReadReply<SELinux::GetSelectionContextReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<SELinux::GetSelectionContextReply>();

  auto& sequence = (*reply).sequence;
  uint32_t context_len{};
  auto& context = (*reply).context;

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // context_len
  Read(&context_len, &buf);

  // pad1
  Pad(&buf, 20);

  // context
  context.resize(context_len);
  for (auto& context_elem : context) {
    // context_elem
    Read(&context_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<SELinux::GetSelectionDataContextReply>
SELinux::GetSelectionDataContext(
    const SELinux::GetSelectionDataContextRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& selection = request.selection;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 20;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // selection
  buf.Write(&selection);

  Align(&buf, 4);

  return x11::SendRequest<SELinux::GetSelectionDataContextReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<SELinux::GetSelectionDataContextReply>
ReadReply<SELinux::GetSelectionDataContextReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<SELinux::GetSelectionDataContextReply>();

  auto& sequence = (*reply).sequence;
  uint32_t context_len{};
  auto& context = (*reply).context;

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // context_len
  Read(&context_len, &buf);

  // pad1
  Pad(&buf, 20);

  // context
  context.resize(context_len);
  for (auto& context_elem : context) {
    // context_elem
    Read(&context_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<SELinux::ListSelectionsReply>
SELinux::ListSelections(
    const SELinux::ListSelectionsRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 21;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  Align(&buf, 4);

  return x11::SendRequest<SELinux::ListSelectionsReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<SELinux::ListSelectionsReply>
ReadReply<SELinux::ListSelectionsReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<SELinux::ListSelectionsReply>();

  auto& sequence = (*reply).sequence;
  uint32_t selections_len{};
  auto& selections = (*reply).selections;

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // selections_len
  Read(&selections_len, &buf);

  // pad1
  Pad(&buf, 20);

  // selections
  selections.resize(selections_len);
  for (auto& selections_elem : selections) {
    // selections_elem
    {
      auto& name = selections_elem.name;
      uint32_t object_context_len{};
      uint32_t data_context_len{};
      auto& object_context = selections_elem.object_context;
      auto& data_context = selections_elem.data_context;

      // name
      Read(&name, &buf);

      // object_context_len
      Read(&object_context_len, &buf);

      // data_context_len
      Read(&data_context_len, &buf);

      // object_context
      object_context.resize(object_context_len);
      for (auto& object_context_elem : object_context) {
        // object_context_elem
        Read(&object_context_elem, &buf);

      }

      // pad0
      Align(&buf, 4);

      // data_context
      data_context.resize(data_context_len);
      for (auto& data_context_elem : data_context) {
        // data_context_elem
        Read(&data_context_elem, &buf);

      }

      // pad1
      Align(&buf, 4);

    }

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<SELinux::GetClientContextReply>
SELinux::GetClientContext(
    const SELinux::GetClientContextRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& resource = request.resource;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 22;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // resource
  buf.Write(&resource);

  Align(&buf, 4);

  return x11::SendRequest<SELinux::GetClientContextReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<SELinux::GetClientContextReply>
ReadReply<SELinux::GetClientContextReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<SELinux::GetClientContextReply>();

  auto& sequence = (*reply).sequence;
  uint32_t context_len{};
  auto& context = (*reply).context;

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // context_len
  Read(&context_len, &buf);

  // pad1
  Pad(&buf, 20);

  // context
  context.resize(context_len);
  for (auto& context_elem : context) {
    // context_elem
    Read(&context_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

}  // namespace x11
