#ifndef GEN_UI_GFX_X_XVMC_H_
#define GEN_UI_GFX_X_XVMC_H_

#include <array>
#include <cstddef>
#include <cstdint>
#include <cstring>
#include <vector>

#include "base/component_export.h"
#include "base/memory/ref_counted_memory.h"
#include "base/memory/scoped_refptr.h"
#include "base/optional.h"
#include "base/files/scoped_file.h"
#include "ui/gfx/x/xproto_types.h"
#include "xproto.h"
#include "xv.h"

namespace x11 {

class Connection;

class COMPONENT_EXPORT(X11) XvMC {
  public:
  static constexpr unsigned major_version = 1;
  static constexpr unsigned minor_version = 1;

  XvMC(Connection* connection,
      const x11::QueryExtensionReply& info);

  uint8_t present() const {
    return info_.present;
  }
  uint8_t major_opcode() const {
    return info_.major_opcode;
  }
  uint8_t first_event() const {
    return info_.first_event;
  }
  uint8_t first_error() const {
    return info_.first_error;
  }

  Connection* connection() const { return connection_; }

  enum class Context : uint32_t {};

  enum class Surface : uint32_t {};

  enum class SubPicture : uint32_t {};

  struct SurfaceInfo {
    Surface id{};
    uint16_t chroma_format{};
    uint16_t pad0{};
    uint16_t max_width{};
    uint16_t max_height{};
    uint16_t subpicture_max_width{};
    uint16_t subpicture_max_height{};
    uint32_t mc_type{};
    uint32_t flags{};
  };

  struct QueryVersionRequest {
  };

  struct QueryVersionReply {
    uint16_t sequence{};
    uint32_t major{};
    uint32_t minor{};
  };

  using QueryVersionResponse = Response<QueryVersionReply>;

  Future<QueryVersionReply> QueryVersion(
      const QueryVersionRequest& request);

  struct ListSurfaceTypesRequest {
    Xv::Port port_id{};
  };

  struct ListSurfaceTypesReply {
    uint16_t sequence{};
    std::vector<SurfaceInfo> surfaces{};
  };

  using ListSurfaceTypesResponse = Response<ListSurfaceTypesReply>;

  Future<ListSurfaceTypesReply> ListSurfaceTypes(
      const ListSurfaceTypesRequest& request);

  struct CreateContextRequest {
    Context context_id{};
    Xv::Port port_id{};
    Surface surface_id{};
    uint16_t width{};
    uint16_t height{};
    uint32_t flags{};
  };

  struct CreateContextReply {
    uint16_t sequence{};
    uint16_t width_actual{};
    uint16_t height_actual{};
    uint32_t flags_return{};
    std::vector<uint32_t> priv_data{};
  };

  using CreateContextResponse = Response<CreateContextReply>;

  Future<CreateContextReply> CreateContext(
      const CreateContextRequest& request);

  struct DestroyContextRequest {
    Context context_id{};
  };

  using DestroyContextResponse = Response<void>;

  Future<void> DestroyContext(
      const DestroyContextRequest& request);

  struct CreateSurfaceRequest {
    Surface surface_id{};
    Context context_id{};
  };

  struct CreateSurfaceReply {
    uint16_t sequence{};
    std::vector<uint32_t> priv_data{};
  };

  using CreateSurfaceResponse = Response<CreateSurfaceReply>;

  Future<CreateSurfaceReply> CreateSurface(
      const CreateSurfaceRequest& request);

  struct DestroySurfaceRequest {
    Surface surface_id{};
  };

  using DestroySurfaceResponse = Response<void>;

  Future<void> DestroySurface(
      const DestroySurfaceRequest& request);

  struct CreateSubpictureRequest {
    SubPicture subpicture_id{};
    Context context{};
    uint32_t xvimage_id{};
    uint16_t width{};
    uint16_t height{};
  };

  struct CreateSubpictureReply {
    uint16_t sequence{};
    uint16_t width_actual{};
    uint16_t height_actual{};
    uint16_t num_palette_entries{};
    uint16_t entry_bytes{};
    std::array<uint8_t, 4> component_order{};
    std::vector<uint32_t> priv_data{};
  };

  using CreateSubpictureResponse = Response<CreateSubpictureReply>;

  Future<CreateSubpictureReply> CreateSubpicture(
      const CreateSubpictureRequest& request);

  struct DestroySubpictureRequest {
    SubPicture subpicture_id{};
  };

  using DestroySubpictureResponse = Response<void>;

  Future<void> DestroySubpicture(
      const DestroySubpictureRequest& request);

  struct ListSubpictureTypesRequest {
    Xv::Port port_id{};
    Surface surface_id{};
  };

  struct ListSubpictureTypesReply {
    uint16_t sequence{};
    std::vector<Xv::ImageFormatInfo> types{};
  };

  using ListSubpictureTypesResponse = Response<ListSubpictureTypesReply>;

  Future<ListSubpictureTypesReply> ListSubpictureTypes(
      const ListSubpictureTypesRequest& request);

  private:
  x11::Connection* const connection_;
  x11::QueryExtensionReply info_{};
};

}  // namespace x11


#endif  // GEN_UI_GFX_X_XVMC_H_
