"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.functions = void 0;
exports.isAvailableFunction = isAvailableFunction;
exports.query = query;
exports.multiQuery = multiQuery;
exports.parseSource = parseSource;
exports.default = createTraverser;
const parseQuery_1 = require("./parseQuery");
const meriyah_1 = require("meriyah");
const nodeutils_1 = require("./nodeutils");
const utils_1 = require("./utils");
const debugLogEnabled = false;
const log = debugLogEnabled ? {
    debug: (...args) => {
        console.debug(...args);
    }
} : undefined;
exports.functions = {
    "join": {
        fn: (result) => {
            if (result.length != 2)
                throw new Error("Invalid number of arugments for join");
            const [values, separators] = result;
            if (separators.length != 1)
                throw new Error("Invalid number of separators for join");
            const separator = separators[0];
            if (typeof separator != "string")
                throw new Error("Separator must be a string");
            if (values.length == 0)
                return [];
            return [values.join(separator)];
        }
    },
    "concat": {
        fn: (result) => {
            // Optimize: combine empty check with manual flattening
            const flattened = [];
            for (let i = 0; i < result.length; i++) {
                if (result[i].length === 0)
                    return [];
                for (let j = 0; j < result[i].length; j++) {
                    flattened.push(result[i][j]);
                }
            }
            return [flattened.join("")];
        }
    },
    "first": {
        fn: (result) => {
            if (result.length != 1)
                throw new Error("Invalid number of arugments for first");
            if (result[0].length == 0)
                return [];
            return [result[0][0]];
        }
    },
    "nthchild": {
        fn: (result) => {
            if (result.length != 2)
                throw new Error("Invalid number of arguments for nthchild");
            if (result[1].length != 1)
                throw new Error("Invalid number of arguments for nthchild");
            const x = result[1][0];
            const number = typeof x == "number" ? x : parseInt(x);
            return [result[0][number]];
        }
    }
};
const functionNames = new Set(Object.keys(exports.functions));
function isAvailableFunction(name) {
    return functionNames.has(name);
}
function breadCrumb(path) {
    if (!debugLogEnabled)
        return "";
    return {
        valueOf() {
            if (path.parentPath == undefined)
                return "@" + path.node.type;
            return breadCrumb(path.parentPath) + "." + (path.parentKey == path.key ? path.key : path.parentKey + "[" + path.key + "]") + "@" + path.node.type;
        }
    };
}
function createQuerier() {
    const traverser = createTraverser();
    const { getChildren, getPrimitiveChildren, getPrimitiveChildrenOrNodePaths, getBinding, createNodePath, traverse } = traverser;
    function createFilter(filter, filterResult) {
        if (filter.type == parseQuery_1.NodeType.AND || filter.type == parseQuery_1.NodeType.OR || filter.type == parseQuery_1.NodeType.EQUALS) {
            return {
                type: filter.type,
                left: createFilter(filter.left, []),
                right: createFilter(filter.right, [])
            };
        }
        else if (filter.type == parseQuery_1.NodeType.LITERAL) {
            const r = [filter.value];
            return {
                node: filter,
                result: r
            };
        }
        return createFNode(filter, filterResult);
    }
    function createFNode(token, result) {
        return {
            node: token,
            result: result
        };
    }
    function addFilterChildrenToState(filter, state) {
        if ("type" in filter && (filter.type == parseQuery_1.NodeType.AND || filter.type == parseQuery_1.NodeType.OR || filter.type == parseQuery_1.NodeType.EQUALS)) {
            addFilterChildrenToState(filter.left, state);
            addFilterChildrenToState(filter.right, state);
        }
        else if ("node" in filter) {
            if (filter.node.type == parseQuery_1.NodeType.CHILD) {
                log?.debug("ADDING FILTER CHILD", filter.node);
                state.child[state.depth + 1].push(filter);
            }
            if (filter.node.type == parseQuery_1.NodeType.DESCENDANT) {
                log?.debug("ADDING FILTER DESCENDANT", filter.node);
                state.descendant[state.depth + 1].push(filter);
            }
        }
    }
    function createFNodeAndAddToState(token, result, state) {
        log?.debug("ADDING FNODE", token);
        const fnode = createFNode(token, result);
        if (token.type == parseQuery_1.NodeType.CHILD) {
            state.child[state.depth + 1].push(fnode);
        }
        else if (token.type == parseQuery_1.NodeType.DESCENDANT) {
            state.descendant[state.depth + 1].push(fnode);
        }
        return fnode;
    }
    function isMatch(fnode, path) {
        if (fnode.node.attribute) {
            const m = fnode.node.value == path.parentKey || fnode.node.value == path.key;
            if (m)
                log?.debug("ATTR MATCH", fnode.node.value, breadCrumb(path));
            return m;
        }
        if (fnode.node.value == "*") {
            return true;
        }
        const m = fnode.node.value == path.node.type;
        if (m)
            log?.debug("NODE MATCH", fnode.node.value, breadCrumb(path));
        return m;
    }
    function addIfTokenMatch(fnode, path, state) {
        if (!isMatch(fnode, path))
            return;
        state.matches[state.depth].push([fnode, path]);
        if (fnode.node.filter) {
            const filter = createFilter(fnode.node.filter, []);
            const filteredResult = [];
            const f = { filter: filter, qNode: fnode.node, node: path.node, result: filteredResult };
            state.filters[state.depth].push(f);
            let fmap = state.filtersMap[state.depth].get(fnode.node);
            if (!fmap) {
                fmap = [];
                state.filtersMap[state.depth].set(fnode.node, fmap);
            }
            fmap.push(f);
            addFilterChildrenToState(filter, state);
            const child = fnode.node.child;
            if (child) {
                if (child.type == parseQuery_1.NodeType.FUNCTION) {
                    const fr = addFunction(fnode, child, path, state);
                    state.functionCalls[state.depth].push(fr);
                }
                else {
                    createFNodeAndAddToState(child, filteredResult, state);
                }
            }
        }
        else {
            const child = fnode.node.child;
            if (child?.type == parseQuery_1.NodeType.FUNCTION) {
                const fr = addFunction(fnode, child, path, state);
                state.functionCalls[state.depth].push(fr);
            }
            else if (child && !fnode.node.binding && !fnode.node.resolve) {
                createFNodeAndAddToState(child, fnode.result, state);
            }
        }
    }
    function addFunction(rootNode, functionCall, path, state) {
        const functionNode = { node: rootNode.node, functionCall: functionCall, parameters: [], result: [] };
        for (const param of functionCall.parameters) {
            if (param.type == parseQuery_1.NodeType.LITERAL) {
                functionNode.parameters.push({ node: param, result: [param.value] });
            }
            else {
                if (param.type == parseQuery_1.NodeType.FUNCTION) {
                    functionNode.parameters.push(addFunction(functionNode, param, path, state));
                }
                else {
                    functionNode.parameters.push(createFNodeAndAddToState(param, [], state));
                }
            }
        }
        return functionNode;
    }
    function addPrimitiveAttributeIfMatch(fnode, path) {
        if (!fnode.node.attribute || fnode.node.value == undefined)
            return;
        if (fnode.node.child || fnode.node.filter)
            return;
        if (!Object.hasOwn(path.node, fnode.node.value))
            return;
        const nodes = getPrimitiveChildren(fnode.node.value, path);
        if (nodes.length == 0)
            return;
        log?.debug("PRIMITIVE", fnode.node.value, nodes);
        fnode.result.push(...nodes);
    }
    function evaluateFilter(filter, path) {
        log?.debug("EVALUATING FILTER", filter, breadCrumb(path));
        if ("type" in filter) {
            if (filter.type == parseQuery_1.NodeType.AND) {
                const left = evaluateFilter(filter.left, path);
                if (left.length == 0) {
                    return [];
                }
                const r = evaluateFilter(filter.right, path);
                return r;
            }
            if (filter.type == parseQuery_1.NodeType.OR) {
                const left = evaluateFilter(filter.left, path);
                if (left.length > 0) {
                    return left;
                }
                const r = evaluateFilter(filter.right, path);
                return r;
            }
            if (filter.type == parseQuery_1.NodeType.EQUALS) {
                const left = evaluateFilter(filter.left, path);
                const right = evaluateFilter(filter.right, path);
                // Optimize: use Set for O(1) lookups instead of O(n) includes
                if (right.length > 3) {
                    const rightSet = new Set(right);
                    const r = [];
                    for (let i = 0; i < left.length; i++) {
                        if (rightSet.has(left[i]))
                            r.push(left[i]);
                    }
                    return r;
                }
                // For small arrays, includes is faster than Set creation
                const r = [];
                for (let i = 0; i < left.length; i++) {
                    if (right.includes(left[i]))
                        r.push(left[i]);
                }
                return r;
            }
            throw new Error("Unknown filter type: " + filter.type);
        }
        if (filter.node.type == parseQuery_1.NodeType.PARENT) {
            const r = resolveFilterWithParent(filter.node, path);
            return r;
        }
        return filter.result;
    }
    function resolveBinding(path) {
        if (!(0, nodeutils_1.isIdentifier)(path.node))
            return undefined;
        log?.debug("RESOLVING BINDING FOR ", path.node);
        const name = path.node.name;
        if (name == undefined || typeof name != "string")
            return undefined;
        //const binding = path.scope.getBinding(name);
        const binding = getBinding(path.scopeId, name);
        if (!binding)
            return undefined;
        log?.debug("THIS IS THE BINDING", binding);
        return binding.path;
    }
    function resolveFilterWithParent(node, path) {
        let startNode = node;
        let startPath = path;
        while (startNode.type == parseQuery_1.NodeType.PARENT) {
            if (!startNode.child)
                throw new Error("Parent filter must have child");
            if (!startPath.parentPath)
                return [];
            log?.debug("STEP OUT", startNode, breadCrumb(startPath));
            startNode = startNode.child;
            startPath = startPath.parentPath;
        }
        return resolveDirectly(startNode, startPath);
    }
    let subQueryCounter = 0;
    const memo = new Map();
    function resolveDirectly(node, path) {
        let startNode = node;
        const startPath = path;
        let paths = [startPath];
        while (startNode.attribute && startNode.type == parseQuery_1.NodeType.CHILD) {
            const lookup = startNode.value;
            if (!lookup)
                throw new Error("Selector must have a value");
            //log?.debug("STEP IN ", lookup, paths.map(p => breadCrumb(p)));
            // Optimize: avoid filter().map().flat() chain - use single loop
            const nodes = [];
            for (let i = 0; i < paths.length; i++) {
                const p = paths[i];
                if (!(0, nodeutils_1.isNodePath)(p))
                    continue;
                const arr = getPrimitiveChildrenOrNodePaths(lookup, p);
                for (let j = 0; j < arr.length; j++) {
                    nodes.push(arr[j]);
                }
            }
            if (nodes.length == 0)
                return [];
            paths = nodes;
            if (startNode.resolve) {
                const resolved = [];
                for (let i = 0; i < paths.length; i++) {
                    const p = paths[i];
                    if (!(0, nodeutils_1.isNodePath)(p))
                        continue;
                    const binding = resolveBinding(p);
                    if (!binding)
                        continue;
                    const children = getChildren("init", binding);
                    for (let j = 0; j < children.length; j++) {
                        resolved.push(children[j]);
                    }
                }
                if (resolved.length > 0)
                    paths = resolved;
            }
            else if (startNode.binding) {
                const bindings = [];
                for (let i = 0; i < paths.length; i++) {
                    const p = paths[i];
                    if (!(0, nodeutils_1.isNodePath)(p))
                        continue;
                    const binding = resolveBinding(p);
                    if (binding)
                        bindings.push(binding);
                }
                paths = bindings;
            }
            const filter = startNode.filter;
            if (filter) {
                const filtered = [];
                for (let i = 0; i < paths.length; i++) {
                    const p = paths[i];
                    if (!(0, nodeutils_1.isNodePath)(p))
                        continue;
                    if (travHandle({ subquery: filter }, p).subquery.length > 0) {
                        filtered.push(p);
                    }
                }
                paths = filtered;
            }
            if (!startNode.child) {
                const results = new Array(paths.length);
                for (let i = 0; i < paths.length; i++) {
                    const p = paths[i];
                    results[i] = (0, nodeutils_1.isPrimitive)(p) ? p : p.node;
                }
                return results;
            }
            startNode = startNode.child;
        }
        //log?.debug("DIRECT TRAV RESOLVE", startNode, paths.map(p => breadCrumb(p)));
        const result = [];
        //console.log(paths.length, subQueryCounter);
        for (const path of paths) {
            if ((0, nodeutils_1.isNodePath)(path)) {
                if (memo.has(startNode) && memo.get(startNode).has(path)) {
                    const cached = memo.get(startNode).get(path);
                    for (let i = 0; i < cached.length; i++) {
                        result.push(cached[i]);
                    }
                }
                else {
                    const subQueryKey = "subquery-" + subQueryCounter++;
                    const subQueryResult = travHandle({ [subQueryKey]: startNode }, path)[subQueryKey];
                    if (!memo.has(startNode))
                        memo.set(startNode, new Map());
                    memo.get(startNode)?.set(path, subQueryResult);
                    for (let i = 0; i < subQueryResult.length; i++) {
                        result.push(subQueryResult[i]);
                    }
                }
            }
        }
        log?.debug("DIRECT TRAV RESOLVE RESULT", result);
        return result;
    }
    function addResultIfTokenMatch(fnode, path, state) {
        const matchingFilters = [];
        //console.log("FILTERS", state.filters[state.depth].length, state.filtersMap[state.depth].get(fnode.node)?.length);
        const filters = [];
        const nodeFilters = state.filtersMap[state.depth].get(fnode.node);
        if (nodeFilters) {
            for (let i = 0; i < nodeFilters.length; i++) {
                const f = nodeFilters[i];
                if (f.qNode !== fnode.node)
                    continue;
                if (f.node !== path.node)
                    continue;
                filters.push(f);
            }
            for (let i = 0; i < filters.length; i++) {
                const f = filters[i];
                if (evaluateFilter(f.filter, path).length > 0) {
                    matchingFilters.push(f);
                }
            }
            if (filters.length > 0 && matchingFilters.length == 0)
                return;
        }
        if (fnode.node.resolve) {
            const binding = resolveBinding(path);
            const resolved = binding ? getChildren("init", binding)[0] : undefined;
            if (fnode.node.child) {
                const result = resolveDirectly(fnode.node.child, resolved ?? path);
                for (let i = 0; i < result.length; i++) {
                    fnode.result.push(result[i]);
                }
            }
            else {
                fnode.result.push(path.node);
            }
        }
        else if (fnode.node.binding) {
            const binding = resolveBinding(path);
            if (binding) {
                if (fnode.node.child) {
                    const result = resolveDirectly(fnode.node.child, binding);
                    for (let i = 0; i < result.length; i++) {
                        fnode.result.push(result[i]);
                    }
                }
                else {
                    fnode.result.push(binding.node);
                }
            }
        }
        else if (!fnode.node.child) {
            fnode.result.push(path.node);
        }
        else if (fnode.node.child.type == parseQuery_1.NodeType.FUNCTION) {
            const functionCallResult = state.functionCalls[state.depth].find(f => f.node == fnode.node);
            if (!functionCallResult)
                throw new Error("Did not find expected function call for " + fnode.node.child.function);
            resolveFunctionCalls(fnode, functionCallResult, path, state);
        }
        else if (matchingFilters.length > 0) {
            log?.debug("HAS MATCHING FILTER", fnode.result.length, matchingFilters.length, breadCrumb(path));
            for (let i = 0; i < matchingFilters.length; i++) {
                const filterResult = matchingFilters[i].result;
                for (let j = 0; j < filterResult.length; j++) {
                    fnode.result.push(filterResult[j]);
                }
            }
        }
    }
    function resolveFunctionCalls(fnode, functionCallResult, path, state) {
        const parameterResults = [];
        for (let i = 0; i < functionCallResult.parameters.length; i++) {
            const p = functionCallResult.parameters[i];
            if ("parameters" in p) {
                resolveFunctionCalls(p, p, path, state);
                parameterResults.push(p.result);
            }
            else {
                parameterResults.push(p.result);
            }
        }
        const functionResult = exports.functions[functionCallResult.functionCall.function].fn(parameterResults);
        log?.debug("PARAMETER RESULTS", functionCallResult.functionCall.function, parameterResults, functionResult);
        for (let i = 0; i < functionResult.length; i++) {
            fnode.result.push(functionResult[i]);
        }
    }
    function travHandle(queries, root) {
        // Optimize: create results object directly instead of Object.fromEntries + map
        const results = {};
        const queryKeys = Object.keys(queries);
        for (let i = 0; i < queryKeys.length; i++) {
            results[queryKeys[i]] = [];
        }
        const state = {
            depth: 0,
            child: [[], []],
            descendant: [[], []],
            filters: [[], []],
            filtersMap: [new Map(), new Map()],
            matches: [[]],
            functionCalls: [[]]
        };
        for (const [name, node] of Object.entries(queries)) {
            createFNodeAndAddToState(node, results[name], state);
        }
        // Optimize: replace forEach with for loop
        const childAtDepth = state.child[state.depth + 1];
        for (let i = 0; i < childAtDepth.length; i++) {
            addPrimitiveAttributeIfMatch(childAtDepth[i], root);
        }
        const descendantSlice = state.descendant.slice(0, state.depth + 1);
        for (let i = 0; i < descendantSlice.length; i++) {
            const fnodes = descendantSlice[i];
            for (let j = 0; j < fnodes.length; j++) {
                addPrimitiveAttributeIfMatch(fnodes[j], root);
            }
        }
        traverse(root.node, {
            enter(path, state) {
                //log?.debug("ENTER", breadCrumb(path));
                state.depth++;
                state.child.push([]);
                state.descendant.push([]);
                state.filters.push([]);
                state.filtersMap.push(new Map());
                state.matches.push([]);
                state.functionCalls.push([]);
                for (const fnode of state.child[state.depth]) {
                    addIfTokenMatch(fnode, path, state);
                }
                for (const fnodes of state.descendant.slice(0, state.depth + 1)) {
                    for (const fnode of fnodes) {
                        addIfTokenMatch(fnode, path, state);
                    }
                }
            },
            exit(path, state) {
                log?.debug("EXIT", breadCrumb(path));
                // Check for attributes as not all attributes are visited
                // Optimize: replace forEach with for loop
                const childAtDepthPlusOne = state.child[state.depth + 1];
                for (let i = 0; i < childAtDepthPlusOne.length; i++) {
                    addPrimitiveAttributeIfMatch(childAtDepthPlusOne[i], path);
                }
                for (let i = 0; i < state.descendant.length; i++) {
                    const fnodes = state.descendant[i];
                    for (let j = 0; j < fnodes.length; j++) {
                        addPrimitiveAttributeIfMatch(fnodes[j], path);
                    }
                }
                const matchesAtDepth = state.matches[state.depth];
                for (let i = 0; i < matchesAtDepth.length; i++) {
                    addResultIfTokenMatch(matchesAtDepth[i][0], matchesAtDepth[i][1], state);
                }
                state.depth--;
                state.child.pop();
                state.descendant.pop();
                state.filters.pop();
                state.filtersMap.pop();
                state.matches.pop();
                state.functionCalls.pop();
            }
        }, root.scopeId, state, root);
        return results;
    }
    function beginHandle(queries, path) {
        const rootPath = createNodePath(path, undefined, undefined, undefined, undefined);
        const r = travHandle(queries, rootPath);
        memo.clear();
        return r;
    }
    return {
        beginHandle
    };
}
const defaultKey = "__default__";
function query(code, query, returnAST) {
    const result = multiQuery(code, { [defaultKey]: query }, returnAST);
    if (returnAST) {
        const r = result[defaultKey];
        r.__AST = result.__AST;
        return r;
    }
    return result[defaultKey];
}
function multiQuery(code, namedQueries, returnAST) {
    const start = Date.now();
    const ast = typeof code == "string" ? parseSource(code) : code;
    if (ast == null)
        throw new Error("Could not pase code");
    // Optimize: parse queries directly instead of Object.fromEntries + map
    const queries = {};
    const entries = Object.entries(namedQueries);
    for (let i = 0; i < entries.length; i++) {
        const [name, queryStr] = entries[i];
        queries[name] = (0, parseQuery_1.parse)(queryStr);
    }
    const querier = createQuerier();
    const result = querier.beginHandle(queries, ast);
    log?.debug("Query time: ", Date.now() - start);
    if (returnAST) {
        return { ...result, __AST: ast };
    }
    return result;
}
function parseSource(source, optimize = true) {
    const parsingOptions = optimize ? { loc: false, ranges: false } : { loc: true, ranges: true };
    try {
        return (0, meriyah_1.parseScript)(source, { module: true, next: true, ...parsingOptions });
    }
    catch (e) {
        return (0, meriyah_1.parseScript)(source, { module: false, next: true, ...parsingOptions, webcompat: true });
    }
}
function createTraverser() {
    let scopeIdCounter = 0;
    const scopes = new Map();
    let removedScopes = 0;
    const nodePathsCreated = {};
    function createScope(parentScopeId) {
        const id = scopeIdCounter++;
        if (parentScopeId != undefined) {
            scopes.set(id, parentScopeId ?? -1);
        }
        return id;
    }
    function getBinding(scopeId, name) {
        let currentScope = scopes.get(scopeId);
        while (currentScope !== undefined) {
            if (typeof currentScope !== "number") {
                // Full scope: Check for binding
                if (currentScope.bindings[name]) {
                    return currentScope.bindings[name];
                }
                // Move to parent scope
                if (currentScope.parentScopeId === -1)
                    break; // No parent scope
                currentScope = scopes.get(currentScope.parentScopeId);
            }
            else {
                // Lightweight scope: Retrieve parent scope
                if (currentScope === -1 || currentScope == undefined)
                    break; // No parent scope
                currentScope = scopes.get(currentScope);
            }
        }
        return undefined; // Binding not found
    }
    function setBinding(scopeId, name, binding) {
        let scope = scopes.get(scopeId);
        if (typeof scope === "number" || scope === undefined) {
            // Upgrade the lightweight scope to a full scope
            scope = { bindings: {}, id: scopeId, parentScopeId: scope };
            scopes.set(scopeId, scope);
        }
        if (scope && typeof scope !== "number") {
            scope.bindings[name] = binding;
        }
    }
    let pathsCreated = 0;
    function getChildren(key, path) {
        if (key in path.node) {
            const r = path.node[key];
            if (Array.isArray(r)) {
                const len = r.length;
                const result = new Array(len);
                for (let i = 0; i < len; i++) {
                    result[i] = createNodePath(r[i], i, key, path.scopeId, path.functionScopeId, path);
                }
                return result;
            }
            else if (r != undefined) {
                return [createNodePath(r, key, key, path.scopeId, path.functionScopeId, path)];
            }
        }
        return [];
    }
    function getPrimitiveChildren(key, path) {
        if (key in path.node) {
            const r = path.node[key];
            const arr = (0, utils_1.toArray)(r);
            // Optimize: single loop instead of chained filter()
            const result = [];
            for (let i = 0; i < arr.length; i++) {
                const item = arr[i];
                if ((0, utils_1.isDefined)(item) && (0, nodeutils_1.isPrimitive)(item)) {
                    result.push(item);
                }
            }
            return result;
        }
        return [];
    }
    function getPrimitiveChildrenOrNodePaths(key, path) {
        if (key in path.node) {
            const r = path.node[key];
            if (Array.isArray(r)) {
                const len = r.length;
                const result = new Array(len);
                for (let i = 0; i < len; i++) {
                    const n = r[i];
                    result[i] = (0, nodeutils_1.isPrimitive)(n) ? n : createNodePath(n, i, key, path.scopeId, path.functionScopeId, path);
                }
                return result;
            }
            else if (r != undefined) {
                return [
                    (0, nodeutils_1.isPrimitive)(r) ? r :
                        createNodePath(r, key, key, path.scopeId, path.functionScopeId, path)
                ];
            }
        }
        return [];
    }
    const nodePathMap = new WeakMap();
    function createNodePath(node, key, parentKey, scopeId, functionScopeId, nodePath) {
        if (nodePathMap.has(node)) {
            //if (node.extra?.nodePath) {
            //const path = node.extra.nodePath;
            const path = nodePathMap.get(node);
            if (nodePath && (0, nodeutils_1.isExportSpecifier)(nodePath.node) && key == "exported" && path.key == "local") {
                //Special handling for "export { someName }" as id is both local and exported
                path.key = "exported";
                path.parentPath = nodePath;
                return path;
            }
            if (key != undefined)
                path.key = typeof (key) == "number" ? key.toString() : key;
            if (parentKey != undefined)
                path.parentKey = parentKey;
            if (nodePath != undefined)
                path.parentPath = nodePath;
            return path;
        }
        const finalScope = ((node.extra && node.extra.scopeId != undefined) ? node.extra.scopeId : scopeId) ?? createScope();
        const finalFScope = ((node.extra && node.extra.functionScopeId != undefined) ? node.extra.functionScopeId : functionScopeId) ?? finalScope;
        const path = {
            node,
            scopeId: finalScope,
            functionScopeId: finalFScope,
            parentPath: nodePath,
            key: typeof (key) == "number" ? key.toString() : key,
            parentKey
        };
        if ((0, nodeutils_1.isNode)(node)) {
            //node.extra = node.extra ?? {};
            //node.extra.nodePath = path;
            //Object.defineProperty(node.extra, "nodePath", { enumerable: false });
            nodePathMap.set(node, path);
        }
        nodePathsCreated[node.type] = (nodePathsCreated[node.type] ?? 0) + 1;
        pathsCreated++;
        return path;
    }
    function registerBinding(stack, scopeId, functionScopeId, key, parentKey) {
        //console.log("x registerBinding?", isIdentifier(node) ? node.name : node.type, parentNode.type, grandParentNode?.type, scopeId, isBinding(node, parentNode, grandParentNode));
        const node = stack[stack.length - 1];
        if (!(0, nodeutils_1.isIdentifier)(node))
            return;
        const parentNode = stack[stack.length - 2];
        if ((0, nodeutils_1.isAssignmentExpression)(parentNode) || (0, nodeutils_1.isMemberExpression)(parentNode) || (0, nodeutils_1.isUpdateExpression)(parentNode) || (0, nodeutils_1.isExportSpecifier)(parentNode))
            return;
        const grandParentNode = stack[stack.length - 3];
        if (!(0, nodeutils_1.isBinding)(node, parentNode, grandParentNode))
            return;
        if (key == "id" && !(0, nodeutils_1.isVariableDeclarator)(parentNode)) {
            setBinding(functionScopeId, node.name, { path: createNodePath(node, undefined, undefined, scopeId, functionScopeId) });
            return;
        }
        if ((0, nodeutils_1.isVariableDeclarator)(parentNode) && (0, nodeutils_1.isVariableDeclaration)(grandParentNode)) {
            if (grandParentNode.kind == "var") {
                setBinding(functionScopeId, node.name, { path: createNodePath(parentNode, undefined, undefined, scopeId, functionScopeId) });
                return;
            }
            else {
                setBinding(scopeId, node.name, { path: createNodePath(parentNode, undefined, undefined, scopeId, functionScopeId) });
                return;
            }
        }
        if ((0, nodeutils_1.isScope)(node, parentNode)) {
            setBinding(scopeId, node.name, { path: createNodePath(node, key, parentKey, scopeId, functionScopeId) });
        } /*else {
          console.log(node.type, parentNode.type, grandParentNode?.type);
        }*/
    }
    let bindingNodesVisited = 0;
    function registerBindings(stack, scopeId, functionScopeId) {
        const node = stack[stack.length - 1];
        if (!(0, nodeutils_1.isNode)(node))
            return;
        if (node.extra?.scopeId != undefined)
            return;
        node.extra = node.extra ?? {};
        node.extra.scopeId = scopeId;
        bindingNodesVisited++;
        const keys = nodeutils_1.VISITOR_KEYS[node.type];
        if (keys.length == 0)
            return;
        let childScopeId = scopeId;
        if ((0, nodeutils_1.isScopable)(node)) {
            childScopeId = createScope(scopeId);
        }
        for (let keyIdx = 0; keyIdx < keys.length; keyIdx++) {
            const key = keys[keyIdx];
            const childNodes = node[key];
            const children = (0, utils_1.toArray)(childNodes);
            for (let i = 0; i < children.length; i++) {
                const child = children[i];
                if (!(0, utils_1.isDefined)(child) || !(0, nodeutils_1.isNode)(child))
                    continue;
                const f = key === "body" && ((0, nodeutils_1.isFunctionDeclaration)(node) || (0, nodeutils_1.isFunctionExpression)(node)) ? childScopeId : functionScopeId;
                stack.push(child);
                if ((0, nodeutils_1.isIdentifier)(child)) {
                    const k = Array.isArray(childNodes) ? i : key;
                    registerBinding(stack, childScopeId, f, k, key);
                }
                else {
                    registerBindings(stack, childScopeId, f);
                }
                stack.pop();
            }
        }
        if (childScopeId != scopeId && typeof scopes.get(childScopeId) == "number") { // Scope has not been populated
            scopes.set(childScopeId, scopes.get(scopeId));
            removedScopes++;
        }
    }
    function traverseInner(node, visitor, scopeId, functionScopeId, state, path) {
        const nodePath = path ?? createNodePath(node, undefined, undefined, scopeId, functionScopeId);
        const keys = nodeutils_1.VISITOR_KEYS[node.type];
        if (nodePath.parentPath) {
            const stack = [];
            if (nodePath.parentPath.parentPath?.node)
                stack.push(nodePath.parentPath.parentPath.node);
            stack.push(nodePath.parentPath.node, nodePath.node);
            registerBindings(stack, nodePath.scopeId, nodePath.functionScopeId);
        }
        // Optimization: Check if we need to traverse children at all
        // If there are no descendant queries and no child queries at next depth, skip traversal
        const stateTyped = state;
        const hasDescendantQueries = stateTyped.descendant && stateTyped.descendant.some(arr => arr.length > 0);
        const hasChildQueriesAtNextDepth = stateTyped.child && stateTyped.child[stateTyped.depth + 1] && stateTyped.child[stateTyped.depth + 1].length > 0;
        // If no queries would match in this subtree, skip traversal entirely
        if (!hasDescendantQueries && !hasChildQueriesAtNextDepth) {
            return;
        }
        for (let keyIdx = 0; keyIdx < keys.length; keyIdx++) {
            const key = keys[keyIdx];
            const childNodes = node[key];
            const children = Array.isArray(childNodes) ? childNodes : childNodes ? [childNodes] : [];
            const nodePaths = [];
            for (let i = 0; i < children.length; i++) {
                const child = children[i];
                if ((0, nodeutils_1.isNode)(child)) {
                    const childPath = createNodePath(child, Array.isArray(childNodes) ? i : key, key, nodePath.scopeId, nodePath.functionScopeId, nodePath);
                    nodePaths.push(childPath);
                }
            }
            for (let i = 0; i < nodePaths.length; i++) {
                const childPath = nodePaths[i];
                visitor.enter(childPath, state);
                traverseInner(childPath.node, visitor, nodePath.scopeId, nodePath.functionScopeId, state, childPath);
                visitor.exit(childPath, state);
            }
        }
    }
    const sOut = [];
    function traverse(node, visitor, scopeId, state, path) {
        const fscope = path?.functionScopeId ?? node.extra?.functionScopeId ?? scopeId;
        traverseInner(node, visitor, scopeId, fscope, state, path);
        if (!sOut.includes(scopeIdCounter)) {
            log?.debug("Scopes created", scopeIdCounter, " Scopes removed", removedScopes, "Paths created", pathsCreated, bindingNodesVisited);
            sOut.push(scopeIdCounter);
            const k = Object.fromEntries(Object.entries(nodePathsCreated).sort((a, b) => a[1] - b[1]));
            log?.debug("Node paths created", k);
        }
    }
    return {
        traverse,
        createNodePath,
        getChildren,
        getPrimitiveChildren,
        getPrimitiveChildrenOrNodePaths,
        getBinding
    };
}
