#!/usr/bin/env python3

"""
easywin.py 0.2 - Windows and AD Tactical Exploitation Toolkit
Copyright (c) 2017-2018 Marco Ivaldi <raptor@0xdeadbeef.info>

"The Other Way to Pen-Test" --HD Moore & Valsmith

If you're like me, when performing a penetration test you
prefer to go for the easy win. Easywin is a Python script
that provides a toolkit for exploit-less attacks aimed at
Windows and Active Directory environments, by leveraging
information gathering and brute force capabilities against
the SMB protocol.

Based on:
http://www.0xdeadbeef.info/code/samba-hax0r
https://github.com/portcullislabs/enum4linux

Requirements:
Python 3 (https://pythonclock.org/ is ticking...)
Samba (smbclient, rpcclient, net)
Impacket with polenum.py from https://github.com/0xdea/polenum

Example usage:
$ ./easywin.py info -f hosts.txt -a
$ ./easywin.py brute -f hosts.txt -u users.txt

TODO:
Add check to determine if the target host/service is up
Enumerate Group Policy Preference (GPP) Saved Passwords
Improve error handling (e.g. external command not found)
Avoid using external commands (it works but it's ugly!)

Get the latest version at:
https://github.com/0xdea/tactical-exploitation/
"""

VERSION = "0.2"
BANNER = """
easywin.py {0} - Windows and AD Tactical Exploitation Toolkit
Copyright (c) 2017-2018 Marco Ivaldi <raptor@0xdeadbeef.info>
""".format(VERSION)

import sys
import argparse
import subprocess
import re

def info(args):
    """
    Information gathering handler
    """

    domain = args.D
    username = args.U
    password = args.P
    start = args.S
    end = args.E
    targets = get_targets(args)

    if start > end:
        print("// error: start sid cannot be greater than end sid")
        sys.exit(1)
    
    if args.all: # get users, groups, and shares
        args.all = False
        args.users = True
        args.groups = True
        args.shares = True
        args.policy = False
        args.rid = False

    for host in targets:

        if args.users: # get user list
            info_users(domain, username, password, host)

        if args.groups: # get group list and membership
            info_groups(domain, username, password, host)

        if args.shares: # get share list
            info_shares(domain, username, password, host)

        if args.policy: # get password policy
            info_policy(domain, username, password, host)

        if args.rid: # get users and groups via rid cycling
            info_rid(domain, username, password, host, start, end)

def info_users(domain, username, password, host):
    """
    Get user list
    """

    print("*** users on {3} ({0}\{1}:{2}) ***\n"
          .format(domain, username, password, host))

    # get user list (querydispinfo)
    cmd = ("rpcclient -W'{0}' -U'{1}%{2}' {3} -c querydispinfo 2>&1"
           .format(domain, username, password, host))
    try:
        o = subprocess.run(cmd, shell=True, check=True, stdout=subprocess.PIPE)
    except (KeyboardInterrupt, SystemExit):
        sys.exit(1)
    except:
        pass
    else:
        print(o.stdout.decode(sys.stdout.encoding))

    # get user list (enumdomusers)
    cmd = ("rpcclient -W'{0}' -U'{1}%{2}' {3} -c enumdomusers 2>&1"
           .format(domain, username, password, host))
    try:
        o = subprocess.run(cmd, shell=True, check=True, stdout=subprocess.PIPE)
    except (KeyboardInterrupt, SystemExit):
        sys.exit(1)
    except:
        pass
    else:
        print(o.stdout.decode(sys.stdout.encoding))

def info_groups(domain, username, password, host):
    """
    Get group list and membership
    """

    output = b""

    print("*** groups on {3} ({0}\{1}:{2}) ***\n"
          .format(domain, username, password, host))

    # get group list (enumalsgroups)
    for grouptype in ["builtin", "domain"]:
        cmd = ("rpcclient -W'{0}' -U'{1}%{2}' {3} -c 'enumalsgroups {4}' 2>&1"
               .format(domain, username, password, host, grouptype))
        try:
            o = subprocess.run(cmd, shell=True, check=True, 
                    stdout=subprocess.PIPE)
        except (KeyboardInterrupt, SystemExit):
            sys.exit(1)
        except:
            pass
        else:
            output += o.stdout

    # get group list (enumdomgroups)
    cmd = ("rpcclient -W'{0}' -U'{1}%{2}' {3} -c enumdomgroups 2>&1"
           .format(domain, username, password, host))
    try:
        o = subprocess.run(cmd, shell=True, check=True, stdout=subprocess.PIPE)
    except (KeyboardInterrupt, SystemExit):
        sys.exit(1)
    except:
        pass
    else:
        output += o.stdout

    if not output: return
    res = output.decode(sys.stdout.encoding)
    print(res)

    p = re.compile("\[.+\] ")
    groups = p.findall(res)
    if not groups: return

    # get group membership
    for group in groups:
        print(group)
        cmd = ("net rpc group members '{4}' -W'{0}' -U'{1}%{2}' -I {3} 2>&1"
               .format(domain, username, password, host, group[1:-2]))
        try:
            o = subprocess.run(cmd, shell=True, check=True, 
                    stdout=subprocess.PIPE)
        except (KeyboardInterrupt, SystemExit):
            sys.exit(1)
        except:
            pass
        else:
            print(o.stdout.decode(sys.stdout.encoding))

def info_shares(domain, username, password, host):
    """
    Get share list
    """

    output = b""

    print("*** shares on {3} ({0}\{1}:{2}) ***\n"
          .format(domain, username, password, host))

    # get share list
    cmd = ("smbclient -W'{0}' -U'{1}%{2}' -L //{3} 2>&1"
           .format(domain, username, password, host))
    try:
        o = subprocess.run(cmd, shell=True, check=True, stdout=subprocess.PIPE)
    except (KeyboardInterrupt, SystemExit):
        sys.exit(1)
    except:
        pass
    else:
        output += o.stdout

    if not output: return
    res = output.decode(sys.stdout.encoding)

    p = re.compile("\n\s+\S+\s+(?:Disk|IPC|Printer).+")
    shares = p.findall(res)
    if not shares: return

    # map and print shares
    for descr in shares:
        share = descr.split()[0]
        cmd = ("smbclient -W'{0}' -U'{1}%{2}' '//{3}/{4}' -c dir 2>&1"
               .format(domain, username, password, host, share))
        try:
            o = subprocess.run(cmd, shell=True, check=True, 
                    stdout=subprocess.PIPE)
        except (KeyboardInterrupt, SystemExit):
            sys.exit(1)
        except:
            print("[ ] Access Denied\t{0}".format(descr.lstrip()))
        else:
            print("[x] Access Granted\t{0}".format(descr.lstrip()))
    else:
        print()

def info_policy(domain, username, password, host):
    """
    Get password policy
    """
    
    print("*** policy on {3} ({0}\{1}:{2}) ***\n"
          .format(domain, username, password, host))

    # get password policy
    cmd = ("polenum.py '{1}:{2}@{3}' 2>&1"
           .format(domain, username, password, host))
    try:
        o = subprocess.run(cmd, shell=True, check=True, stdout=subprocess.PIPE)
    except (KeyboardInterrupt, SystemExit):
        sys.exit(1)
    except:
        print("// error launching polenum.py")
    else:
        print(o.stdout.decode(sys.stdout.encoding).strip())

    print()

def info_rid(domain, username, password, host, start, end):
    """
    Get users and groups via RID cycling
    """

    output = b""
    known_users = ["administrator",
                   "guest",
                   "krbtgt",
                   "domain admins",
                   "root",
                   "none"]

    print("*** rid cycling on {3} ({0}\{1}:{2}) ({4}-{5}) ***\n"
          .format(domain, username, password, host, start, end))

    # get sids (lookupnames)
    for user in known_users:
        cmd = ("rpcclient -W'{0}' -U'{1}%{2}' {3} -c 'lookupnames {4}' 2>&1"
               .format(domain, username, password, host, user))
        try:
            o = subprocess.run(cmd, shell=True, check=True, 
                    stdout=subprocess.PIPE)
        except (KeyboardInterrupt, SystemExit):
            sys.exit(1)
        except:
            pass
        else:
            output += o.stdout

    # get sids (lsaenumsid)
    cmd = ("rpcclient -W'{0}' -U'{1}%{2}' {3} -c lsaenumsid 2>&1"
           .format(domain, username, password, host))
    try:
        o = subprocess.run(cmd, shell=True, check=True, stdout=subprocess.PIPE)
    except (KeyboardInterrupt, SystemExit):
        sys.exit(1)
    except:
        pass
    else:
        output += o.stdout

    if not output: return
    res = output.decode(sys.stdout.encoding)

    # parse sids
    p = re.compile("(S-1-5-21-[\d-]+)-") # windows
    sids = set(p.findall(res))
    p = re.compile("(S-1-22-[\d-]+)-") # unix
    sids |= set(p.findall(res))
    if not sids: return

    # perform rid cycling for each sid found
    for sid in sorted(sids):
        cmd = ("rpcclient -W'{0}' -U'{1}%{2}' {3} -c 'lookupsids {4}' 2>&1"
               .format(domain, username, password, host, sid))
        try:
            o = subprocess.run(cmd, shell=True, check=True, 
                    stdout=subprocess.PIPE)
        except (KeyboardInterrupt, SystemExit):
            sys.exit(1)
        except:
            pass
        else:
            print("[{0}]".format(o.stdout.decode(sys.stdout.encoding).rstrip()))

        # enumerate objects via rid cycling
        for i in range(start, end + 1):
            rid = sid + "-" + str(i)
            cmd = ("rpcclient -W'{0}' -U'{1}%{2}' {3} -c 'lookupsids {4}' 2>&1"
                   .format(domain, username, password, host, rid))
            try:
                o = subprocess.run(cmd, shell=True, check=True,
                        stdout=subprocess.PIPE)
            except (KeyboardInterrupt, SystemExit):
                sys.exit(1)
            except:
                pass
            else:
                print(o.stdout.decode(sys.stdout.encoding), end="")

        print()

def brute(args):
    """
    Brute force handler
    """

    domain = args.d
    usernames = [u.rstrip() for u in args.u]
    passwords = []
    if args.p:
        passwords = [p.rstrip() for p in args.p]
    targets = get_targets(args)
    found_glob = 0

    for host in targets:

        found_host = 0

        if passwords: # test username:username and username:password
            for username in usernames:
                if brute_do(domain, username, username, host):
                    found_host += 1
                    continue
                for password in passwords:
                    if brute_do(domain, username, password, host):
                        found_host += 1
                        break

        else: # test only username:username
            for username in usernames:
                found_host += brute_do(domain, username, username, host)

        found_glob += found_host
        print("\n*** {0} accounts found on {1} ***\n".format(found_host, host))

    print("*** {0} accounts found globally ***\n".format(found_glob))

def brute_do(domain, username, password, host):
    """
    Perform the brute force attack
    """

    cmd = ("smbclient -W{0} -U'{1}%{2}' -L {3} >/dev/null 2>&1"
           .format(domain, username, password, host))
    try:
        subprocess.run(cmd, shell=True, check=True, stdout=subprocess.PIPE)
    except (KeyboardInterrupt, SystemExit):
        sys.exit(1)
    except: 
        # invalid credentials
        print("[ ]\t{3}\t{0}\{1}:{2}".format(domain, username, password, host))
        return 0
    else: 
        # valid credentials
        print("[x]\t{3}\t{0}\{1}:{2}".format(domain, username, password, host))
        return 1

def get_targets(args):
    """
    Get targets for info or brute mode
    """

    if args.t: return [args.t]
    return [t.rstrip() for t in args.f]

def get_args():
    """
    Get command line arguments
    """

    parser = argparse.ArgumentParser()
    subparsers = parser.add_subparsers(
            title="commands",
            help="choose mode of operation")

    # info gathering mode
    parser_i = subparsers.add_parser(
            "info", 
            help="enter info gathering mode")
    parser_i.set_defaults(func=info)

    # info: actions
    group_i_actions = parser_i.add_mutually_exclusive_group(required=True)
    group_i_actions.add_argument(
            "-u", "--users", 
            action="store_true",
            help="get user list")
    group_i_actions.add_argument(
            "-g", "--groups", 
            action="store_true",
            help="get group list and membership")
    group_i_actions.add_argument(
            "-s", "--shares", 
            action="store_true",
            help="get share list")
    group_i_actions.add_argument(
            "-p", "--policy", 
            action="store_true",
            help="get password policy")
    group_i_actions.add_argument(
            "-r", "--rid", 
            action="store_true",
            help="get users and groups via rid cycling")
    group_i_actions.add_argument(
            "-a", "--all", 
            action="store_true",
            help="get users, groups, and shares")

    # info: targets
    group_i_targets = parser_i.add_mutually_exclusive_group(required=True)
    group_i_targets.add_argument(
            "-t", 
            metavar="HOST",
            help="specify target hostname or IP address")
    group_i_targets.add_argument(
            "-f", 
            metavar="FILE",
            type=argparse.FileType("r"),
            help="specify file containing a list of targets")

    # info: authentication
    parser_i.add_argument(
            "-D",
            metavar="DOMAIN", 
            default=".",
            help="specify domain/workgroup to use")
    parser_i.add_argument(
            "-U",
            metavar="USER", 
            default="",
            help="specify username to use")
    parser_i.add_argument(
            "-P",
            metavar="PASS", 
            default="",
            help="specify password to use")

    # info: rid cycling
    parser_i.add_argument(
            "-S",
            metavar="START",
            type=int,
            default=500,
            help="specify start rid (default: 500)")
    parser_i.add_argument(
            "-E",
            metavar="END",
            type=int,
            default=1100,
            help="specify end rid (default: 1100)")

    # brute force mode
    parser_b = subparsers.add_parser(
            "brute", 
            help="enter brute force mode")
    parser_b.set_defaults(func=brute)

    # brute: targets
    group_b_targets = parser_b.add_mutually_exclusive_group(required=True)
    group_b_targets.add_argument(
            "-t", 
            metavar="HOST",
            help="specify target hostname or IP address")
    group_b_targets.add_argument(
            "-f", 
            metavar="FILE",
            type=argparse.FileType("r"),
            help="specify file containing a list of targets")

    # brute: credentials
    parser_b.add_argument(
            "-u", 
            metavar="USERFILE", 
            type=argparse.FileType("r"),
            required=True,
            help="specify user list")
    parser_b.add_argument(
            "-p", 
            metavar="PASSFILE",
            type=argparse.FileType("r"),
            help="specify password list")
    parser_b.add_argument(
            "-d", 
            metavar="DOMAIN",
            default=".",
            help="specify domain/workgroup to use")

    if len(sys.argv) == 1:
        parser.print_help()
        sys.exit(0)

    return parser.parse_args()

def main():
    """
    Main function
    """

    print(BANNER)

    if sys.version_info[0] != 3:
        print("// error: this script requires python 3")
        sys.exit(1)

    args = get_args()
    args.func(args)

if __name__ == "__main__":
    main()
