%feature("docstring") OT::QuantileConfidence
"Estimate confidence intervals of a quantile.

Refer to :ref:`quantile_confidence_estimation`.

.. warning::
    This class is experimental and likely to be modified in future releases.
    To use it, import the ``openturns.experimental`` submodule.

Parameters
----------
alpha : float
    Quantile level
beta : float, optional
    Confidence level. Default value is 0.95

Notes
-----
This class estimates bounds of the quantile of level :math:`\\alpha \\in [0,1]` of the random variable :math:`X`
with a confidence greater than :math:`\\beta` using a given order statistics.

Let :math:`x_{\\alpha}` be the unknown quantile of level :math:`\\alpha` of the random variable :math:`X` of dimension 1.
Let :math:`(X_1, \\dots, X_\\sampleSize)` be a sample of independent and identically distributed variables according to :math:`X`.

The bounds of the interval are computed from order statistics that we now introduce.
Let :math:`X_{(k)}` be the :math:`k` -th order statistics of :math:`(X_1, \\dots, X_\\sampleSize)` which means that
:math:`X_{(k)}` is the :math:`k` -th maximum of :math:`(X_1, \\dots, X_\\sampleSize)` for :math:`1 \\leq k \\leq \\sampleSize`.

For example, :math:`X_{(1)} = \\min (X_1, \\dots, X_\\sampleSize)` is the minimum
and :math:`X_{(\\sampleSize)} = \\max (X_1, \\dots, X_\\sampleSize)` is the maximum.

We have:

.. math::

    X_{(1)} \\leq X_{(2)} \\leq \\dots \\leq X_{(\\sampleSize)}

Examples
--------
>>> import openturns as ot
>>> import openturns.experimental as otexp
>>> alpha = 0.05
>>> beta = 0.95
>>> sample = ot.Gumbel().getSample(100)
>>> algo = otexp.QuantileConfidence(alpha, beta)
>>> ci = algo.computeUnilateralConfidenceInterval(sample)
"

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileConfidence::computeUnilateralRank
"Evaluate an unilateral rank of a quantile.

The lower tail rank :math:`k_{low}` is the largest integer :math:`k` such that:

.. math::

    \\Prob{X_{(k)} \\leq x_{\\alpha}} \\geq \\beta.

In other words, the interval :math:`\\left[ X_{(k_{low})}, +\\infty\\right[` is a unilateral confidence
interval for the quantile :math:`x_\\alpha` with confidence :math:`\\beta`.

It is given by:

.. math::

    k_{low} = \\overline{F}^{-1}_{n, \\alpha}(\\beta)

where :math:`\\overline{F}^{-1}_{n, \\alpha}` is the complementary quantile function 
of the binomial distribution with parameters :math:`n` and :math:`\\alpha`.

The problem has a solution only if:

.. math::

    1 - (1 - \\alpha)^n \\geq \\beta.

The upper tail rank :math:`k_{up}` is the smallest integer :math:`k` such that:

.. math::

    \\Prob{x_{\\alpha} \\leq X_{(k)}} \\geq \\beta.

In other words, the interval :math:`\\left]-\\infty, X_{(k_{up})}\\right]` is a unilateral confidence
interval for the quantile :math:`x_\\alpha` with confidence :math:`\\beta`.
The solution is:

.. math::

    k_{up} = F^{-1}_{n, \\alpha}(\\beta)

where :math:`F^{-1}_{n, \\alpha}` is the quantile function of the cumulative distribution function of the 
binomial distribution with parameters :math:`n` and :math:`\\alpha`.
The problem has a solution only if:

.. math::

    1 - \\alpha^n \\geq \\beta.

Parameters
----------
size : int
    Sample size
tail : bool, optional
    True indicates the interval is bounded by a lower value.
    False indicates the interval is bounded by an upper value. Default value is False.

Returns
-------
rank : int
    Rank :math:`k \\in \\{0, \\hdots, n-1\\}`

Examples
--------
>>> import openturns as ot
>>> import openturns.experimental as otexp
>>> alpha = 0.05
>>> beta = 0.95
>>> algo = otexp.QuantileConfidence(alpha, beta)
>>> rank = algo.computeUnilateralRank(100)
>>> print(rank)
9
"

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileConfidence::computeBilateralRank
"Evaluate the bilateral rank of a quantile.

The bilateral ranks :math:`k_1, k_2` are the integers minimizing the probability:

.. math::

    \\argmin_{0 \\leq k_1 \\leq k_2 \\leq n - 1 \\, | \\, \\Prob{k_1 < X \\leq k_2} \\geq \\beta} \\Prob{k_1 < X \\leq k_2}

where :math:`X \\sim \\cB(n, \\alpha)` is the binomial distribution with parameters :math:`n` and :math:`\\alpha`.

so that:

.. math::

    \\Prob{X_{(k_1)} \\leq x_{\\alpha} \\leq X_{(k_2)}} \\geq \\beta.

These ranks exist only if:

.. math::

    1 - \\alpha^n - (1-\\alpha)^n \\ge \\beta.

Parameters
----------
size : int
    Sample size :math:`n`

Returns
-------
ranks : :class:`~openturns.Indices` of size 2
    Pair of lower and upper ranks :math:`(k_1, k_2)` such that :math:`0 \\leq k_1 \\leq k_2 \\leq n - 1`.

Examples
--------
>>> import openturns as ot
>>> import openturns.experimental as otexp
>>> alpha = 0.05
>>> beta = 0.95
>>> algo = otexp.QuantileConfidence(alpha, beta)
>>> k1, k2 = algo.computeBilateralRank(100)
>>> print((k1, k2))
(1, 10)
"

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileConfidence::computeUnilateralConfidenceInterval
"Evaluate an unilateral confidence interval of a quantile.

The lower tail confidence interval is given by the order statistics:

.. math::

    [X_{(k_{low})}, +\\infty[

so that:

.. math::

    \\Prob{X_{(k_{low})} \\leq x_{\\alpha}} \\geq \\beta.

and the upper tail confidence interval is given by the order statistics:

.. math::

    ]-\\infty, X_{(k_{up})}]

so that:

.. math::

    \\Prob{x_{\\alpha} \\leq X_{(k_{up})}} \\geq \\beta.

where the lower or upper ranks :math:`0 \\leq k_{low} \\leq n - 1, 0 \\leq k_{up} \\leq n - 1` are given by :meth:`computeUnilateralRank`.

Parameters
----------
sample : 2-d sequence of float
    Quantile level
tail : bool, optional
    True indicates the interval is bounded by a lower value.
    False indicates the interval is bounded by an upper value. Default value is False.

Returns
-------
ci : :class:`~openturns.Interval`
    Quantile confidence interval

Examples
--------
>>> import openturns as ot
>>> import openturns.experimental as otexp
>>> alpha = 0.05
>>> beta = 0.95
>>> algo = otexp.QuantileConfidence(alpha, beta)
>>> sample = ot.Gumbel().getSample(100)
>>> ci = algo.computeUnilateralConfidenceInterval(sample)
"

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileConfidence::computeBilateralConfidenceInterval
"Evaluate a bilateral confidence interval of a quantile.

The confidence interval for the quantile of level :math:`\\alpha` is given by the order statistics:

.. math::

    [X_{(k_1)}, X_{(k_2)}]

so that:

.. math::

    \\Prob{X_{(k_1)} \\leq x_{\\alpha} \\leq X_{(k_2)}} \\geq \\beta

where the ranks :math:`0 \\leq k_1 \\leq k_2 \\leq n - 1` are given by :meth:`computeBilateralRank`.

Parameters
----------
sample : 2-d sequence of float
    Sample of the variable :math:`X`

Returns
-------
ci : :class:`~openturns.Interval`
    Quantile confidence interval

Examples
--------
>>> import openturns as ot
>>> import openturns.experimental as otexp
>>> alpha = 0.05
>>> beta = 0.95
>>> algo = otexp.QuantileConfidence(alpha, beta)
>>> sample = ot.Gumbel().getSample(60)
>>> ci = algo.computeBilateralConfidenceInterval(sample)
"

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileConfidence::computeUnilateralMinimumSampleSize
"Evaluate the minimum sample size for the unilateral case.

For the lower bound the minimal unilateral sample size is the smallest integer :math:`n` such that:

.. math::

    \\Prob{X_{(r)} \\leq x_{\\alpha}} \\geq \\beta.

For the upper bound the minimal unilateral sample size is the smallest integer :math:`n` such that:

.. math::

    \\Prob{x_{\\alpha} \\leq X_{(n-r-1)}} \\geq \\beta.

In the general case the minimum size of the sample is the smallest :math:`n` such that:

.. math::

    F_{n,\\alpha}(n-r-1) \\geq \\beta

where :math:`F_{n,\\alpha}` is the cumulative distribution function of the Binomial distribution with parameters :math:`n` and :math:`\\alpha`.

When the rank :math:`r` is zero the solution is analytical.

- For the lower bound the minimum sample size is the integer that satisfies:

  .. math::

      1 - (1 - \\alpha)^n \\geq \\beta

  which is given by:

  .. math::

      n_{low} = \\left\\lceil \\frac{\\log(1-\\beta)}{\\log(1-\\alpha)} \\right\\rceil.

- And for the upper bound the minimum sample size is the integer that satisfies:

  .. math::

      1 - \\alpha^n \\geq \\beta

  which is given by:

  .. math::

      n_{up} = \\left\\lceil \\frac{\\log(1-\\beta)}{\\log(\\alpha)} \\right\\rceil.

Parameters
----------
rank : int, optional
    Rank of the quantile. Default value is 0.
tail : bool, optional
    True indicates the interval is bounded by a lower value.
    False indicates the interval is bounded by an upper value. Default value is False.

Returns
-------
size : int
    Minimum sample size

Examples
--------
>>> import openturns as ot
>>> import openturns.experimental as otexp
>>> alpha = 0.05
>>> beta = 0.95
>>> algo = otexp.QuantileConfidence(alpha, beta)
>>> size = algo.computeUnilateralMinimumSampleSize(0)
>>> print(size)
59
"

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileConfidence::computeBilateralMinimumSampleSize
"Evaluate the minimum size of the sample for the bilateral case.

The minimal bilateral sample size is the smallest integer :math:`n` such that:

.. math::

    \\Prob{X_{(1)} \\leq x_{\\alpha} \\leq X_{(n)}} \\geq \\beta.

In other words, this is the minimum sample size such that the interval
:math:`\\left[X_{(1)}, X_{(n)} \\right]` is a confidence interval of the quantile :math:`x_\\alpha`
with confidence :math:`\\beta`.
The solution of this problem is the minimum value of :math:`n` such that:

.. math::

    1-\\alpha^n-(1-\\alpha)^n \\geq \\beta

The sample size :math:`n` is searched inside the bounds given by the inequality:

.. math::

    \\left\\lfloor \\frac{\\log(1 - \\beta)}{\\log(\\gamma)} \\right\\rfloor \\leq n \\leq \\left\\lceil \\frac{\\log(\\frac{1-\\beta}{2})}{\\log(\\gamma)} \\right\\rceil

where :math:`\\gamma = \\max(\\alpha, 1- \\alpha)`.

Returns
-------
size : int
    Minimum sample size :math:`n`

Examples
--------
>>> import openturns as ot
>>> import openturns.experimental as otexp
>>> alpha = 0.05
>>> beta = 0.95
>>> algo = otexp.QuantileConfidence(alpha, beta)
>>> size = algo.computeBilateralMinimumSampleSize()
>>> print(size)
59
"

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileConfidence::computeAsymptoticBilateralRank
"Evaluate the asymptotic bilateral rank of a quantile.

This method computes two integers :math:`(k_1, k_2)` such that:

.. math::
    \\lim_{n \\rightarrow \\infty} \\Prob{X_{(k_1)} \\leq x_\\alpha \\leq X_{(k_2)}} = \\beta, 0 \\leq k_1 \\leq k_2 \\leq n - 1

In other words, the interval :math:`\\left[X_{(k_1)}, X_{(k_2)} \\right]` is an asymptotic
confidence interval for :math:`x_\\alpha` with asymptotic confidence :math:`\\beta`.
The asymptotic bilateral ranks :math:`k_1, k_2 \\in \\llbracket 0, n - 1 \\rrbracket` are estimated from:

.. math::

    k_1 & = \\left\\lfloor n \\alpha - \\sqrt{n} z_{1-\\beta/2} \\sqrt{\\alpha (1 - \\alpha)} \\right\\rfloor - 1 \\\\
    k_2 & = \\left\\lfloor n \\alpha + \\sqrt{n} z_{1-\\beta/2} \\sqrt{\\alpha (1 - \\alpha)} \\right\\rfloor - 1

with :math:`z_{1-\\beta/2}` the standard Gaussian quantile of order :math:`1-\\beta/2`, see [delmas2006]_ proposition 12.2.13 page 257.

Parameters
----------
size : int
    Sample size

Returns
-------
ranks : :class:`~openturns.Indices` of size 2
    Pair of lower and upper ranks :math:`(k_1, k_2)` with :math:`0 \\leq k_1 \\leq k_2 \\leq n-1`.

Examples
--------
>>> import openturns as ot
>>> import openturns.experimental as otexp
>>> alpha = 0.05
>>> beta = 0.95
>>> algo = otexp.QuantileConfidence(alpha, beta)
>>> k1, k2 = algo.computeAsymptoticBilateralRank(100)
>>> print(k1, k2)
0 8
"

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileConfidence::computeAsymptoticBilateralConfidenceInterval
"Evaluate asymptotic bilateral bounds of a quantile.

The asymptotic bounds are given by the order statistics:

.. math::

    [X_{(k_1)}, X_{(k_2)}]

so that:

.. math::

    \\lim_{n \\rightarrow \\infty} \\Prob{X_{(k_1)} \\leq x_{\\alpha} \\leq X_{(k_2)}} \\geq \\beta

where the ranks :math:`0 \\leq k_1 \\leq k_2 \\leq n - 1` are given by :meth:`computeAsymptoticBilateralRank`.

Parameters
----------
sample : 2-d sequence of float
    Sample of the variable :math:`X`

Returns
-------
ci : :class:`~openturns.Interval`
    Quantile confidence interval

Examples
--------
>>> import openturns as ot
>>> import openturns.experimental as otexp
>>> alpha = 0.05
>>> beta = 0.95
>>> algo = otexp.QuantileConfidence(alpha, beta)
>>> sample = ot.Gumbel().getSample(60)
>>> ci = algo.computeAsymptoticBilateralConfidenceInterval(sample)
"

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileConfidence::setAlpha
"Quantile level accessor.

Parameters
----------
alpha : float
    Quantile level
"

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileConfidence::getAlpha
"Quantile level accessor.

Returns
-------
alpha : float
    Quantile level
"

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileConfidence::setBeta
"Confidence level accessor.

Parameters
----------
beta : float
    Confidence level
"

// ---------------------------------------------------------------------

%feature("docstring") OT::QuantileConfidence::getBeta
"Confidence level accessor.

Returns
-------
beta : float
    Confidence level
"
