/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { updateExtensionByName, updateAllUpdatableExtensions, } from '../../config/extension.js';
import { getErrorMessage } from '../../utils/errors.js';
const updateOutput = (info) => `Extension "${info.name}" successfully updated: ${info.originalVersion} → ${info.updatedVersion}.`;
export async function handleUpdate(args) {
    if (args.all) {
        try {
            const updateInfos = await updateAllUpdatableExtensions();
            if (updateInfos.length === 0) {
                console.log('No extensions to update.');
                return;
            }
            console.log(updateInfos.map((info) => updateOutput(info)).join('\n'));
        }
        catch (error) {
            console.error(getErrorMessage(error));
        }
    }
    if (args.name)
        try {
            // TODO(chrstnb): we should list extensions if the requested extension is not installed.
            const updatedExtensionInfo = await updateExtensionByName(args.name);
            console.log(`Extension "${args.name}" successfully updated: ${updatedExtensionInfo.originalVersion} → ${updatedExtensionInfo.updatedVersion}.`);
        }
        catch (error) {
            console.error(getErrorMessage(error));
        }
}
export const updateCommand = {
    command: 'update [--all] [name]',
    describe: 'Updates all extensions or a named extension to the latest version.',
    builder: (yargs) => yargs
        .positional('name', {
        describe: 'The name of the extension to update.',
        type: 'string',
    })
        .option('all', {
        describe: 'Update all extensions.',
        type: 'boolean',
    })
        .conflicts('name', 'all')
        .check((argv) => {
        if (!argv.all && !argv.name) {
            throw new Error('Either an extension name or --all must be provided');
        }
        return true;
    }),
    handler: async (argv) => {
        await handleUpdate({
            name: argv['name'],
            all: argv['all'],
        });
    },
};
//# sourceMappingURL=update.js.map