/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 */

/**
 * This test measures the latency between a write and a read from a Sh4lt.
 * This is done as follows: writer measures curent time, and writes the value into the sh4lt.
 * When called back, the reader mesures its curent time and compute the latency with the
 * time value read from the sh4lt.
 *
 * The test succeed if the duration between a write and a read is less than a millisecond.
 * The actual duration, however, is more likely being around few microseconds.
 **/

#undef NDEBUG  // get assert in release mode

#include <array>
#include <cassert>
#include <future>
#include <iostream>
#include <thread>

#include "../sh4lt/follower.hpp"
#include "../sh4lt/logger/console.hpp"
#include "../sh4lt/writer.hpp"

using namespace sh4lt;

auto main() -> int {
  using namespace sh4lt;
  auto log = std::make_shared<logger::Console>();
  auto on_data_called = false;

  {
    Writer w(ShType("application/x-check-sh4lt", "check-latency", "testing"), 1, log);
    assert(w);

    Follower follower(
        ShType::get_path("check-latency", "testing"),
        [&](void* data, size_t /*size*/, const sh4lt_time_info_t*) {
          // const auto reading_time = std::chrono::steady_clock::now().duration;
          const auto now = std::chrono::steady_clock::now();
          const auto reading_time =
              std::chrono::duration_cast<std::chrono::microseconds, int64_t>(now.time_since_epoch())
                  .count();
          auto writing_time = static_cast<int64_t*>(data);
          std::cout << "latency " << reading_time - *writing_time << "μs" << std::endl;
          // assert transmission is less than than 10 milliseconds
          assert(reading_time - *writing_time < 10000);
          on_data_called = true;
        },
        nullptr,
        nullptr,
        log);
    // testing 100 writes
    auto i = 100;
    std::this_thread::sleep_for(std::chrono::milliseconds(50));
    while (0 != i--) {
      const auto start = std::chrono::steady_clock::now();
      const auto duration =
          std::chrono::duration_cast<std::chrono::microseconds, int64_t>(start.time_since_epoch())
              .count();
      assert(w.copy_to_shm(&duration, sizeof(int64_t), -1, -1));
    }
  }
  if (!on_data_called) return 1;
  return 0;
}

