// =============================================================================
// PROJECT CHRONO - http://projectchrono.org
//
// Copyright (c) 2014 projectchrono.org
// All rights reserved.
//
// Use of this source code is governed by a BSD-style license that can be found
// in the LICENSE file at the top level of the distribution and at
// http://projectchrono.org/license-chrono.txt.
//
// =============================================================================
// Authors: Alessandro Tasora, Radu Serban
// =============================================================================

#ifndef CHELEMENTBAR_H
#define CHELEMENTBAR_H

#include "chrono/fea/ChElementGeneric.h"
#include "chrono/fea/ChNodeFEAxyz.h"

namespace chrono {
namespace fea {

/// @addtogroup fea_elements
/// @{

/// Simple finite element with two nodes and a bar that connects them.
/// No bending and torsion stiffness, just like a bar with two spherical joints.
/// In practical terms, this element works a bit like the class ChElementSpring,
/// but also adds mass along the element, hence point-like mass in the two nodes
/// is not needed.
class ChApi ChElementBar : public ChElementGeneric {
  public:
    ChElementBar();
    ~ChElementBar();

    virtual unsigned int GetNumNodes() override { return 2; }

    virtual unsigned int GetNumCoordsPosLevel() override { return 2 * 3; }

    virtual unsigned int GetNodeNumCoordsPosLevel(unsigned int n) override { return 3; }

    virtual std::shared_ptr<ChNodeFEAbase> GetNode(unsigned int n) override { return nodes[n]; }

    virtual void SetNodes(std::shared_ptr<ChNodeFEAxyz> nodeA, std::shared_ptr<ChNodeFEAxyz> nodeB);

    //
    // FEM functions
    //

    /// Fills the D vector with the current field values at the nodes of the element, with proper ordering.
    /// If the D vector size is not this->GetNumCoordsPosLevel(), it will be resized.
    virtual void GetStateBlock(ChVectorDynamic<>& mD) override;

    /// Add contribution of element inertia to total nodal masses
    virtual void ComputeNodalMass() override;

    /// Sets H as the global stiffness matrix K, scaled  by Kfactor. Optionally, also
    /// superimposes global damping matrix R, scaled by Rfactor, and global mass matrix M multiplied by Mfactor.
    /// (For the spring matrix there is no need to corotate local matrices: we already know a closed form expression.)
    virtual void ComputeKRMmatricesGlobal(ChMatrixRef H,
                                          double Kfactor,
                                          double Rfactor = 0,
                                          double Mfactor = 0) override;

    /// Computes the internal forces (ex. the actual position of nodes is not in relaxed reference position) and set
    /// values in the Fi vector.
    virtual void ComputeInternalForces(ChVectorDynamic<>& Fi) override;

    //
    // Custom properties functions
    //

    /// Set the cross sectional area of the bar (m^2) (also changes stiffness keeping same E modulus)
    void SetArea(double ma) { this->area = ma; }
    double GetArea() { return this->area; }

    /// Set the density of the bar (kg/m^3)
    void SetDensity(double md) { this->density = md; }
    double GetDensity() { return this->density; }

    /// Set the Young elastic modulus (N/m^2) (also sets stiffness)
    void SetYoungModulus(double mE) { this->E = mE; }
    double GetYoungModulus() { return this->E; }

    /// Set the Rayleigh damping ratio r (as in: R = r * K )
    void SetRayleighDamping(double mr) { this->rdamping = mr; }
    double GetRayleighDamping() { return this->rdamping; }

    /// The full mass of the bar
    double GetMass() { return this->mass; }

    /// The rest length of the bar
    double GetRestLength() { return this->length; }

    /// The current length of the bar (might be after deformation)
    double GetCurrentLength() { return (nodes[1]->GetPos() - nodes[0]->GetPos()).Length(); }

    /// Get the strain (epsilon), after deformation.
    double GetStrain() { return (GetCurrentLength() - GetRestLength()) / GetRestLength(); }

    /// Get the elastic stress (sigma), after deformation.
    double GetStress() { return GetYoungModulus() * GetStrain(); }

    /// Get the current force transmitted along the bar direction,
    /// including the effect of the damper. Positive if pulled. (N)
    virtual double GetCurrentForce();

    //
    // Functions for interfacing to the solver
    //            (***not needed, thank to bookkeeping in parent class ChElementGeneric)

  private:
    /// Initial setup. Precompute mass and matrices that do not change during the simulation, such as the local tangent
    /// stiffness Kl of each element, if needed, etc.
    virtual void SetupInitial(ChSystem* system) override;

    std::vector<std::shared_ptr<ChNodeFEAxyz> > nodes;
    double area;
    double density;
    double E;
    double rdamping;
    double mass;
    double length;
};

/// @} fea_elements

}  // end namespace fea
}  // end namespace chrono

#endif
