// =============================================================================
// PROJECT CHRONO - http://projectchrono.org
//
// Copyright (c) 2014 projectchrono.org
// All right reserved.
//
// Use of this source code is governed by a BSD-style license that can be found
// in the LICENSE file at the top level of the distribution and at
// http://projectchrono.org/license-chrono.txt.
//
// =============================================================================
// Authors: Radu Serban, Rainer Gericke
// =============================================================================
//
// MTV vehicle model (5t truck)
//
// =============================================================================

#ifndef MTV_VEHICLE_H
#define MTV_VEHICLE_H

#include "chrono_vehicle/wheeled_vehicle/ChWheeledVehicle.h"

#include "chrono_models/ChApiModels.h"
#include "chrono_models/vehicle/ChVehicleModelDefs.h"

namespace chrono {
namespace vehicle {
namespace fmtv {

/// @addtogroup vehicle_models_fmtv
/// @{

/// MTV vehicle system (5t truck).
class CH_MODELS_API MTV_Vehicle : public ChWheeledVehicle {
  public:
    MTV_Vehicle(const bool fixed,
                bool use_walking_beam,
                BrakeType brake_type,
                ChContactMethod contact_method = ChContactMethod::NSC,
                CollisionType chassis_collision_type = CollisionType::NONE);

    MTV_Vehicle(ChSystem* system,
                const bool fixed,
                bool use_walking_beam,
                BrakeType brake_type,
                CollisionType chassis_collision_type = CollisionType::NONE);

    ~MTV_Vehicle();

    virtual unsigned int GetNumberAxles() const override { return 3; }

    virtual double GetWheelbase() const override { return 4.1; }
    virtual double GetMinTurningRadius() const override { return 11.0; }
    virtual double GetMaxSteeringAngle() const override { return 24.6 * CH_DEG_TO_RAD; }

    void SetInitWheelAngVel(const std::vector<double>& omega) {
        assert(omega.size() == 6);
        m_omega = omega;
    }

    double GetSpringForce(int axle, VehicleSide side) const;
    double GetSpringLength(int axle, VehicleSide side) const;
    double GetSpringDeformation(int axle, VehicleSide side) const;

    double GetShockForce(int axle, VehicleSide side) const;
    double GetShockLength(int axle, VehicleSide side) const;
    double GetShockVelocity(int axle, VehicleSide side) const;

    virtual void Initialize(const ChCoordsys<>& chassisPos, double chassisFwdVel = 0) override;

    // Log debugging information
    void LogHardpointLocations();  /// suspension hardpoints at design
    void DebugLog(int what);       /// shock forces and lengths, constraints, etc.

  private:
    void Create(bool fixed, bool use_walking_beam, BrakeType brake_type, CollisionType chassis_collision_type);

    std::vector<double> m_omega;
};

/// @} vehicle_models_fmtv

}  // namespace fmtv
}  // end namespace vehicle
}  // end namespace chrono

#endif
