// =============================================================================
// PROJECT CHRONO - http://projectchrono.org
//
// Copyright (c) 2014 projectchrono.org
// All rights reserved.
//
// Use of this source code is governed by a BSD-style license that can be found
// in the LICENSE file at the top level of the distribution and at
// http://projectchrono.org/license-chrono.txt.
//
// =============================================================================
// Authors: Alessandro Tasora, Radu Serban
// =============================================================================

#ifndef CHGNUPLOT_H
#define CHGNUPLOT_H

#include <sstream>
#include <iostream>
#include <iomanip>

#include "chrono/core/ChMatrix.h"
#include "chrono/assets/ChColor.h"
#include "chrono/functions/ChFunctionBase.h"
#include "chrono/functions/ChFunctionInterp.h"

#include "chrono_postprocess/ChApiPostProcess.h"

namespace chrono {

/// Namespace with classes for the postprocess unit.
namespace postprocess {

/// @addtogroup postprocess_module
/// @{

class ChGnuPlotDataplot {
  public:
    std::string command;
    ChMatrixDynamic<double> data;
};

/// Class for plotting data with GNUplot.
/// This is a basic utility class which saves a temporary gpl file on disk and then calls the GNUplot utility from the
/// system shell, so it does not require to link GNUplot libraries. If the GNUplot is not installed, simply nothing
/// happens.
/// Notes:
/// - requires GNUplot version 4.6 or newer.
/// - the GNUplot executable must be available from a command window (make sure the executable inb is in the search
/// path).
/// - if no plot is displayed, open a cmd shell, type "gnuplot __tmp_gnuplot.gpl -persist" and see which error is
/// displayed.
class ChGnuPlot {
  public:
    ChGnuPlot(const std::string& filename = "__tmp_gnuplot.gpl") {
        commandfile += "# This is an autogenerated .gpl file that is executed by GNUplot \n";
        commandfile += "# It is created by the ChGnuPlot.h class of Chrono::Engine \n\n";
        persist = true;
        gpl_filename = filename;
    }

    virtual ~ChGnuPlot() {
        FlushPlots(commandfile);
        ExecuteGnuplot(this->commandfile);
    }

    /// Enable/disable persistent plots (default: true).
    /// If false, the output window will close soon after the plot.
    /// For plotting in windows, it is better to use the default 'true' setting.
    /// When plotting to a file (EPS, PNG, PDF), this setting is usually set to 'false'.
    void SetPersist(bool persist) { this->persist = persist; }

    /// Add arbitrary GnuPlot commands in the gnuplot script.
    /// Basically you would just need calls to SetCommand() followed by an Output() at the end, however to make things
    /// easier, there are some shortcut functions such as SetRangeX(), SetGrid(), Plot(), etc. which populate the
    /// command file.
    void SetCommand(const std::string& command) {
        commandfile += command;
        commandfile += "\n";
    }

    /// This is equivalent to SetCommand().
    ChGnuPlot& operator<<(const std::string& command) {
        this->SetCommand(command);
        return *this;
    }

    /// Plot 2D (x,y) data from an external file.
    void Plot(const std::string& datfile,
              int colX,
              int colY,
              const std::string& title,
              const std::string& customsettings = " with lines ") {
        ChGnuPlotDataplot mdataplot;

        mdataplot.command += " \"";
        mdataplot.command += datfile;
        mdataplot.command += "\" using ";
        mdataplot.command += std::to_string(colX);
        mdataplot.command += ":";
        mdataplot.command += std::to_string(colY);
        mdataplot.command += " ";
        mdataplot.command += customsettings;
        mdataplot.command += " title \"";
        mdataplot.command += title;
        mdataplot.command += "\" ";

        this->plots.push_back(mdataplot);
    }

    /// Plot 2D (x,y) data from two dynamic Eigen vectors.
    void Plot(ChVectorDynamic<>& x,
              ChVectorDynamic<>& y,
              const std::string& title = "",
              const std::string& customsettings = " with lines ") {
        assert(x.size() == y.size());

        ChGnuPlotDataplot mdataplot;
        mdataplot.data.resize(x.size(), 2);
        mdataplot.data.col(0) = x;
        mdataplot.data.col(1) = y;

        mdataplot.command += " \"-\" using 1:2 ";
        mdataplot.command += customsettings;
        mdataplot.command += " title \"";
        mdataplot.command += title;
        mdataplot.command += "\" ";

        this->plots.push_back(mdataplot);
    }

    /// Plot 2D (x,y) data from two columns of a matrix.
    void Plot(ChMatrixConstRef data,
              int colX,
              int colY,
              const std::string& title = "",
              const std::string& customsettings = " with lines ") {
        ChVectorDynamic<> x(data.rows());
        ChVectorDynamic<> y(data.rows());
        x = data.col(colX);
        y = data.col(colY);
        Plot(x, y, title, customsettings);
    }

    /// Plot 2D (x,y) data from a ChFunctionInterp
    void Plot(ChFunctionInterp& fun_table,
              const std::string& title = "",
              const std::string& customsettings = " with lines ") {
        ChVectorDynamic<> x(fun_table.GetTable().size());
        ChVectorDynamic<> y(fun_table.GetTable().size());

        int i = 0;
        for (auto iter = fun_table.GetTable().begin(); iter != fun_table.GetTable().end(); ++iter) {
            x(i) = iter->first;
            y(i) = iter->second;
            ++i;
        }
        Plot(x, y, title, customsettings);
    }

    /// Plot 2D (x,y) data from two vectors.
    void Plot(const std::vector<double>& vals_x,
              const std::vector<double>& vals_y,
              const std::string& title = "",
              const std::string& customsettings = " with lines ") {
        ChVectorDynamic<> x(vals_x.size());
        ChVectorDynamic<> y(vals_y.size());

        int i = 0;
        for (auto iter : vals_x) {
            x(i) = iter;
            ++i;
        }
        i = 0;
        for (auto iter : vals_y) {
            y(i) = iter;
            ++i;
        }
        Plot(x, y, title, customsettings);
    }

    /// Plot 2D (x,y) data from a generic ChFunction.
    /// Note that if the ChFunction is of type ChFunctionInterp there
    /// is a specific Plot() function
    void Plot(ChFunction& funct,
              double xmin,
              double xmax,
              double dx,
              const std::string& title = "",
              const std::string& customsettings = " with lines ") {
        int samples = (int)floor((xmax - xmin) / dx);
        ChVectorDynamic<> x(samples);
        ChVectorDynamic<> y(samples);

        double x_crt = xmin;
        for (int i = 0; i < samples; ++i) {
            x(i) = x_crt;
            y(i) = funct.GetVal(x_crt);
            x_crt += dx;
        }
        Plot(x, y, title, customsettings);
    }

    /// Plot 2D (x,y) data from a generic ChFunction's 0th, 1st, 2nd, or 3rd derivative.
    void Plot(ChFunction& funct,
              int der_order,
              double xmin,
              double xmax,
              double dx,
              const std::string& title = "",
              const std::string& customsettings = " with lines ") {
        int samples = (int)floor((xmax - xmin) / dx);
        ChVectorDynamic<> x(samples);
        ChVectorDynamic<> y(samples);

        double x_crt = xmin;
        for (int i = 0; i < samples; ++i) {
            x(i) = x_crt;
            if (der_order == 0)
                y(i) = funct.GetVal(x_crt);
            else if (der_order == 1)
                y(i) = funct.GetDer(x_crt);
            else if (der_order == 2)
                y(i) = funct.GetDer2(x_crt);
            else if (der_order == 3)
                y(i) = funct.GetDer3(x_crt);
            else
                y(i) = 0;
            x_crt += dx;
        }
        Plot(x, y, title, customsettings);
    }

    /// Replot last plots (since the last time any of the functions Output*** was used).
    void Replot() { commandfile += "replot \n"; }

    /// Set the X data range.
    void SetRangeX(double xmin, double xmax, bool automin = false, bool automax = false) {
        commandfile += "set xrange [";
        if (automin)
            commandfile += "*";
        else
            commandfile += std::to_string(xmin);
        commandfile += ":";
        if (automax)
            commandfile += "*";
        else
            commandfile += std::to_string(xmax);
        commandfile += "] \n";
    }

    /// Set the Y data range.
    void SetRangeY(double ymin, double ymax, bool automin = false, bool automax = false) {
        commandfile += "set yrange [";
        if (automin)
            commandfile += "*";
        else
            commandfile += std::to_string(ymin);
        commandfile += ":";
        if (automax)
            commandfile += "*";
        else
            commandfile += std::to_string(ymax);
        commandfile += "] \n";
    }

    /// Set the Y2 data range.
    void SetRangeY2(double ymin, double ymax, bool automin = false, bool automax = false) {
        commandfile += "set y2range [";
        if (automin)
            commandfile += "*";
        else
            commandfile += std::to_string(ymin);
        commandfile += ":";
        if (automax)
            commandfile += "*";
        else
            commandfile += std::to_string(ymax);
        commandfile += "] \n";
    }

    /// Set the Z data range.
    void SetRangeZ(double zmin, double zmax, bool automin = false, bool automax = false) {
        commandfile += "set zrange [";
        if (automin)
            commandfile += "*";
        else
            commandfile += std::to_string(zmin);
        commandfile += ":";
        if (automax)
            commandfile += "*";
        else
            commandfile += std::to_string(zmax);
        commandfile += "] \n";
    }

    /// Add a plot title.
    void SetTitle(const std::string& label) {
        commandfile += "set title \"";
        commandfile += label;
        commandfile += "\" \n";
    }

    /// Add a label to the X axis.
    void SetLabelX(const std::string& label) {
        commandfile += "set xlabel \"";
        commandfile += label;
        commandfile += "\" \n";
    }

    /// Add a label to the Y axis.
    void SetLabelY(const std::string& label) {
        commandfile += "set ylabel \"";
        commandfile += label;
        commandfile += "\" \n";
    }

    /// Add a label to the Y2 axis.
    void SetLabelY2(const std::string& label) {
        commandfile += "set y2label \"";
        commandfile += label;
        commandfile += "\" \n";
    }

    /// Add a label to the Z axis.
    void SetLabelZ(const std::string& label) {
        commandfile += "set zlabel \"";
        commandfile += label;
        commandfile += "\" \n";
    }

    /// Enable and specify parameters for a plot grid.
    void SetGrid(bool dashed = true, double linewidth = 1.0, const ChColor& mcolor = ChColor(0, 0, 0)) {
        commandfile += "set grid ";
        if (dashed)
            commandfile += "lt 0 ";
        else
            commandfile += "lt 1 ";
        commandfile += "lw ";
        commandfile += std::to_string(linewidth);
        commandfile += " lc ";
        commandfile += col_to_hex(mcolor);
        commandfile += "\n";
    }

    /// Hide plot legend.
    void HideLegend() { commandfile += " unset key\n"; }

    /// Show legend and set optional parameters.
    /// Example: SetLegend("bottom right box opaque")
    void SetLegend(const std::string& customsettings) {
        commandfile += " set key ";
        commandfile += customsettings;
        commandfile += "\n";
    }

    /// Set axes to equal size (i.e., square box).
    void SetAxesEqual() { commandfile += " set size square \n"; }

    /// Set plot in a window.
    /// For multiple windows, call this with increasing windownum, interleaving with Plot() statements etc.
    /// Call this before Plot() statements. Otherwise call Replot() just after.
    void OutputWindow(int windownum = 0) {
        FlushPlots(commandfile);
        commandfile += "set terminal wxt ";
        commandfile += std::to_string(windownum);
        commandfile += "\n";
    }

    /// Save plot in a PNG file.
    /// Call this before Plot() statements. Otherwise call Replot() just after.
    void OutputPNG(const std::string& filename, int sizex = 400, int sizey = 300) {
        FlushPlots(commandfile);
        commandfile += "set terminal png size ";
        commandfile += std::to_string(sizex);
        commandfile += ",";
        commandfile += std::to_string(sizey);
        commandfile += "\n";
        commandfile += "set output '";
        commandfile += filename;
        commandfile += "'\n";
    }

    /// Save plot in a PDF file.
    /// Call this before Plot() statements. Otherwise call Replot() just after.
    void OutputPDF(const std::string& filename) {
        FlushPlots(commandfile);
        commandfile += "set terminal pdf\n";
        commandfile += "set output '";
        commandfile += filename;
        commandfile += "'\n";
    }

    /// Save plot in a EPS file.
    /// Call this before Plot() statements. Otherwise call Replot() just after.
    void OutputEPS(const std::string& filename, double inchsizex = 4, double inchsizey = 3, bool color = true) {
        FlushPlots(commandfile);
        commandfile += "set terminal postscript eps ";
        if (color)
            commandfile += " color ";
        commandfile += " size ";
        commandfile += std::to_string(inchsizex);
        commandfile += ",";
        commandfile += std::to_string(inchsizey);
        commandfile += "\n";
        commandfile += "set output '";
        commandfile += filename;
        commandfile += "'\n";
    }

    /// Save plot in a custom terminal.
    /// For instance try terminalsetting ="set terminal svg size 350,262 fname 'Verdana' fsize 10"
    /// Call this before Plot() statements. Otherwise call Replot() just after.
    void OutputCustomTerminal(const std::string& filename, const std::string& terminalsetting) {
        FlushPlots(commandfile);
        commandfile += terminalsetting;
        commandfile += "set output '";
        commandfile += filename;
        commandfile += "'\n";
    }

    /// Close current plot command (e.g., for use in multiplot).
    void FlushPlots() { FlushPlots(commandfile); }

  protected:
    void FlushPlots(std::string& script) {
        // generate the  plot xxx , yyy , zzz stuff:
        if (plots.size() > 0)
            script += "plot ";
        for (int i = 0; i < this->plots.size(); ++i) {
            if (i > 0)
                script += " ,\\\n";
            script += this->plots[i].command;
        }
        script += "\n";

        // Embed plot data in the .gpl file
        for (int i = 0; i < this->plots.size(); ++i) {
            if ((plots[i].data.cols() > 0) && (plots[i].data.rows() > 0)) {
                for (int ir = 0; ir < plots[i].data.rows(); ++ir) {
                    for (int ic = 0; ic < plots[i].data.cols(); ++ic) {
                        script += std::to_string(plots[i].data(ir, ic));
                        script += " ";
                    }
                    script += "\n";
                }
                script += "end\n";
            }
        }
        this->plots.resize(0);
    }

    void ExecuteGnuplot(std::string& script) {
        // Create a tmp .gpl file
        {
            std::ofstream gnuplot_command(this->gpl_filename);
            gnuplot_command << script;
        }

        std::string syscmd;

        // Launch the GNUplot from shell
#ifdef _WIN32
        // "start /b gnuplot __tmp_gnuplot.gpl -persist"
        // where /b avoids showing the black cmd window
        syscmd += "start /b gnuplot \"";
        syscmd += this->gpl_filename;
        syscmd += "\"";
        if (persist)
            syscmd += " -persist";
        /*int err =*/system(syscmd.c_str());
#else
        // Unix like systems:
        // "gnuplot __tmp_gnuplot.gpl -persist &"
        syscmd += "gnuplot \"";
        syscmd += this->gpl_filename;
        syscmd += "\"";
        if (persist)
            syscmd += " -persist";
        syscmd += " &";  // to launch and forget
        /*int err =*/system(syscmd.c_str());
#endif
    }

    std::string col_to_hex(ChColor color) {
        std::string ret;
        ret += "rgb \"#";
        std::stringstream streamR;
        streamR << std::setfill('0') << std::setw(2) << std::hex << (int)(color.R * 255);
        std::string sr(streamR.str());
        ret += sr;
        std::stringstream streamG;
        streamG << std::setfill('0') << std::setw(2) << std::hex << (int)(color.G * 255);
        std::string sg(streamG.str());
        ret += sg;
        std::stringstream streamB;
        streamB << std::setfill('0') << std::setw(2) << std::hex << (int)(color.B * 255);
        std::string sb(streamB.str());
        ret += sb;
        ret += "\"";
        return ret;
    }

    std::string gpl_filename;
    std::string commandfile;
    std::vector<ChGnuPlotDataplot> plots;

    bool persist;
};

/// @} postprocess_module

}  // end namespace postprocess
}  // end namespace chrono

#endif
