/*!
 * \file   include/TFEL/Math/T2toT2/T2toT2T2toT2ProductExpr.hxx
 * \brief
 * \author Thomas Helfer
 * \date   16 juin 2014
 * \copyright Copyright (C) 2006-2025 CEA/DEN, EDF R&D. All rights
 * reserved.
 * This project is publicly released under either the GNU GPL Licence with
 * linking exception or the CECILL-A licence. A copy of thoses licences are
 * delivered with the sources of TFEL. CEA or EDF may also distribute this
 * project under specific licensing conditions.
 */

#ifndef LIB_TFEL_MATH_T2TOT2T2TOT2PRODUCTEXPR_HXX
#define LIB_TFEL_MATH_T2TOT2T2TOT2PRODUCTEXPR_HXX

#include "TFEL/Config/TFELConfig.hxx"
#include "TFEL/Math/General/EmptyRunTimeProperties.hxx"
#include "TFEL/Math/T2toT2/T2toT2Concept.hxx"

namespace tfel::math {

  //! Empty structure used for partial specialisation of the
  //! Expr class
  template <unsigned short N>
  struct TFEL_VISIBILITY_LOCAL T2toT2T2toT2ProductExpr {
  };  // end of struct T2toT2T2toT2ProductExpr

  /*!
   * Partial specialisation
   */
  template <typename T2toT2ResultType>
  struct TFEL_VISIBILITY_LOCAL
      Expr<T2toT2ResultType, T2toT2T2toT2ProductExpr<1u>>
      : public T2toT2ConceptBase<
            Expr<T2toT2ResultType, T2toT2T2toT2ProductExpr<1u>>>,
        public array_holder<
            TensorDimeToSize<getSpaceDimension<T2toT2ResultType>()>::value *
                TensorDimeToSize<getSpaceDimension<T2toT2ResultType>()>::value,
            numeric_type<T2toT2ResultType>> {
    static_assert(getSpaceDimension<T2toT2ResultType>() == 1u);
    //! a simple alias
    typedef EmptyRunTimeProperties RunTimeProperties;
    //! a simple alias
    typedef numeric_type<T2toT2ResultType> value_type;
    /*!
     * \param[in] a : first term of the product
     * \param[in] b : second term of the product
     */
    template <T2toT2Concept T2toT2Type, T2toT2Concept T2toT2Type2>
    TFEL_HOST_DEVICE constexpr Expr(const T2toT2Type& a,
                                    const T2toT2Type2& b) noexcept {
      static_assert(getSpaceDimension<T2toT2Type>() == 1u);
      static_assert(getSpaceDimension<T2toT2Type2>() == 1u);
      this->v[0] = a(0, 0) * b(0, 0) + a(0, 1) * b(1, 0) + a(0, 2) * b(2, 0);
      this->v[1] = a(0, 0) * b(0, 1) + a(0, 1) * b(1, 1) + a(0, 2) * b(2, 1);
      this->v[2] = a(0, 0) * b(0, 2) + a(0, 1) * b(1, 2) + a(0, 2) * b(2, 2);
      this->v[3] = a(1, 0) * b(0, 0) + a(1, 1) * b(1, 0) + a(1, 2) * b(2, 0);
      this->v[4] = a(1, 0) * b(0, 1) + a(1, 1) * b(1, 1) + a(1, 2) * b(2, 1);
      this->v[5] = a(1, 0) * b(0, 2) + a(1, 1) * b(1, 2) + a(1, 2) * b(2, 2);
      this->v[6] = a(2, 0) * b(0, 0) + a(2, 1) * b(1, 0) + a(2, 2) * b(2, 0);
      this->v[7] = a(2, 0) * b(0, 1) + a(2, 1) * b(1, 1) + a(2, 2) * b(2, 1);
      this->v[8] = a(2, 0) * b(0, 2) + a(2, 1) * b(1, 2) + a(2, 2) * b(2, 2);
    }  // end of Expr
    /*!
     * \brief access operator
     * \param[in] i : line index
     * \param[in] j : column index
     */
    TFEL_HOST_DEVICE constexpr const value_type& operator()(
        const unsigned short i, const unsigned short j) const {
      return this->v[i * 3 + j];
    }  // end of operator()
    //! \return the runtime properties
    TFEL_HOST_DEVICE constexpr auto getRunTimeProperties() const {
      return RunTimeProperties();
    }
  };  // end of struct Expr<T2toT2ResultType,T2toT2T2toT2ProductExpr>

  /*!
   * Partial specialisation
   */
  template <typename T2toT2ResultType>
  struct TFEL_VISIBILITY_LOCAL
      Expr<T2toT2ResultType, T2toT2T2toT2ProductExpr<2u>>
      : public T2toT2ConceptBase<
            Expr<T2toT2ResultType, T2toT2T2toT2ProductExpr<2u>>>,
        public array_holder<
            TensorDimeToSize<getSpaceDimension<T2toT2ResultType>()>::value *
                TensorDimeToSize<getSpaceDimension<T2toT2ResultType>()>::value,
            numeric_type<T2toT2ResultType>> {
    static_assert(getSpaceDimension<T2toT2ResultType>() == 2u);
    //! a simple alias
    typedef EmptyRunTimeProperties RunTimeProperties;
    //! a simple alias
    typedef numeric_type<T2toT2ResultType> value_type;
    /*!
     * \param[in] a : first term of the product
     * \param[in] b : second term of the product
     */
    template <T2toT2Concept T2toT2Type, T2toT2Concept T2toT2Type2>
    TFEL_HOST_DEVICE constexpr Expr(const T2toT2Type& a,
                                    const T2toT2Type2& b) noexcept {
      static_assert(getSpaceDimension<T2toT2Type>() == 2u);
      static_assert(getSpaceDimension<T2toT2Type2>() == 2u);
      this->v[0] = a(0, 0) * b(0, 0) + a(0, 1) * b(1, 0) + a(0, 2) * b(2, 0) +
                   a(0, 3) * b(3, 0) + a(0, 4) * b(4, 0);
      this->v[1] = a(0, 0) * b(0, 1) + a(0, 1) * b(1, 1) + a(0, 2) * b(2, 1) +
                   a(0, 3) * b(3, 1) + a(0, 4) * b(4, 1);
      this->v[2] = a(0, 0) * b(0, 2) + a(0, 1) * b(1, 2) + a(0, 2) * b(2, 2) +
                   a(0, 3) * b(3, 2) + a(0, 4) * b(4, 2);
      this->v[3] = a(0, 0) * b(0, 3) + a(0, 1) * b(1, 3) + a(0, 2) * b(2, 3) +
                   a(0, 3) * b(3, 3) + a(0, 4) * b(4, 3);
      this->v[4] = a(0, 0) * b(0, 4) + a(0, 1) * b(1, 4) + a(0, 2) * b(2, 4) +
                   a(0, 3) * b(3, 4) + a(0, 4) * b(4, 4);
      this->v[5] = a(1, 0) * b(0, 0) + a(1, 1) * b(1, 0) + a(1, 2) * b(2, 0) +
                   a(1, 3) * b(3, 0) + a(1, 4) * b(4, 0);
      this->v[6] = a(1, 0) * b(0, 1) + a(1, 1) * b(1, 1) + a(1, 2) * b(2, 1) +
                   a(1, 3) * b(3, 1) + a(1, 4) * b(4, 1);
      this->v[7] = a(1, 0) * b(0, 2) + a(1, 1) * b(1, 2) + a(1, 2) * b(2, 2) +
                   a(1, 3) * b(3, 2) + a(1, 4) * b(4, 2);
      this->v[8] = a(1, 0) * b(0, 3) + a(1, 1) * b(1, 3) + a(1, 2) * b(2, 3) +
                   a(1, 3) * b(3, 3) + a(1, 4) * b(4, 3);
      this->v[9] = a(1, 0) * b(0, 4) + a(1, 1) * b(1, 4) + a(1, 2) * b(2, 4) +
                   a(1, 3) * b(3, 4) + a(1, 4) * b(4, 4);
      this->v[10] = a(2, 0) * b(0, 0) + a(2, 1) * b(1, 0) + a(2, 2) * b(2, 0) +
                    a(2, 3) * b(3, 0) + a(2, 4) * b(4, 0);
      this->v[11] = a(2, 0) * b(0, 1) + a(2, 1) * b(1, 1) + a(2, 2) * b(2, 1) +
                    a(2, 3) * b(3, 1) + a(2, 4) * b(4, 1);
      this->v[12] = a(2, 0) * b(0, 2) + a(2, 1) * b(1, 2) + a(2, 2) * b(2, 2) +
                    a(2, 3) * b(3, 2) + a(2, 4) * b(4, 2);
      this->v[13] = a(2, 0) * b(0, 3) + a(2, 1) * b(1, 3) + a(2, 2) * b(2, 3) +
                    a(2, 3) * b(3, 3) + a(2, 4) * b(4, 3);
      this->v[14] = a(2, 0) * b(0, 4) + a(2, 1) * b(1, 4) + a(2, 2) * b(2, 4) +
                    a(2, 3) * b(3, 4) + a(2, 4) * b(4, 4);
      this->v[15] = a(3, 0) * b(0, 0) + a(3, 1) * b(1, 0) + a(3, 2) * b(2, 0) +
                    a(3, 3) * b(3, 0) + a(3, 4) * b(4, 0);
      this->v[16] = a(3, 0) * b(0, 1) + a(3, 1) * b(1, 1) + a(3, 2) * b(2, 1) +
                    a(3, 3) * b(3, 1) + a(3, 4) * b(4, 1);
      this->v[17] = a(3, 0) * b(0, 2) + a(3, 1) * b(1, 2) + a(3, 2) * b(2, 2) +
                    a(3, 3) * b(3, 2) + a(3, 4) * b(4, 2);
      this->v[18] = a(3, 0) * b(0, 3) + a(3, 1) * b(1, 3) + a(3, 2) * b(2, 3) +
                    a(3, 3) * b(3, 3) + a(3, 4) * b(4, 3);
      this->v[19] = a(3, 0) * b(0, 4) + a(3, 1) * b(1, 4) + a(3, 2) * b(2, 4) +
                    a(3, 3) * b(3, 4) + a(3, 4) * b(4, 4);
      this->v[20] = a(4, 0) * b(0, 0) + a(4, 1) * b(1, 0) + a(4, 2) * b(2, 0) +
                    a(4, 3) * b(3, 0) + a(4, 4) * b(4, 0);
      this->v[21] = a(4, 0) * b(0, 1) + a(4, 1) * b(1, 1) + a(4, 2) * b(2, 1) +
                    a(4, 3) * b(3, 1) + a(4, 4) * b(4, 1);
      this->v[22] = a(4, 0) * b(0, 2) + a(4, 1) * b(1, 2) + a(4, 2) * b(2, 2) +
                    a(4, 3) * b(3, 2) + a(4, 4) * b(4, 2);
      this->v[23] = a(4, 0) * b(0, 3) + a(4, 1) * b(1, 3) + a(4, 2) * b(2, 3) +
                    a(4, 3) * b(3, 3) + a(4, 4) * b(4, 3);
      this->v[24] = a(4, 0) * b(0, 4) + a(4, 1) * b(1, 4) + a(4, 2) * b(2, 4) +
                    a(4, 3) * b(3, 4) + a(4, 4) * b(4, 4);
    }  // end of Expr
    /*!
     * \brief access operator
     * \param[in] i : line index
     * \param[in] j : column index
     */
    TFEL_HOST_DEVICE constexpr const value_type& operator()(
        const unsigned short i, const unsigned short j) const noexcept {
      return this->v[i * 5 + j];
    }  // end of operator()
    //! \return the runtime properties
    TFEL_HOST_DEVICE constexpr auto getRunTimeProperties() const noexcept {
      return RunTimeProperties();
    }
  };  // end of struct Expr<T2toT2ResultType,T2toT2T2toT2ProductExpr>

  /*!
   * Partial specialisation
   */
  template <typename T2toT2ResultType>
  struct TFEL_VISIBILITY_LOCAL
      Expr<T2toT2ResultType, T2toT2T2toT2ProductExpr<3u>>
      : public T2toT2ConceptBase<
            Expr<T2toT2ResultType, T2toT2T2toT2ProductExpr<3u>>>,
        public array_holder<
            TensorDimeToSize<getSpaceDimension<T2toT2ResultType>()>::value *
                TensorDimeToSize<getSpaceDimension<T2toT2ResultType>()>::value,
            numeric_type<T2toT2ResultType>> {
    static_assert(getSpaceDimension<T2toT2ResultType>() == 3u);
    //! a simple alias
    typedef EmptyRunTimeProperties RunTimeProperties;
    //! a simple alias
    typedef numeric_type<T2toT2ResultType> value_type;
    /*!
     * \param[in] a : first term of the product
     * \param[in] b : second term of the product
     */
    template <T2toT2Concept T2toT2Type, T2toT2Concept T2toT2Type2>
    TFEL_HOST_DEVICE constexpr Expr(const T2toT2Type& a,
                                    const T2toT2Type2& b) noexcept {
      static_assert(getSpaceDimension<T2toT2Type>() == 3u);
      static_assert(getSpaceDimension<T2toT2Type2>() == 3u);
      this->v[0] = a(0, 0) * b(0, 0) + a(0, 1) * b(1, 0) + a(0, 2) * b(2, 0) +
                   a(0, 3) * b(3, 0) + a(0, 4) * b(4, 0) + a(0, 5) * b(5, 0) +
                   a(0, 6) * b(6, 0) + a(0, 7) * b(7, 0) + a(0, 8) * b(8, 0);
      this->v[1] = a(0, 0) * b(0, 1) + a(0, 1) * b(1, 1) + a(0, 2) * b(2, 1) +
                   a(0, 3) * b(3, 1) + a(0, 4) * b(4, 1) + a(0, 5) * b(5, 1) +
                   a(0, 6) * b(6, 1) + a(0, 7) * b(7, 1) + a(0, 8) * b(8, 1);
      this->v[2] = a(0, 0) * b(0, 2) + a(0, 1) * b(1, 2) + a(0, 2) * b(2, 2) +
                   a(0, 3) * b(3, 2) + a(0, 4) * b(4, 2) + a(0, 5) * b(5, 2) +
                   a(0, 6) * b(6, 2) + a(0, 7) * b(7, 2) + a(0, 8) * b(8, 2);
      this->v[3] = a(0, 0) * b(0, 3) + a(0, 1) * b(1, 3) + a(0, 2) * b(2, 3) +
                   a(0, 3) * b(3, 3) + a(0, 4) * b(4, 3) + a(0, 5) * b(5, 3) +
                   a(0, 6) * b(6, 3) + a(0, 7) * b(7, 3) + a(0, 8) * b(8, 3);
      this->v[4] = a(0, 0) * b(0, 4) + a(0, 1) * b(1, 4) + a(0, 2) * b(2, 4) +
                   a(0, 3) * b(3, 4) + a(0, 4) * b(4, 4) + a(0, 5) * b(5, 4) +
                   a(0, 6) * b(6, 4) + a(0, 7) * b(7, 4) + a(0, 8) * b(8, 4);
      this->v[5] = a(0, 0) * b(0, 5) + a(0, 1) * b(1, 5) + a(0, 2) * b(2, 5) +
                   a(0, 3) * b(3, 5) + a(0, 4) * b(4, 5) + a(0, 5) * b(5, 5) +
                   a(0, 6) * b(6, 5) + a(0, 7) * b(7, 5) + a(0, 8) * b(8, 5);
      this->v[6] = a(0, 0) * b(0, 6) + a(0, 1) * b(1, 6) + a(0, 2) * b(2, 6) +
                   a(0, 3) * b(3, 6) + a(0, 4) * b(4, 6) + a(0, 5) * b(5, 6) +
                   a(0, 6) * b(6, 6) + a(0, 7) * b(7, 6) + a(0, 8) * b(8, 6);
      this->v[7] = a(0, 0) * b(0, 7) + a(0, 1) * b(1, 7) + a(0, 2) * b(2, 7) +
                   a(0, 3) * b(3, 7) + a(0, 4) * b(4, 7) + a(0, 5) * b(5, 7) +
                   a(0, 6) * b(6, 7) + a(0, 7) * b(7, 7) + a(0, 8) * b(8, 7);
      this->v[8] = a(0, 0) * b(0, 8) + a(0, 1) * b(1, 8) + a(0, 2) * b(2, 8) +
                   a(0, 3) * b(3, 8) + a(0, 4) * b(4, 8) + a(0, 5) * b(5, 8) +
                   a(0, 6) * b(6, 8) + a(0, 7) * b(7, 8) + a(0, 8) * b(8, 8);
      this->v[9] = a(1, 0) * b(0, 0) + a(1, 1) * b(1, 0) + a(1, 2) * b(2, 0) +
                   a(1, 3) * b(3, 0) + a(1, 4) * b(4, 0) + a(1, 5) * b(5, 0) +
                   a(1, 6) * b(6, 0) + a(1, 7) * b(7, 0) + a(1, 8) * b(8, 0);
      this->v[10] = a(1, 0) * b(0, 1) + a(1, 1) * b(1, 1) + a(1, 2) * b(2, 1) +
                    a(1, 3) * b(3, 1) + a(1, 4) * b(4, 1) + a(1, 5) * b(5, 1) +
                    a(1, 6) * b(6, 1) + a(1, 7) * b(7, 1) + a(1, 8) * b(8, 1);
      this->v[11] = a(1, 0) * b(0, 2) + a(1, 1) * b(1, 2) + a(1, 2) * b(2, 2) +
                    a(1, 3) * b(3, 2) + a(1, 4) * b(4, 2) + a(1, 5) * b(5, 2) +
                    a(1, 6) * b(6, 2) + a(1, 7) * b(7, 2) + a(1, 8) * b(8, 2);
      this->v[12] = a(1, 0) * b(0, 3) + a(1, 1) * b(1, 3) + a(1, 2) * b(2, 3) +
                    a(1, 3) * b(3, 3) + a(1, 4) * b(4, 3) + a(1, 5) * b(5, 3) +
                    a(1, 6) * b(6, 3) + a(1, 7) * b(7, 3) + a(1, 8) * b(8, 3);
      this->v[13] = a(1, 0) * b(0, 4) + a(1, 1) * b(1, 4) + a(1, 2) * b(2, 4) +
                    a(1, 3) * b(3, 4) + a(1, 4) * b(4, 4) + a(1, 5) * b(5, 4) +
                    a(1, 6) * b(6, 4) + a(1, 7) * b(7, 4) + a(1, 8) * b(8, 4);
      this->v[14] = a(1, 0) * b(0, 5) + a(1, 1) * b(1, 5) + a(1, 2) * b(2, 5) +
                    a(1, 3) * b(3, 5) + a(1, 4) * b(4, 5) + a(1, 5) * b(5, 5) +
                    a(1, 6) * b(6, 5) + a(1, 7) * b(7, 5) + a(1, 8) * b(8, 5);
      this->v[15] = a(1, 0) * b(0, 6) + a(1, 1) * b(1, 6) + a(1, 2) * b(2, 6) +
                    a(1, 3) * b(3, 6) + a(1, 4) * b(4, 6) + a(1, 5) * b(5, 6) +
                    a(1, 6) * b(6, 6) + a(1, 7) * b(7, 6) + a(1, 8) * b(8, 6);
      this->v[16] = a(1, 0) * b(0, 7) + a(1, 1) * b(1, 7) + a(1, 2) * b(2, 7) +
                    a(1, 3) * b(3, 7) + a(1, 4) * b(4, 7) + a(1, 5) * b(5, 7) +
                    a(1, 6) * b(6, 7) + a(1, 7) * b(7, 7) + a(1, 8) * b(8, 7);
      this->v[17] = a(1, 0) * b(0, 8) + a(1, 1) * b(1, 8) + a(1, 2) * b(2, 8) +
                    a(1, 3) * b(3, 8) + a(1, 4) * b(4, 8) + a(1, 5) * b(5, 8) +
                    a(1, 6) * b(6, 8) + a(1, 7) * b(7, 8) + a(1, 8) * b(8, 8);
      this->v[18] = a(2, 0) * b(0, 0) + a(2, 1) * b(1, 0) + a(2, 2) * b(2, 0) +
                    a(2, 3) * b(3, 0) + a(2, 4) * b(4, 0) + a(2, 5) * b(5, 0) +
                    a(2, 6) * b(6, 0) + a(2, 7) * b(7, 0) + a(2, 8) * b(8, 0);
      this->v[19] = a(2, 0) * b(0, 1) + a(2, 1) * b(1, 1) + a(2, 2) * b(2, 1) +
                    a(2, 3) * b(3, 1) + a(2, 4) * b(4, 1) + a(2, 5) * b(5, 1) +
                    a(2, 6) * b(6, 1) + a(2, 7) * b(7, 1) + a(2, 8) * b(8, 1);
      this->v[20] = a(2, 0) * b(0, 2) + a(2, 1) * b(1, 2) + a(2, 2) * b(2, 2) +
                    a(2, 3) * b(3, 2) + a(2, 4) * b(4, 2) + a(2, 5) * b(5, 2) +
                    a(2, 6) * b(6, 2) + a(2, 7) * b(7, 2) + a(2, 8) * b(8, 2);
      this->v[21] = a(2, 0) * b(0, 3) + a(2, 1) * b(1, 3) + a(2, 2) * b(2, 3) +
                    a(2, 3) * b(3, 3) + a(2, 4) * b(4, 3) + a(2, 5) * b(5, 3) +
                    a(2, 6) * b(6, 3) + a(2, 7) * b(7, 3) + a(2, 8) * b(8, 3);
      this->v[22] = a(2, 0) * b(0, 4) + a(2, 1) * b(1, 4) + a(2, 2) * b(2, 4) +
                    a(2, 3) * b(3, 4) + a(2, 4) * b(4, 4) + a(2, 5) * b(5, 4) +
                    a(2, 6) * b(6, 4) + a(2, 7) * b(7, 4) + a(2, 8) * b(8, 4);
      this->v[23] = a(2, 0) * b(0, 5) + a(2, 1) * b(1, 5) + a(2, 2) * b(2, 5) +
                    a(2, 3) * b(3, 5) + a(2, 4) * b(4, 5) + a(2, 5) * b(5, 5) +
                    a(2, 6) * b(6, 5) + a(2, 7) * b(7, 5) + a(2, 8) * b(8, 5);
      this->v[24] = a(2, 0) * b(0, 6) + a(2, 1) * b(1, 6) + a(2, 2) * b(2, 6) +
                    a(2, 3) * b(3, 6) + a(2, 4) * b(4, 6) + a(2, 5) * b(5, 6) +
                    a(2, 6) * b(6, 6) + a(2, 7) * b(7, 6) + a(2, 8) * b(8, 6);
      this->v[25] = a(2, 0) * b(0, 7) + a(2, 1) * b(1, 7) + a(2, 2) * b(2, 7) +
                    a(2, 3) * b(3, 7) + a(2, 4) * b(4, 7) + a(2, 5) * b(5, 7) +
                    a(2, 6) * b(6, 7) + a(2, 7) * b(7, 7) + a(2, 8) * b(8, 7);
      this->v[26] = a(2, 0) * b(0, 8) + a(2, 1) * b(1, 8) + a(2, 2) * b(2, 8) +
                    a(2, 3) * b(3, 8) + a(2, 4) * b(4, 8) + a(2, 5) * b(5, 8) +
                    a(2, 6) * b(6, 8) + a(2, 7) * b(7, 8) + a(2, 8) * b(8, 8);
      this->v[27] = a(3, 0) * b(0, 0) + a(3, 1) * b(1, 0) + a(3, 2) * b(2, 0) +
                    a(3, 3) * b(3, 0) + a(3, 4) * b(4, 0) + a(3, 5) * b(5, 0) +
                    a(3, 6) * b(6, 0) + a(3, 7) * b(7, 0) + a(3, 8) * b(8, 0);
      this->v[28] = a(3, 0) * b(0, 1) + a(3, 1) * b(1, 1) + a(3, 2) * b(2, 1) +
                    a(3, 3) * b(3, 1) + a(3, 4) * b(4, 1) + a(3, 5) * b(5, 1) +
                    a(3, 6) * b(6, 1) + a(3, 7) * b(7, 1) + a(3, 8) * b(8, 1);
      this->v[29] = a(3, 0) * b(0, 2) + a(3, 1) * b(1, 2) + a(3, 2) * b(2, 2) +
                    a(3, 3) * b(3, 2) + a(3, 4) * b(4, 2) + a(3, 5) * b(5, 2) +
                    a(3, 6) * b(6, 2) + a(3, 7) * b(7, 2) + a(3, 8) * b(8, 2);
      this->v[30] = a(3, 0) * b(0, 3) + a(3, 1) * b(1, 3) + a(3, 2) * b(2, 3) +
                    a(3, 3) * b(3, 3) + a(3, 4) * b(4, 3) + a(3, 5) * b(5, 3) +
                    a(3, 6) * b(6, 3) + a(3, 7) * b(7, 3) + a(3, 8) * b(8, 3);
      this->v[31] = a(3, 0) * b(0, 4) + a(3, 1) * b(1, 4) + a(3, 2) * b(2, 4) +
                    a(3, 3) * b(3, 4) + a(3, 4) * b(4, 4) + a(3, 5) * b(5, 4) +
                    a(3, 6) * b(6, 4) + a(3, 7) * b(7, 4) + a(3, 8) * b(8, 4);
      this->v[32] = a(3, 0) * b(0, 5) + a(3, 1) * b(1, 5) + a(3, 2) * b(2, 5) +
                    a(3, 3) * b(3, 5) + a(3, 4) * b(4, 5) + a(3, 5) * b(5, 5) +
                    a(3, 6) * b(6, 5) + a(3, 7) * b(7, 5) + a(3, 8) * b(8, 5);
      this->v[33] = a(3, 0) * b(0, 6) + a(3, 1) * b(1, 6) + a(3, 2) * b(2, 6) +
                    a(3, 3) * b(3, 6) + a(3, 4) * b(4, 6) + a(3, 5) * b(5, 6) +
                    a(3, 6) * b(6, 6) + a(3, 7) * b(7, 6) + a(3, 8) * b(8, 6);
      this->v[34] = a(3, 0) * b(0, 7) + a(3, 1) * b(1, 7) + a(3, 2) * b(2, 7) +
                    a(3, 3) * b(3, 7) + a(3, 4) * b(4, 7) + a(3, 5) * b(5, 7) +
                    a(3, 6) * b(6, 7) + a(3, 7) * b(7, 7) + a(3, 8) * b(8, 7);
      this->v[35] = a(3, 0) * b(0, 8) + a(3, 1) * b(1, 8) + a(3, 2) * b(2, 8) +
                    a(3, 3) * b(3, 8) + a(3, 4) * b(4, 8) + a(3, 5) * b(5, 8) +
                    a(3, 6) * b(6, 8) + a(3, 7) * b(7, 8) + a(3, 8) * b(8, 8);
      this->v[36] = a(4, 0) * b(0, 0) + a(4, 1) * b(1, 0) + a(4, 2) * b(2, 0) +
                    a(4, 3) * b(3, 0) + a(4, 4) * b(4, 0) + a(4, 5) * b(5, 0) +
                    a(4, 6) * b(6, 0) + a(4, 7) * b(7, 0) + a(4, 8) * b(8, 0);
      this->v[37] = a(4, 0) * b(0, 1) + a(4, 1) * b(1, 1) + a(4, 2) * b(2, 1) +
                    a(4, 3) * b(3, 1) + a(4, 4) * b(4, 1) + a(4, 5) * b(5, 1) +
                    a(4, 6) * b(6, 1) + a(4, 7) * b(7, 1) + a(4, 8) * b(8, 1);
      this->v[38] = a(4, 0) * b(0, 2) + a(4, 1) * b(1, 2) + a(4, 2) * b(2, 2) +
                    a(4, 3) * b(3, 2) + a(4, 4) * b(4, 2) + a(4, 5) * b(5, 2) +
                    a(4, 6) * b(6, 2) + a(4, 7) * b(7, 2) + a(4, 8) * b(8, 2);
      this->v[39] = a(4, 0) * b(0, 3) + a(4, 1) * b(1, 3) + a(4, 2) * b(2, 3) +
                    a(4, 3) * b(3, 3) + a(4, 4) * b(4, 3) + a(4, 5) * b(5, 3) +
                    a(4, 6) * b(6, 3) + a(4, 7) * b(7, 3) + a(4, 8) * b(8, 3);
      this->v[40] = a(4, 0) * b(0, 4) + a(4, 1) * b(1, 4) + a(4, 2) * b(2, 4) +
                    a(4, 3) * b(3, 4) + a(4, 4) * b(4, 4) + a(4, 5) * b(5, 4) +
                    a(4, 6) * b(6, 4) + a(4, 7) * b(7, 4) + a(4, 8) * b(8, 4);
      this->v[41] = a(4, 0) * b(0, 5) + a(4, 1) * b(1, 5) + a(4, 2) * b(2, 5) +
                    a(4, 3) * b(3, 5) + a(4, 4) * b(4, 5) + a(4, 5) * b(5, 5) +
                    a(4, 6) * b(6, 5) + a(4, 7) * b(7, 5) + a(4, 8) * b(8, 5);
      this->v[42] = a(4, 0) * b(0, 6) + a(4, 1) * b(1, 6) + a(4, 2) * b(2, 6) +
                    a(4, 3) * b(3, 6) + a(4, 4) * b(4, 6) + a(4, 5) * b(5, 6) +
                    a(4, 6) * b(6, 6) + a(4, 7) * b(7, 6) + a(4, 8) * b(8, 6);
      this->v[43] = a(4, 0) * b(0, 7) + a(4, 1) * b(1, 7) + a(4, 2) * b(2, 7) +
                    a(4, 3) * b(3, 7) + a(4, 4) * b(4, 7) + a(4, 5) * b(5, 7) +
                    a(4, 6) * b(6, 7) + a(4, 7) * b(7, 7) + a(4, 8) * b(8, 7);
      this->v[44] = a(4, 0) * b(0, 8) + a(4, 1) * b(1, 8) + a(4, 2) * b(2, 8) +
                    a(4, 3) * b(3, 8) + a(4, 4) * b(4, 8) + a(4, 5) * b(5, 8) +
                    a(4, 6) * b(6, 8) + a(4, 7) * b(7, 8) + a(4, 8) * b(8, 8);
      this->v[45] = a(5, 0) * b(0, 0) + a(5, 1) * b(1, 0) + a(5, 2) * b(2, 0) +
                    a(5, 3) * b(3, 0) + a(5, 4) * b(4, 0) + a(5, 5) * b(5, 0) +
                    a(5, 6) * b(6, 0) + a(5, 7) * b(7, 0) + a(5, 8) * b(8, 0);
      this->v[46] = a(5, 0) * b(0, 1) + a(5, 1) * b(1, 1) + a(5, 2) * b(2, 1) +
                    a(5, 3) * b(3, 1) + a(5, 4) * b(4, 1) + a(5, 5) * b(5, 1) +
                    a(5, 6) * b(6, 1) + a(5, 7) * b(7, 1) + a(5, 8) * b(8, 1);
      this->v[47] = a(5, 0) * b(0, 2) + a(5, 1) * b(1, 2) + a(5, 2) * b(2, 2) +
                    a(5, 3) * b(3, 2) + a(5, 4) * b(4, 2) + a(5, 5) * b(5, 2) +
                    a(5, 6) * b(6, 2) + a(5, 7) * b(7, 2) + a(5, 8) * b(8, 2);
      this->v[48] = a(5, 0) * b(0, 3) + a(5, 1) * b(1, 3) + a(5, 2) * b(2, 3) +
                    a(5, 3) * b(3, 3) + a(5, 4) * b(4, 3) + a(5, 5) * b(5, 3) +
                    a(5, 6) * b(6, 3) + a(5, 7) * b(7, 3) + a(5, 8) * b(8, 3);
      this->v[49] = a(5, 0) * b(0, 4) + a(5, 1) * b(1, 4) + a(5, 2) * b(2, 4) +
                    a(5, 3) * b(3, 4) + a(5, 4) * b(4, 4) + a(5, 5) * b(5, 4) +
                    a(5, 6) * b(6, 4) + a(5, 7) * b(7, 4) + a(5, 8) * b(8, 4);
      this->v[50] = a(5, 0) * b(0, 5) + a(5, 1) * b(1, 5) + a(5, 2) * b(2, 5) +
                    a(5, 3) * b(3, 5) + a(5, 4) * b(4, 5) + a(5, 5) * b(5, 5) +
                    a(5, 6) * b(6, 5) + a(5, 7) * b(7, 5) + a(5, 8) * b(8, 5);
      this->v[51] = a(5, 0) * b(0, 6) + a(5, 1) * b(1, 6) + a(5, 2) * b(2, 6) +
                    a(5, 3) * b(3, 6) + a(5, 4) * b(4, 6) + a(5, 5) * b(5, 6) +
                    a(5, 6) * b(6, 6) + a(5, 7) * b(7, 6) + a(5, 8) * b(8, 6);
      this->v[52] = a(5, 0) * b(0, 7) + a(5, 1) * b(1, 7) + a(5, 2) * b(2, 7) +
                    a(5, 3) * b(3, 7) + a(5, 4) * b(4, 7) + a(5, 5) * b(5, 7) +
                    a(5, 6) * b(6, 7) + a(5, 7) * b(7, 7) + a(5, 8) * b(8, 7);
      this->v[53] = a(5, 0) * b(0, 8) + a(5, 1) * b(1, 8) + a(5, 2) * b(2, 8) +
                    a(5, 3) * b(3, 8) + a(5, 4) * b(4, 8) + a(5, 5) * b(5, 8) +
                    a(5, 6) * b(6, 8) + a(5, 7) * b(7, 8) + a(5, 8) * b(8, 8);
      this->v[54] = a(6, 0) * b(0, 0) + a(6, 1) * b(1, 0) + a(6, 2) * b(2, 0) +
                    a(6, 3) * b(3, 0) + a(6, 4) * b(4, 0) + a(6, 5) * b(5, 0) +
                    a(6, 6) * b(6, 0) + a(6, 7) * b(7, 0) + a(6, 8) * b(8, 0);
      this->v[55] = a(6, 0) * b(0, 1) + a(6, 1) * b(1, 1) + a(6, 2) * b(2, 1) +
                    a(6, 3) * b(3, 1) + a(6, 4) * b(4, 1) + a(6, 5) * b(5, 1) +
                    a(6, 6) * b(6, 1) + a(6, 7) * b(7, 1) + a(6, 8) * b(8, 1);
      this->v[56] = a(6, 0) * b(0, 2) + a(6, 1) * b(1, 2) + a(6, 2) * b(2, 2) +
                    a(6, 3) * b(3, 2) + a(6, 4) * b(4, 2) + a(6, 5) * b(5, 2) +
                    a(6, 6) * b(6, 2) + a(6, 7) * b(7, 2) + a(6, 8) * b(8, 2);
      this->v[57] = a(6, 0) * b(0, 3) + a(6, 1) * b(1, 3) + a(6, 2) * b(2, 3) +
                    a(6, 3) * b(3, 3) + a(6, 4) * b(4, 3) + a(6, 5) * b(5, 3) +
                    a(6, 6) * b(6, 3) + a(6, 7) * b(7, 3) + a(6, 8) * b(8, 3);
      this->v[58] = a(6, 0) * b(0, 4) + a(6, 1) * b(1, 4) + a(6, 2) * b(2, 4) +
                    a(6, 3) * b(3, 4) + a(6, 4) * b(4, 4) + a(6, 5) * b(5, 4) +
                    a(6, 6) * b(6, 4) + a(6, 7) * b(7, 4) + a(6, 8) * b(8, 4);
      this->v[59] = a(6, 0) * b(0, 5) + a(6, 1) * b(1, 5) + a(6, 2) * b(2, 5) +
                    a(6, 3) * b(3, 5) + a(6, 4) * b(4, 5) + a(6, 5) * b(5, 5) +
                    a(6, 6) * b(6, 5) + a(6, 7) * b(7, 5) + a(6, 8) * b(8, 5);
      this->v[60] = a(6, 0) * b(0, 6) + a(6, 1) * b(1, 6) + a(6, 2) * b(2, 6) +
                    a(6, 3) * b(3, 6) + a(6, 4) * b(4, 6) + a(6, 5) * b(5, 6) +
                    a(6, 6) * b(6, 6) + a(6, 7) * b(7, 6) + a(6, 8) * b(8, 6);
      this->v[61] = a(6, 0) * b(0, 7) + a(6, 1) * b(1, 7) + a(6, 2) * b(2, 7) +
                    a(6, 3) * b(3, 7) + a(6, 4) * b(4, 7) + a(6, 5) * b(5, 7) +
                    a(6, 6) * b(6, 7) + a(6, 7) * b(7, 7) + a(6, 8) * b(8, 7);
      this->v[62] = a(6, 0) * b(0, 8) + a(6, 1) * b(1, 8) + a(6, 2) * b(2, 8) +
                    a(6, 3) * b(3, 8) + a(6, 4) * b(4, 8) + a(6, 5) * b(5, 8) +
                    a(6, 6) * b(6, 8) + a(6, 7) * b(7, 8) + a(6, 8) * b(8, 8);
      this->v[63] = a(7, 0) * b(0, 0) + a(7, 1) * b(1, 0) + a(7, 2) * b(2, 0) +
                    a(7, 3) * b(3, 0) + a(7, 4) * b(4, 0) + a(7, 5) * b(5, 0) +
                    a(7, 6) * b(6, 0) + a(7, 7) * b(7, 0) + a(7, 8) * b(8, 0);
      this->v[64] = a(7, 0) * b(0, 1) + a(7, 1) * b(1, 1) + a(7, 2) * b(2, 1) +
                    a(7, 3) * b(3, 1) + a(7, 4) * b(4, 1) + a(7, 5) * b(5, 1) +
                    a(7, 6) * b(6, 1) + a(7, 7) * b(7, 1) + a(7, 8) * b(8, 1);
      this->v[65] = a(7, 0) * b(0, 2) + a(7, 1) * b(1, 2) + a(7, 2) * b(2, 2) +
                    a(7, 3) * b(3, 2) + a(7, 4) * b(4, 2) + a(7, 5) * b(5, 2) +
                    a(7, 6) * b(6, 2) + a(7, 7) * b(7, 2) + a(7, 8) * b(8, 2);
      this->v[66] = a(7, 0) * b(0, 3) + a(7, 1) * b(1, 3) + a(7, 2) * b(2, 3) +
                    a(7, 3) * b(3, 3) + a(7, 4) * b(4, 3) + a(7, 5) * b(5, 3) +
                    a(7, 6) * b(6, 3) + a(7, 7) * b(7, 3) + a(7, 8) * b(8, 3);
      this->v[67] = a(7, 0) * b(0, 4) + a(7, 1) * b(1, 4) + a(7, 2) * b(2, 4) +
                    a(7, 3) * b(3, 4) + a(7, 4) * b(4, 4) + a(7, 5) * b(5, 4) +
                    a(7, 6) * b(6, 4) + a(7, 7) * b(7, 4) + a(7, 8) * b(8, 4);
      this->v[68] = a(7, 0) * b(0, 5) + a(7, 1) * b(1, 5) + a(7, 2) * b(2, 5) +
                    a(7, 3) * b(3, 5) + a(7, 4) * b(4, 5) + a(7, 5) * b(5, 5) +
                    a(7, 6) * b(6, 5) + a(7, 7) * b(7, 5) + a(7, 8) * b(8, 5);
      this->v[69] = a(7, 0) * b(0, 6) + a(7, 1) * b(1, 6) + a(7, 2) * b(2, 6) +
                    a(7, 3) * b(3, 6) + a(7, 4) * b(4, 6) + a(7, 5) * b(5, 6) +
                    a(7, 6) * b(6, 6) + a(7, 7) * b(7, 6) + a(7, 8) * b(8, 6);
      this->v[70] = a(7, 0) * b(0, 7) + a(7, 1) * b(1, 7) + a(7, 2) * b(2, 7) +
                    a(7, 3) * b(3, 7) + a(7, 4) * b(4, 7) + a(7, 5) * b(5, 7) +
                    a(7, 6) * b(6, 7) + a(7, 7) * b(7, 7) + a(7, 8) * b(8, 7);
      this->v[71] = a(7, 0) * b(0, 8) + a(7, 1) * b(1, 8) + a(7, 2) * b(2, 8) +
                    a(7, 3) * b(3, 8) + a(7, 4) * b(4, 8) + a(7, 5) * b(5, 8) +
                    a(7, 6) * b(6, 8) + a(7, 7) * b(7, 8) + a(7, 8) * b(8, 8);
      this->v[72] = a(8, 0) * b(0, 0) + a(8, 1) * b(1, 0) + a(8, 2) * b(2, 0) +
                    a(8, 3) * b(3, 0) + a(8, 4) * b(4, 0) + a(8, 5) * b(5, 0) +
                    a(8, 6) * b(6, 0) + a(8, 7) * b(7, 0) + a(8, 8) * b(8, 0);
      this->v[73] = a(8, 0) * b(0, 1) + a(8, 1) * b(1, 1) + a(8, 2) * b(2, 1) +
                    a(8, 3) * b(3, 1) + a(8, 4) * b(4, 1) + a(8, 5) * b(5, 1) +
                    a(8, 6) * b(6, 1) + a(8, 7) * b(7, 1) + a(8, 8) * b(8, 1);
      this->v[74] = a(8, 0) * b(0, 2) + a(8, 1) * b(1, 2) + a(8, 2) * b(2, 2) +
                    a(8, 3) * b(3, 2) + a(8, 4) * b(4, 2) + a(8, 5) * b(5, 2) +
                    a(8, 6) * b(6, 2) + a(8, 7) * b(7, 2) + a(8, 8) * b(8, 2);
      this->v[75] = a(8, 0) * b(0, 3) + a(8, 1) * b(1, 3) + a(8, 2) * b(2, 3) +
                    a(8, 3) * b(3, 3) + a(8, 4) * b(4, 3) + a(8, 5) * b(5, 3) +
                    a(8, 6) * b(6, 3) + a(8, 7) * b(7, 3) + a(8, 8) * b(8, 3);
      this->v[76] = a(8, 0) * b(0, 4) + a(8, 1) * b(1, 4) + a(8, 2) * b(2, 4) +
                    a(8, 3) * b(3, 4) + a(8, 4) * b(4, 4) + a(8, 5) * b(5, 4) +
                    a(8, 6) * b(6, 4) + a(8, 7) * b(7, 4) + a(8, 8) * b(8, 4);
      this->v[77] = a(8, 0) * b(0, 5) + a(8, 1) * b(1, 5) + a(8, 2) * b(2, 5) +
                    a(8, 3) * b(3, 5) + a(8, 4) * b(4, 5) + a(8, 5) * b(5, 5) +
                    a(8, 6) * b(6, 5) + a(8, 7) * b(7, 5) + a(8, 8) * b(8, 5);
      this->v[78] = a(8, 0) * b(0, 6) + a(8, 1) * b(1, 6) + a(8, 2) * b(2, 6) +
                    a(8, 3) * b(3, 6) + a(8, 4) * b(4, 6) + a(8, 5) * b(5, 6) +
                    a(8, 6) * b(6, 6) + a(8, 7) * b(7, 6) + a(8, 8) * b(8, 6);
      this->v[79] = a(8, 0) * b(0, 7) + a(8, 1) * b(1, 7) + a(8, 2) * b(2, 7) +
                    a(8, 3) * b(3, 7) + a(8, 4) * b(4, 7) + a(8, 5) * b(5, 7) +
                    a(8, 6) * b(6, 7) + a(8, 7) * b(7, 7) + a(8, 8) * b(8, 7);
      this->v[80] = a(8, 0) * b(0, 8) + a(8, 1) * b(1, 8) + a(8, 2) * b(2, 8) +
                    a(8, 3) * b(3, 8) + a(8, 4) * b(4, 8) + a(8, 5) * b(5, 8) +
                    a(8, 6) * b(6, 8) + a(8, 7) * b(7, 8) + a(8, 8) * b(8, 8);
    }  // end of Expr
    /*!
     * \brief access operator
     * \param[in] i : line index
     * \param[in] j : column index
     */
    TFEL_HOST_DEVICE constexpr const value_type& operator()(
        const unsigned short i, const unsigned short j) const noexcept {
      return this->v[i * 9 + j];
    }  // end of operator()
    //! \return the runtime properties
    TFEL_HOST_DEVICE constexpr auto getRunTimeProperties() const noexcept {
      return RunTimeProperties();
    }
  };  // end of struct Expr<T2toT2ResultType,T2toT2T2toT2ProductExpr>

}  // end of namespace tfel::math

#endif /* LIB_TFEL_MATH_T2TOT2T2TOT2PRODUCTEXPR_HXX */
